package com.polestar.sensors;


import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.bluetooth.BluetoothAdapter.LeScanCallback;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.os.Build;
import android.os.Handler;
import android.os.SystemClock;
import android.support.annotation.RequiresApi;

import com.polestar.helpers.Log;
import com.polestar.models.SensorNao;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.TSENSORPOWERMODE;

import java.util.ArrayList;
import java.util.List;




@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
public class BleSensor43  extends BleSensor implements LeScanCallback  {
	
	boolean receivingMeasurement = false;
	private ScanCallback myScanCallback;
	private int mScanMode = ScanSettings.SCAN_MODE_LOW_LATENCY;
	final List<ScanFilter> mScanfilter = new ArrayList<>();

	private boolean isTimedown = false;
	private boolean isTimeUp = false;

	/**
	 * constructor
	 * @param caller
	 */
	public BleSensor43(ContextWrapper caller, ISensorObserver observer) {
		super(caller,observer);
	}


	/**
	 * initScanCallback
	 * @return
	 */
	@TargetApi(Build.VERSION_CODES.LOLLIPOP)
	private ScanCallback initScanCallback() {
		if (myScanCallback == null) {
			myScanCallback = new ScanCallback() {
				 /**
				    * Callback when a BLE advertisement has been found.
				    *
				    * @param callbackType Determines how this callback was triggered. Currently could only be
				    *            {@link ScanSettings#CALLBACK_TYPE_ALL_MATCHES}.
				    * @param result A Bluetooth LE scan result.
				    */
				   public void onScanResult(int callbackType, final ScanResult result) {
					   if (!receivingMeasurement){
						   Log.alwaysWarn(this.getClass().getName(),"BLE Scan : success, first measurement received from the sensor");
					   }

					   scanResultsHandler.post(new Runnable() {
						   @Override
						   public void run() {
							   receivingMeasurement = true;
							   String adress = result.getDevice().getAddress();
							   long ts = System.currentTimeMillis();
							   collectBleData(adress, result.getRssi(), ts, result.getScanRecord().getBytes());
							   if (isBleDataReadyToSend(ts) &&  (mBleCollectorHandler != null)) {
								   mBleCollectorHandler.removeMessages(MSG_READY_TO_SEND);
								   mBleCollectorHandler.sendEmptyMessage(MSG_READY_TO_SEND);
							   }
						   }
					   });
				   }

				   /**
				    * Callback when batch results are delivered.
				    *
				    * @param results List of scan results that are previously scanned.
				    */
				   public void onBatchScanResults(List<ScanResult> results) {
					   Log.alwaysWarn(this.getClass().getName(), "BLE Scan : onBatchScanResults ..........................");
				   }

				   /**
				    * Callback when scan could not be started.
				    *
				    * @param errorCode Error code (one of SCAN_FAILED_*) for scan failure.
				    */
				   public void onScanFailed(int errorCode) {
					   Log.alwaysWarn(this.getClass().getName(),"BLE Scan : onScanFailed with code : " + errorCode);

					   //Restart Ble
					   if(errorCode == SCAN_FAILED_APPLICATION_REGISTRATION_FAILED) {
						   launchScan();
					   }
				   }
			};
		}
		return myScanCallback;
	}

	protected void resetAfterScanPeriod() {

		Handler mainHandler = new Handler(myLooper);
		mainHandler.postDelayed(new Runnable() {
			@Override
			public void run() {
				resetBleScan();
			}
		}, this.mScanPeriod);
	}

	public void resetBleScan() {
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
			try {
				stopBleScan();
			} catch (Exception e) {
				Log.alwaysWarn(this.getClass().getName(), "Internal Android exception scanning for beacons");
			}
			if (this.mStatus == STATUS_RUNNING) {
				startBleScan();
			}
		}
	}



	
	/**
	 * initBleSensors
	 * register to listen to BLE Android sensors and start sensing
	 * @return
	 */
	protected synchronized boolean initBleSensors(){
		bleMeasurement.reset();
		// the timestamps
		lastPushedDataTimestamp = System.currentTimeMillis();
		if(bluetooth == null) {
			//Device does not support bluetooth
			Log.alwaysWarn(this.getClass().getName(),"BLE Sensor BluetoothAdapter.getDefaultAdapter() is null");
			return false;
		}

		if(launchScan()) {
			return true ;
		} else {
			return false ;
		}
	}
	

	@SuppressWarnings("deprecation")
	private boolean launchScan(){
		
		if(bluetooth != null){
			boolean succeedInStartScan = false ;
			int nbMaxTries = 3 ;
			int nbTries = 0 ;
			while (succeedInStartScan==false && nbTries<nbMaxTries)
			{
				if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
					bluetooth.stopLeScan(this);
					succeedInStartScan = bluetooth.startLeScan(this);
				} else {
					succeedInStartScan = startBleScan();
				}
							   	
				if(!succeedInStartScan) {
					Log.alwaysWarn(this.getClass().getName(),"BLE Sensor launchScan(): didnt suceed in start scan, new try...");
					try {
						Thread.sleep(500) ;
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}

				if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
					//succeedInStartScan = receivingMeasurement;
				}
				
				nbTries++ ;
			}
			
			if(succeedInStartScan) {
				Log.alwaysWarn(this.getClass().getName(),"BLE Sensor launchScan(): scan launched !");
				return true ;
			} else {
				Log.alwaysWarn(this.getClass().getName(),"BLE Sensor launchScan(): failed to launch scan");
				return false ;
			}
			
		} else {
			Log.alwaysWarn(this.getClass().getName(),"BLE Sensor launchScan(): bluetooth is null !!");
			return false ;
		}
	}
	
	/**
	 * startBleScan
	 * @return
	 */
	@TargetApi(Build.VERSION_CODES.LOLLIPOP)
	synchronized private boolean startBleScan() {

		if(this.mPowerMode==BLEPOWERMODE.VERY_LOW ) {
		   if(!isTimedown && !isTimeUp) {
			   scheduleTimeDown();
			   //registerTimeDownAlarm();
			   return true;
		   } else if(isTimedown) {
			   return true; // do ntohing when Ble scanner is already on time down
		   }
		}

		final ScanSettings.Builder settings = new ScanSettings.Builder();
		settings.setScanMode(mScanMode);
		settings.setReportDelay(0);
		settings.build();

		// Wen need to set callback type for anderod version > = android M
		if(android.os.Build.VERSION.SDK_INT >= 23) {
			settings.setCallbackType(ScanSettings.CALLBACK_TYPE_ALL_MATCHES);
			settings.setMatchMode(ScanSettings.MATCH_MODE_AGGRESSIVE);
		}

		final ScanCallback scanCallback = initScanCallback();
		try {

			Log.restricted(this.getClass().getName(),"BLE Sensor: startScan (" + scanCallback+ ")");
			Handler mainHandler = new Handler(myLooper);
			Runnable myRunnable = new Runnable() {
				@Override
				public void run() {
					if(bluetooth != null && bluetooth.isEnabled()){
						BluetoothLeScanner scanner = bluetooth.getBluetoothLeScanner();
						if(scanner != null){
							scanner.startScan(mScanfilter, settings.build(), scanCallback);
						}
					}

				}
			};
			mainHandler.post(myRunnable);

		} catch (IllegalArgumentException e) {
			return false;
		}

		//Reset after scan delay only for version < SDK API 24
		if (android.os.Build.VERSION.SDK_INT < 24 && resetEnabled ) {
			//not stop scanning
			resetAfterScanPeriod();
		}

		return true;
	}

	public void scheduleTimeDown() {

		long timedown = getTimeDown();
		isTimedown = true;
		Handler mainHandler = new Handler(myLooper);
		mainHandler.postDelayed(new Runnable() {
			@Override
			public void run() {
				if(isTimedown) {
					isTimedown = false;
					scheduleTimeUp();
					resetBleScan();
				}
			}
		}, timedown * 1000);

	}

	public void scheduleTimeUp() {
		long timeup = getTimeUp();
		isTimeUp = true;
		Handler mainHandler = new Handler(myLooper);
		mainHandler.postDelayed(new Runnable() {
			@Override
			public void run() {
				isTimeUp = false;
				//for sdk >= 24, do reset if power mode is very low
				if (android.os.Build.VERSION.SDK_INT  >= 24) {
					resetBleScan();
				}
			}
		}, timeup * 1000);
	}

	private long getTimeDown() {
		if(mUuidMap == null || mUuidMap.getOptions().isEmpty()) {
			return 0;
		}
		String timed= mUuidMap.getOptions().get("timeDown");
		return (timed==null)?0:Long.valueOf(timed);
	}


	private long getTimeUp() {
		if(mUuidMap == null || mUuidMap.getOptions().isEmpty()) {
			return 0;
		}
		String timeup= mUuidMap.getOptions().get("timeUp");
		return (timeup==null)?0:Long.valueOf(timeup);
	}


	/**
	 * onLeScan - Ble callback
	 * @param device
	 * @param rssi
	 * @param scanRecord
	 */
	public void onLeScan(final BluetoothDevice device, final int rssi, final byte[] scanRecord) {
		scanResultsHandler.post(
			new Runnable() {
				@Override
				public void run() {
					receivingMeasurement = true;
					String adress = device.getAddress();
					long ts = System.currentTimeMillis();
					collectBleData(adress, rssi, ts, scanRecord);
					if (isBleDataReadyToSend(ts) &&  (mBleCollectorHandler != null)) {
						mBleCollectorHandler.removeMessages(MSG_READY_TO_SEND);
						mBleCollectorHandler.sendEmptyMessage(MSG_READY_TO_SEND);
					}
				}
			});

	}
	
	/**
	 *  clean-up BLE sensors and stop sensing
	 */
	protected void finalizeBleSensors(){
		if(bluetooth != null){
			if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
				bluetooth.stopLeScan(this);
			} else {
				stopBleScan();
			}
		}
		receivingMeasurement = false;

		synchronized (bleMeasurement){
			bleMeasurement.reset();
		}
	}

	@Override
	public void setDevicesFilter(List<String> devicesAdrr) {
		mScanfilter.clear();
		if (mScanfilter == null) {
			Log.alwaysWarn(getClass().getSimpleName(), "Scan filter not initialized: unable to set filer");
			return;
		}
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
			for (String mac : devicesAdrr) {
				mScanfilter.add(new ScanFilter.Builder().setDeviceAddress(mac).build());
			}
		} else {
			Log.alwaysWarn(getClass().getSimpleName(), "cannot set scan filter before API 21");
			return;
		}
	}


	/**
	 * stopBleScan
	 */
	@TargetApi(Build.VERSION_CODES.LOLLIPOP)
	synchronized private void stopBleScan() {
		if( bluetooth != null && bluetooth.isEnabled() && bluetooth.getBluetoothLeScanner() !=null && myScanCallback!= null){
			try {
				Log.restricted(this.getClass().getName(), "BLE Sensor: stopScan (" + myScanCallback + ")");
				bluetooth.getBluetoothLeScanner().stopScan(myScanCallback);
			} catch (IllegalArgumentException e) {
				Log.alwaysWarn(this.getClass().getName(),"BLE Sensor: can't stop scan ble (bluetooth.getBluetoothLeScanner().stopScan)");
			} catch (NullPointerException e) {
				// This catch is meant to avoid the seemingly random exceptions thrown :
				// java.lang.NullPointerException: Attempt to invoke virtual method 'android.os.Looper android.os.Handler.getLooper()' on a null object reference
			} catch (IllegalStateException e){
				Log.alwaysWarn(this.getClass().getName(),"BLE Sensor: can't stop scan ble (bluetooth.getBluetoothLeScanner().stopScan)");
			}
		}
	}

	@Override
	public void start() {
		Log.alwaysWarn(this.getClass().getName(),"Starting Ble sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_ON, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	@Override
	public boolean isRunning() {
		//Log.alwaysWarn(this.getClass().getName(),"is ble RUNNING : " + (mStatus == STATUS_RUNNING));
		return (mStatus == STATUS_RUNNING);
	}

	@Override
	public void stop() {
		Log.alwaysWarn(this.getClass().getName(),"Stoping ble sensors from native .....");
		switchSensor(SensorNao.ORDER_TURN_OFF, SensorNao.ORDER_PROVENANCE_SERVICE);
	}

	@Override
	public void setPowerMode(TSENSORPOWERMODE mode) {
		Log.alwaysWarn(this.getClass().getName(), "set Power Mode from native : " +mode);

		if(mode == TSENSORPOWERMODE.HIGH ) {
			this.mPowerMode = BLEPOWERMODE.HIGH;
			registerTimeDownAlarm();
			resetBleScan();
		} else if (mode == TSENSORPOWERMODE.LOW ) {
			if(this.getTimeDown() > 0 && this.getTimeUp() > 0) {
				this.mPowerMode = BLEPOWERMODE.VERY_LOW;
			} else {
				this.mPowerMode = BLEPOWERMODE.LOW;
			}
		}

		//Reset if power mode is very low
		if (android.os.Build.VERSION.SDK_INT  >= 24) {
			if(this.mPowerMode == BLEPOWERMODE.VERY_LOW) {resetBleScan();}
			return;
		}
	}


	protected PendingIntent wakeUpPending() {
		if (mWakeUpIntent == null && mCtxtWrap!=null) {
			Intent wakeupIntent = new Intent(mCtxtWrap,BleSensorTimeDownBroadcastReceiver.class);
			mWakeUpIntent = PendingIntent.getBroadcast(mCtxtWrap, 0, wakeupIntent, PendingIntent.FLAG_UPDATE_CURRENT);
		}
		return mWakeUpIntent;
	}

	// In case we go into very low power mode, we will set up a wakeup alarm when in the background
	protected void registerTimeDownAlarm() {

		if(mCtxtWrap!=null) {
			isTimedown = true;
			long timeDown = getTimeDown() * 1000;
			AlarmManager alarmManager= (AlarmManager) mCtxtWrap.getSystemService(Context.ALARM_SERVICE);
			alarmManager.set(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + timeDown, wakeUpPending());
			Log.writeToLog(this.getClass().getName(),"Set a wakeup alarm to go off " + timeDown + " : " + mCtxtWrap.getApplicationContext());
		}
	}

	protected void unregisterTimeDownAlarm() {
		isTimedown = false;
		if(mWakeUpIntent != null) {
			mWakeUpIntent.cancel();
		}
	}

	public void setIsTimeDown(boolean istimedown) {
		this.isTimedown = istimedown;
	}

}

