package com.polestar.naosync;

import com.polestar.helpers.FileHelper;
import com.polestar.helpers.Log;
import com.polestar.helpers.HttpHelper;
import com.polestar.helpers.ZipHelper;
import com.polestar.naosdk.api.ISynchroBroker;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import okhttp3.Response;

/**
 * Created by ASimonigh on 28/09/2016.
 */

public class SynchroBroker extends ISynchroBroker {

	HttpHelper m_httpHelper ;

	public SynchroBroker(HttpHelper httpHelper) {
		m_httpHelper = httpHelper;
	}

	@Override
    public int getFile(String src, String dest) {
        return getFileIfModified(src, dest, "");
    }

	@Override
	public int getFileIfModified(String src, String dest, String lastModifiedDate) {
		Response response = m_httpHelper.downloadFile(src, dest, lastModifiedDate);
		if(response != null){
			return response.code();
		} else {
			return 0;
		}
	}

	@Override
	public boolean sendFile(String src, String dest,  HashMap<String, String> uploadParams) {
		Log.restricted(getClass().getSimpleName(), "Send file: " + src);

		ArrayList<String> paramsList = new ArrayList<>();
		for(Map.Entry<String, String> entry : uploadParams.entrySet()){
			paramsList.add(entry.getKey());
			paramsList.add(entry.getValue());
		}

		File file = new File(src);
		if(file.exists() && file.isFile()) {
			return uploadFile(src, dest, paramsList);

		} else if(file.exists() && file.isDirectory()){
			//zip folder and send file
			String zipFile = zipDirectory(src);
			if(null == zipFile){
				Log.alwaysError(getClass().getSimpleName(), "Unable to find file to send");
				return false;
			}
			return uploadFile(zipFile, dest, paramsList);
		}
		return false;

	}

	/**
	 * Upload and delete file if upload succeeds
	 * @param sourceFile
	 * @param dest
	 * @param uploadParams
	 * @return
	 */
	private boolean uploadFile(String sourceFile, String dest, ArrayList<String> uploadParams){

		String[] params = new String[uploadParams.size()];
		params = uploadParams.toArray(params);

		Response result = m_httpHelper.uploadFile(sourceFile, dest, params);

		if(HttpHelper.checkHttpResponse(result)){
			Log.alwaysWarn(getClass().getSimpleName(), "Upload to S3: success");

			try {
				//delete file if upload is ok
				if(!FileHelper.delete(sourceFile)){
					Log.alwaysWarn(getClass().getSimpleName(),"Cannot delete file: "+ sourceFile);
				}
			} catch (Exception e) {
				e.printStackTrace();
			}

			return true;
		}else{
			Log.alwaysWarn(getClass().getSimpleName(), "Upload to S3: fail");
			return false;
		}
	}


	/**
	 * Zip directory in param
	 * @param src
	 * @return null if failed, zipFile path if success
	 */
	private String zipDirectory(String src){
		String zipFile = src + ".zip";
		try {
			ZipHelper.zipFolder(src,zipFile);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
		return zipFile;
	}


}
