//______________________________________________________________________________________
//
//  NaoLocationService.java
//
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.naosdk.managers;

import android.app.Service;
import android.content.ComponentName;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Binder;
import android.os.IBinder;
import android.os.PowerManager;
import android.support.annotation.Nullable;

import com.polestar.helpers.AndroidDeviceInfo;
import com.polestar.helpers.Log;
import com.polestar.helpers.PrefHelper;

import static com.polestar.helpers.NaoUtils.isServiceAvailable;

/**
 * The service manager.
 *
 * <br/>
 * <br/>
 * <b>Revision history:</b><br/>
 * <table border>
 * <tr>
 * <td>Author</td>
 * <td>Modification date</td>
 * <td>Tracking Number</td>
 * <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 * <td>jchouki</td>
 * <td>04 december 2015</td>
 * <td>trunk</td>
 * <td> Nao service manager  :android service to be bind</td>
 * </tr>
 * </table>
 *
 * @author jchouki
 */
public abstract class NaoServiceManager extends Service {

	/**
	 * @exclude
	 */
	private NaoContext naoContext;
	private static NaoServiceManager mService;
	protected final IBinder mBinder = new LocalServiceManager();
	private  static Context mContext;
	/** Flag indicating whether we have called bind on the service. */
	private static boolean mBound = false;
	private static String TAG ="com.polestar.naosdk.managers.NaoServiceManager" ;
	private static PowerManager.WakeLock mWakeLock;


	public static NaoServiceManager getService() {
		return mService;
	}

	public NaoContext getNaoContext() {
		return naoContext;
	}

	public static void startService(Context context, Class<?> cls) {
		// Bind to LocalService
		mContext = context.getApplicationContext();
		Intent intent = new Intent(context, cls);
		mContext.bindService(intent, mConnection, Context.BIND_AUTO_CREATE);
	}

	public static void stopService() {
		if(mContext != null && mBound == true ) {
			// Bind to LocalService
			try {
				mContext.unbindService(mConnection);
			} catch (IllegalArgumentException e) {
				Log.alwaysWarn(TAG,"can't unbind nao service : " + e.toString());
			}

			mService = null;
			mBound = false;
		}
	}

	/** Defines callbacks for service binding, passed to bindService() */
	private static ServiceConnection mConnection = new ServiceConnection() {

		@Override
		public void onServiceConnected(ComponentName className,
									   IBinder service) {
			Log.alwaysWarn(getClass().getName(), "onServiceConnected --> " + className.getPackageName() + " : " + className.getShortClassName());
			// We've bound to LocalService, cast the IBinder and get LocalService instance
			NaoServiceManager.LocalServiceManager binder = (NaoServiceManager.LocalServiceManager) service;
			mService = binder.getService();
			mBound = true;
		}

		@Override
		public void onServiceDisconnected(ComponentName className) {
			Log.alwaysWarn(getClass().getName(), "onServiceDisconnected --> " + className.getPackageName() + " : " + className.getShortClassName());
			mBound = false;
		}
	};

	/**
	 * Class used for the client Binder.  Because we know this service always
	* runs in the same process as its clients, we don't need to deal with IPC.
	*/
	public class LocalServiceManager extends Binder {
		public NaoServiceManager getService() {
			// Return this instance of LocalService so clients can call public methods
			return NaoServiceManager.this;
		}
	}

	protected ContextWrapper myContext;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onCreate() {
		super.onCreate();
		myContext = this;
		Log.alwaysWarn(getClass().getName(), "onCreate service ............. ");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		Log.alwaysWarn(getClass().getName(), "onStartCommand service ............. START_STICKY");
		PrefHelper.put(getApplicationContext(), PrefHelper.PREF_SERVICE_NAME, this.getClass().getName());
		return START_STICKY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean onUnbind(Intent intent) {
		// note : for an unknown reason the unbindService method does not result
		// in a call to onUnbind...
		return false;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onDestroy() {
		super.onDestroy();
	}

	@Nullable
	@Override
	public IBinder onBind(Intent intent) {
		Log.alwaysWarn(getClass().getName(), "onBind service ............. ");
		naoContext = new NaoContext(this);
		PrefHelper.put(getApplicationContext(), PrefHelper.PREF_SERVICE_NAME, this.getClass().getName());
		return mBinder;
	}


	/**
	 * acquire WakeLock to indicate that the application needs to have the device stay on (for background use)
	 * @param ctx
	 */
	public static void acquireWakeLock(Context ctx) {
		//return if wakelock permission not granted
		if(!isWakeLockEnabledForBackground(ctx))
			return;

		if(mWakeLock == null) {
			//otherwise acquire the wakelock
			PowerManager mPowerManager = (PowerManager) ctx.getSystemService(Context.POWER_SERVICE);
			mWakeLock = mPowerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "NAO_SDK_PARTIAL_WAKE_LOCK");
		}
		//get current level of wake lock
		if (mWakeLock != null && (!mWakeLock.isHeld())) {
			Log.writeToLog(TAG, "acquireWakeLock");
			mWakeLock.acquire();
			if(!mWakeLock.isHeld()) {
				Log.writeToLog(TAG, "Unable to hold wakeLock.");
			}
		}
	}

	/**
	 * release WakeLock (not need to have the device staying on)
	 */
	public static  void releaseWakeLock() {
		if ((mWakeLock != null) && (mWakeLock.isHeld())) {
			Log.writeToLog(TAG, "releaseWakeLock");
			mWakeLock.release();
			if(mWakeLock.isHeld()) {
				Log.writeToLog(TAG, "Unable to release wakeLock.");
			}
		}
	}

	public static boolean isWakeLockEnabledForBackground(Context ctx){
		return AndroidDeviceInfo.checkPermission(ctx, "android.permission.WAKE_LOCK") && isServiceAvailable(ctx,"com.polestar.naosdk.controllers.AndroidGeofencingService");
	}
}
