package com.polestar.naosdk.managers;

import android.app.Service;
import android.content.ContextWrapper;

import com.polestar.Constants;
import com.polestar.helpers.AndroidDeviceInfo;
import com.polestar.helpers.AndroidPlatformThreads;
import com.polestar.helpers.ConnectivityHelper;
import com.polestar.helpers.HttpHelper;
import com.polestar.helpers.Log;
import com.polestar.helpers.PrefHelper;
import com.polestar.naosdk.api.GeofencePDB;
import com.polestar.naosdk.api.INAOServiceManager;
import com.polestar.naosdk.api.LoggerNaoLocationListener;
import com.polestar.naosdk.api.external.NAOSyncListener;
import com.polestar.naosdk.api.external.TPOWERMODE;
import com.polestar.naosdk.controllers.GeofencingOSController;
import com.polestar.naosync.NAOSyncListenerWrapper;
import com.polestar.naosync.SynchroBroker;

import java.util.ArrayList;

/**
 * Created by jchouki on 12/01/2016.
 */
public class NaoContext {
    public static class NAOProxy {
        public NAOProxy(String host, int port, String user, String password) {
            this.host = host;
            this.port = port;
            this.user = user;
            this.password = password;
        }
        String host;
        int port;
        String user;
        String password;
    }

    public static final String libnaojni = "naojni";
    public static final String libdjinni= "djinni";
    public static String rootURL;
    public static NAOProxy proxy;
    private Service myService;

    public boolean isStarted() {
        return isStarted;
    }

    private boolean isStarted = false;

    private HttpHelper httpHelper;
    private SynchroBroker synchroBroker;

    //Load library when class is loaded
    static {
        try {
            //System.loadLibrary(libdjinni);
            System.loadLibrary(libnaojni);
        } catch (UnsatisfiedLinkError e) {
            Log.alwaysError("NaoContext", libnaojni +  " : library not found! -- " + e.toString());
        }
    }

    public INAOServiceManager naoServiceManager;

    protected void finalize()
    {
        //naoServiceManager.stop(); //TODO this line caused crashes, so commented out for now, I dont think it's necessary
    }

    //stop service manager main loop thread
    public void shutdown()
    {
        naoServiceManager.stopAndDestroyServices();
        myNaoSensorsManager.quit();
        isStarted = false;
        //myService.stopSelf();
        NaoServiceManager.stopService();
    }

    private NaoSensorsManager myNaoSensorsManager;

    public NaoContext(ContextWrapper context){

        this.myNaoSensorsManager = new NaoSensorsManager(context);
        String rootPath = "";

        httpHelper = new HttpHelper(context, true);
        synchroBroker = new SynchroBroker(httpHelper);

        try {
            rootPath = context.getExternalFilesDir(Constants.NAO_ROOT_DIRECTORY).getAbsolutePath();
        } catch (Exception e) {
            Log.alwaysWarn(this.getClass().getName(),"NaoContext.<init> : Attempt to invoke method  context.getExternalFilesDir(Constants.NAO_ROOT_DIRECTORY).getAbsolutePath()");
        }

        this.naoServiceManager = INAOServiceManager.createInstance(myNaoSensorsManager,
                new AndroidPlatformThreads(),
                synchroBroker,
                httpHelper.new HttpHelperProxy(),
                new PrefHelper().new PrefHelperProxy(context),
                new ConnectivityHelper(). new ConnectivityHelperProxy(context),
                AndroidDeviceInfo.get(context),
                rootPath);

        if(this.rootURL != null)
            this.naoServiceManager.setRootURL(this.rootURL);


        if(this.proxy != null ){
            httpHelper.setHttpProxy(proxy.host, proxy.port, proxy.user, proxy.password, true);
        }

        //register GPS manager to native
        GeofencingOSController geofencingOSController = new GeofencingOSController(context);
        this.naoServiceManager.setGPSGeofenceController(geofencingOSController);

        isStarted = true;
    }

    public NaoContext(Service service){
        this((ContextWrapper) service);
        this.myService = service;
    }

    public NaoSensorsManager getSensorsManager() {

        return myNaoSensorsManager;
    }

    public void registerLoggerListener(LoggerNaoLocationListener listener) {
        myNaoSensorsManager.regsiterLoggerListener(listener);
    }

    public static String getRootURL(){
        return INAOServiceManager.getDefaultRootURL();
    }

    public static String getSoftwareVersion(){
        return INAOServiceManager.getSoftwareVersion();
    }

    public ContextWrapper getContext() {
        return myService;
    }

    /** Sets the root URL
     *   @param rootURL url of the server hosting NAO services input data
     *                  Default points to Pole Star NAO Cloud server.
     */
    public void setRootURL(String rootURL)
    {
        NaoContext.rootURL = rootURL;
        naoServiceManager.setRootURL(rootURL);
    }

    public void setProxy(String hostname, int port, String username, String password){
       httpHelper.setHttpProxy(hostname, port, username, password, true);
    }


    /**
     * Gets the global power mode of NAO SDK, resulting from all the various running Services.
     * If at least one service handle uses HIGH power mode, the global power mode will be HIGH.
     * @return power mode currently used in NAO SDK
     *
     */
    public TPOWERMODE getPowerMode()
    {
        return naoServiceManager.getPowerMode();
    }

    public void writeToLog(String tag, String txt){
        Log.alwaysWarn(tag, txt);
        naoServiceManager.writeToLog(tag + " : " + txt);
    }

    public void synchronizeData(String apiKey, NAOSyncListener listener){
        NAOSyncListenerWrapper listenerWrapper = new NAOSyncListenerWrapper(getContext(), apiKey, listener);
        naoServiceManager.synchronizeData(apiKey, listenerWrapper);
    }

    public static ArrayList<GeofencePDB> getGeofenceGPS(String absolutePath, String apiKey) {
        return INAOServiceManager.getGeofenceGPS(absolutePath,apiKey);
    }
}
