package com.polestar.naosdk.managers;

import android.content.Context;
import android.location.Location;
import android.os.Bundle;
import android.os.SystemClock;
import android.support.annotation.NonNull;

import com.polestar.helpers.Log;
import com.polestar.naosdk.api.INAOLocationClient;
import com.polestar.naosdk.api.INAOServiceProvider;
import com.polestar.naosdk.api.LocationFix;
import com.polestar.naosdk.api.LoggerNaoLocationListener;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOLocationListener;
import com.polestar.naosdk.api.external.NAOSensorsListener;
import com.polestar.naosdk.api.external.TNAOFIXSTATUS;
import com.polestar.naosdk.api.external.TPOWERMODE;
import com.polestar.naosdk.emulators.Emulator;
import com.polestar.naosdk.emulators.KmlPlayer;

/**
 * Created by jchouki on 14/12/2015.
 */
public class NAOLocationProvider extends INAOServiceProvider<NAOLocationListener, INAOLocationClient> {

    public static final String NAO_LOCATION_PROVIDER = "NAOSDK";
    public static final int NAONATIVEBRIDGE_MAX_NUMBER_OF_LIMITED_ACCURACY = 10;

    private int numberOfLimtedAccuracy = 0;
    private LoggerNaoLocationListener myLoggerNaoLocationListener;

    private boolean hasReceivedLocation = false;


    public NAOLocationProvider(Context context, Class<?> cls, @NonNull String apiKey, @NonNull NAOLocationListener client, @NonNull NAOSensorsListener sensorListener) {
        super(context, cls, apiKey, client, sensorListener);
    }

    @Override
    protected boolean registerClient() {
        return getServiceManager().registerLocationClient(mInternalClient, myApiKey, myISensorRequestListener);
    }

    @Override
    public void unregisterClient() {
        getServiceManager().unregisterLocationClient(mInternalClient);
    }

    @Override
    protected INAOLocationClient createInternalClient(){
        return this.new InternalClient();
    }

    private class InternalClient extends INAOLocationClient {

        @Override
        public void onNewLocation(LocationFix locFix) {

            if (!hasReceivedLocation) {
                hasReceivedLocation = true;
                Log.writeToLog(this.getClass().getName(), "First location received : " + locFix.getX() + ", " + locFix.getY() + ", " + locFix.getZ());
            }

            final Location loc = new Location(NAO_LOCATION_PROVIDER);

            // set date
            loc.setTime(locFix.getTime());

            // Note that the UTC time on a device is not monotonic: it can jump forwards or backwards unpredictably.
            // So always use getElapsedRealtimeNanos() when calculating time deltas.
            if(android.os.Build.VERSION.SDK_INT >=  android.os.Build.VERSION_CODES.JELLY_BEAN_MR1) {
                loc.setElapsedRealtimeNanos(SystemClock.elapsedRealtimeNanos());
            }
            // copy coordinates
            loc.setLongitude(locFix.getX());
            loc.setLatitude(locFix.getY());
            loc.setAltitude(locFix.getZ());

            // if the fix is not on the NAO graph, remove the altitude, which is neither accurate nor needed
            boolean isNao = locFix.getIsOnGraph();
            if (!isNao) {
                loc.removeAltitude();
            }

            // set accuracy
            double acc = locFix.getHrzAccuracy();
            if (acc > 0) {
                if (acc > 1000.0) {
                    acc = (float) (acc - 1000.0);
                    numberOfLimtedAccuracy++;
                    //control whether or not the app need to display a warning message about a poor location
                    if (myLoggerNaoLocationListener != null && numberOfLimtedAccuracy == NAONATIVEBRIDGE_MAX_NUMBER_OF_LIMITED_ACCURACY) {
                        myLoggerNaoLocationListener.deviceHasLimitedAccuracy();
                    }
                }
                loc.setAccuracy((float) acc);
            }

            // set heading
            float hea = (float) locFix.getHeading();
            if (hea >= 0 && hea < 360) {
                loc.setBearing(hea);
            } else if (hea < 0 && hea >= -360) {
                loc.setBearing(hea + 360);
                //control whether or not the app need to display calibration message to the final user
                //notifyCalibrationMessageWhenNeeded();
            } else if (hea < -360 && hea >= -720) {
                loc.setBearing(hea + 720);
            } else if (hea < 720 && hea >= 360) {
                loc.setBearing(hea - 360);
            }

            //set heading accuracy (value can be < 0) as extra  (value in  setBearing input will be wrapped into the range (0.0, 360.0])
            Bundle bundle = new Bundle();
            bundle.putFloat("heading_accuracy", hea);
            loc.setExtras(bundle);


            NAOLocationProvider.this.executeInHandler(new Runnable() {
                @Override
                public void run() {
                    if(null != myExternalListener){
                        myExternalListener.onLocationChanged(loc);
                    }
                }
            });
        }

        @Override
        public void onStatusChanged(final TNAOFIXSTATUS fixStatus) {
            NAOLocationProvider.this.executeInHandler(new Runnable() {
                @Override
                public void run() {
                    if(null != myExternalListener) {
                        myExternalListener.onLocationStatusChanged(fixStatus);
                    }
                }
            });
        }

        @Override
        public void onEnterSite(final String site) {
            NAOLocationProvider.this.executeInHandler(new Runnable() {
                @Override
                public void run() {
                    if(null != myExternalListener){
                        myExternalListener.onEnterSite(site);
                    }

                }
            });
        }

        @Override
        public void onExitSite(final String site) {
            NAOLocationProvider.this.executeInHandler(new Runnable() {
                @Override
                public void run() {
                    if(null != myExternalListener) {
                        myExternalListener.onExitSite(site);
                    }
                }
            });
        }

        @Override
        public void onError(final NAOERRORCODE code, final String msg) {
            Log.writeToLog(this.getClass().getName(), "onError code=" + code + " msg=" + msg);

            NAOLocationProvider.this.executeInHandler(new Runnable() {
                @Override
                public void run() {
                    if(null != myExternalListener) {

                        myExternalListener.onError(code, msg);
                    }
                }
            });
        }
    }

    @Override
    public Emulator createEmulator(){
        return new KmlPlayer(mInternalClient, myContext);
    }

    @Override
    protected void setPowerModeInternal(@NonNull TPOWERMODE mode) {
        getServiceManager().setLocationClientPowerMode(myPowerMode, mInternalClient);
    }

    public String getDatabaseVersions(){
        if(NaoServiceManager.getService().getNaoContext() != null) {
            return NaoServiceManager.getService().getNaoContext().naoServiceManager.getDatabaseVersions();
        }
        return "Unknown";
    }

    public void registerLoggerListener(LoggerNaoLocationListener loggerlistener) {
        myLoggerNaoLocationListener = loggerlistener;
    }
}
