package com.polestar.naosdk.managers;

import android.content.Context;
import android.support.annotation.NonNull;

import com.polestar.helpers.Log;
import com.polestar.naosdk.api.external.NAOGeofenceData;
import com.polestar.naosdk.api.external.NAOGeofenceListener;
import com.polestar.naosdk.api.external.NaoAlert;
import com.polestar.naosdk.api.INAOGeofencingClient;
import com.polestar.naosdk.api.INAOServiceProvider;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOGeofencingListener;
import com.polestar.naosdk.api.external.NAOSensorsListener;
import com.polestar.naosdk.api.external.TALERTRULE;
import com.polestar.naosdk.api.external.TPOWERMODE;
import com.polestar.naosdk.emulators.AlertsPlayer;
import com.polestar.naosdk.emulators.Emulator;


public class NAOGeofencingProvider extends INAOServiceProvider<NAOGeofencingListener, INAOGeofencingClient> {

	public NAOGeofencingProvider(Context context, Class<?> cls, @NonNull String apiKey, @NonNull NAOGeofencingListener client, @NonNull NAOSensorsListener sensorsListener) {
        super(context, cls, apiKey, client, sensorsListener);
	}

    @Override
    protected INAOGeofencingClient createInternalClient(){
        return this.new InternalClient();
    }

    private class InternalClient extends INAOGeofencingClient
    {
        @Override
        public void onFireNaoAlert(final NaoAlert alert, TALERTRULE rule){

            Log.writeToLog(getClass().getName(), "fireAlert : " + alert.getName());

            NAOGeofencingProvider.this.executeInHandler(
                new Runnable() {
                    @Override
                    public void run() { myExternalListener.onFireNaoAlert(alert); }
                });
            }

        @Override
        public void onGeofenceStatusChange(final TALERTRULE rule, final int regionId, final String regionName) {

            if(myExternalListener instanceof NAOGeofenceListener)
            {
                switch(rule)
                {
                    case ENTERGEOFENCERULE:
                        NAOGeofencingProvider.this.executeInHandler(
                                new Runnable() {
                                    @Override
                                    public void run() { ((NAOGeofenceListener)myExternalListener).onEnterGeofence(regionId, regionName); }
                                });
                        break;

                    case EXITGEOFENCERULE:
                        NAOGeofencingProvider.this.executeInHandler(
                                new Runnable() {
                                    @Override
                                    public void run() { ((NAOGeofenceListener)myExternalListener).onExitGeofence(regionId, regionName); }
                                });
                        break;
                }
            }
        }

        @Override
        public void onError(final NAOERRORCODE code, final String msg) {
            Log.writeToLog(this.getClass().getName(), "onError code=" + code + " msg=" + msg );

            NAOGeofencingProvider.this.executeInHandler(new Runnable() {
                    @Override
                    public void run() {
                            myExternalListener.onError(code, msg);
                    }
                });
        }
    }

    @Override
    protected boolean registerClient() {
        return getServiceManager().registerGeofencingClient(mInternalClient, myApiKey, myISensorRequestListener);
    }

    @Override
    protected void unregisterClient() {
        getServiceManager().unregisterGeofencingClient(mInternalClient);
	}

    @Override
    protected Emulator createEmulator(){
        return new AlertsPlayer(myExternalListener, this.myContext);
    }

	@Override
	protected void setPowerModeInternal(@NonNull TPOWERMODE mode) {
		getServiceManager().setGeofencingClientPowerMode(myPowerMode, mInternalClient);
	}


    public NAOGeofenceData getGeofenceData(int regionId){
        if(NaoServiceManager.getService().getNaoContext() != null) {
            try{
                return NaoServiceManager.getService().getNaoContext().naoServiceManager.getGeofenceData(regionId);

            } catch(RuntimeException e){
                Log.alwaysError(getClass().getSimpleName(), e.getMessage());
                return null;
            }
        }
        return null;
    }
}
