package com.polestar.naosdk.gatt;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.RequiresApi;
import android.util.Log;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import static android.bluetooth.BluetoothDevice.TRANSPORT_LE;

/**
 * Created by asimonigh on 13/04/2017.
 */

@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
public class GattManager extends GattInterface {
    private static final String TAG = "GattManager";

    private Context context;
    private BleConnectionCompat bleConnectionCompat;
    private INAOGattListener gattListener;

    private BluetoothAdapter mBluetoothAdapter;
    private BluetoothManager mBluetoothManager;


    private Handler mHandle = new Handler(Looper.getMainLooper());

    private ConcurrentHashMap<String, BluetoothGatt> myGattConnections;
    private ConcurrentHashMap<String,HashMap<UUID, UUID>> devicesServiceCharacteristicTable ;


    public GattManager(Context ctxt, INAOGattListener listener){
        context = ctxt;
        gattListener = listener;
        myGattConnections = new ConcurrentHashMap<>();
        devicesServiceCharacteristicTable = new ConcurrentHashMap<>();
        initBluetoothAdapter();
        bleConnectionCompat = new BleConnectionCompat(ctxt);

    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
    public void initBluetoothAdapter() {
        if (mBluetoothManager == null) {
            mBluetoothManager = (BluetoothManager) context.getSystemService(Context.BLUETOOTH_SERVICE);
            if (mBluetoothManager == null) {
                Log.e(getClass().getSimpleName(), "Unable to initialize BluetoothManager.");
                return;
            }
        }

        mBluetoothAdapter = mBluetoothManager.getAdapter();
        if (mBluetoothAdapter == null) {
            Log.e(getClass().getSimpleName(), "Unable to obtain a BluetoothAdapter.");
            return;
        }

    }


    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
    @Override
    public boolean connect(final String deviceMac, boolean autoconnect) {
        if (myGattConnections.containsKey(deviceMac) && myGattConnections.get(deviceMac) != null ) {
            //Device already connected
            Log.e(getClass().getSimpleName(), deviceMac + "already connected !");
            return false;
        }
        for(BluetoothDevice device : mBluetoothManager.getConnectedDevices(BluetoothProfile.GATT)){
            if(device.getAddress() == deviceMac){
                Log.e(getClass().getSimpleName(), deviceMac + "already connected but can't be used !");
                return false;
            }
        }


        final BluetoothDevice device = mBluetoothAdapter.getRemoteDevice(deviceMac);

        //use class from RXAndroidBle
        BluetoothGatt gatt = bleConnectionCompat.connectGatt(device, autoconnect, mGattCallback);
        try {
            refreshDeviceCache(gatt);
        } catch (Exception e) {
           Log.e(getClass().getSimpleName(), "Reflection error: cannot call refreshDeviceCache");
        }

//        BluetoothGatt gatt = device.connectGatt(context.getApplicationContext(), autoconnect, mGattCallback);
        if(null != gatt){
            myGattConnections.put(device.getAddress(), gatt);
            return true;
        }
        return false;
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
    @Override
    public void disconnect(String deviceMac) {
        Log.i(getClass().getSimpleName(), "call disconnect on: " + deviceMac);
        if(!myGattConnections.containsKey(deviceMac)){
            //Device already connected
            Log.e(getClass().getSimpleName(), deviceMac + "already disconnected !");
            return;
        }
        cleanDisconnectedDevice(deviceMac);
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
    private void cleanDisconnectedDevice(String device){
        if(myGattConnections.containsKey(device)){
            myGattConnections.get(device).close();
            myGattConnections.get(device).disconnect();
            myGattConnections.remove(device);
        }
    }


    @Override
    public boolean discoverServices(String device) {
       BluetoothGatt gatt = myGattConnections.get(device);
       if(null == gatt){
           Log.e(TAG, "discoverServices Gatt null");
           return false;
       }
       boolean res = myGattConnections.get(device).discoverServices();
        Log.i(getClass().getSimpleName(), "discover services: "+res);

       return res;
    }

    @Override
    public boolean read(String device, String serviceUUID, String characteristicUUID) {
        BluetoothGatt gatt = myGattConnections.get(device);
        if(null == gatt){
            return false;
        }

        UUID charactuuid = UUID.fromString(characteristicUUID);
        UUID serviceuuid = devicesServiceCharacteristicTable.get(device).get(charactuuid);
        BluetoothGattService service = gatt.getService(serviceuuid);
        BluetoothGattCharacteristic characteristic = service.getCharacteristic(charactuuid);
        boolean res = gatt.readCharacteristic(characteristic);

        return res;

    }

    @Override
    public boolean write(String device, String serviceUUID, String characteristicUUID, byte[] value) {
        Log.d(getClass().getSimpleName(), characteristicUUID);
        BluetoothGatt gatt = myGattConnections.get(device);
        if(null == gatt){
            return false;
        }
        BluetoothGattService service = gatt.getService(devicesServiceCharacteristicTable.get(device).get(UUID.fromString(characteristicUUID)));
        BluetoothGattCharacteristic characteristic = service.getCharacteristic(UUID.fromString(characteristicUUID));
        characteristic.setValue(value);
        boolean res = gatt.writeCharacteristic(characteristic);
//        Log.v(getClass().getSimpleName(), "write " + characteristicUUID + "res: " +res);

        return res;
    }

    @Override
    public void clearGatt() {
        for(BluetoothGatt gatt : myGattConnections.values()){
            gatt.close();
            gatt.disconnect();
        }
        myGattConnections.clear();
    }



    private boolean refreshDeviceCache(BluetoothGatt gatt) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        Log.v(getClass().getSimpleName(), "Trying to refresh device cache");
        BluetoothGatt localBluetoothGatt = gatt;
        Method refreshMethod = localBluetoothGatt.getClass().getDeclaredMethod("refresh");
        refreshMethod.setAccessible(true);
        return ((Boolean) refreshMethod.invoke(localBluetoothGatt)).booleanValue();
//            Method localMethod = localBluetoothGatt.getClass().getMethod("refresh", new Class[0]);
//            if (localMethod != null) {
//                boolean bool = ((Boolean) localMethod.invoke(localBluetoothGatt, new Object[0])).booleanValue();
//                return bool;
//            }
//
//            return false;
    }

    private final BluetoothGattCallback mGattCallback = new BluetoothGattCallback() {
        @Override
        public void onConnectionStateChange(final BluetoothGatt gatt, final int status, int newState) {

            Log.i(TAG, "OnConnectionStateChange: " + gatt.getDevice().getAddress() + " Status: " + status);
            if(status == 133)
            {
                cleanDisconnectedDevice(gatt.getDevice().getAddress());
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onGattError(gatt.getDevice().getAddress(), status );

                    }
                });
                thread.start();
            }

            if (newState == BluetoothProfile.STATE_CONNECTED && status == BluetoothGatt.GATT_SUCCESS) {

//				//If api > 21 request high priority connection
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    gatt.requestConnectionPriority(BluetoothGatt.CONNECTION_PRIORITY_HIGH);
                }
                Log.d(TAG, "Device" + gatt.getDevice().getAddress() + "connected!");
                myGattConnections.put(gatt.getDevice().getAddress(), gatt);
                Log.d(TAG, "Connection size: " + myGattConnections.size());
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onConnected(gatt.getDevice().getAddress());
                    }
                });
                thread.start();


            } else if (newState == BluetoothProfile.STATE_DISCONNECTED) {

                cleanDisconnectedDevice(gatt.getDevice().getAddress());

                Log.i(TAG, "Disconnected: " + gatt.getDevice().getAddress() );
                Log.d(TAG, "Connection size: " + myGattConnections.size());

                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        try{
                            gattListener.onDisconnected(gatt.getDevice().getAddress());
                        }
                        catch(Exception e){//catch exception when beacon has been disconnected while action are not completed
                            Log.e(TAG, e.getMessage());
                        }
                    }
                });
                thread.start();


            } else if (newState == BluetoothProfile.STATE_CONNECTING) {
                Log.d(TAG, "Connecting: " + gatt.getDevice().getAddress());
            }
            else {
                //unknown state, consider as error
                Log.i(TAG, "onGattError");
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onGattError(gatt.getDevice().getAddress(), status);
                    }
                });
                thread.start();
            }
        }

        @Override
        public void onServicesDiscovered(final BluetoothGatt gatt, final int status) {

            Log.i(TAG, "onService discovered status: " + status);
            if (status == BluetoothGatt.GATT_SUCCESS) {
                Log.i(TAG, gatt.getServices().size() + " services discovered");

                final HashSet<String> characteristics = new HashSet<>();
                HashMap<UUID, UUID> servicesCharac = new HashMap<>();
                for(BluetoothGattService service : gatt.getServices())
                {
                    for(BluetoothGattCharacteristic attr : service.getCharacteristics())
                    {
                        servicesCharac.put(attr.getUuid(), service.getUuid());
                        characteristics.add(attr.getUuid().toString().toUpperCase());
                    }
                }
                //store service characteristics association (used for read and write)
                devicesServiceCharacteristicTable.put(gatt.getDevice().getAddress(), servicesCharac);

                Log.i(TAG, characteristics.size() + " characteristics found");
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onServicesDiscovered(gatt.getDevice().getAddress(),characteristics);
                    }
                });
                thread.start();

            } else {
                Log.w(TAG, "onServicesDiscovered received: " + status);
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onGattError(gatt.getDevice().getAddress(), status);
                    }
                });
                thread.start();

            }
        }

        @Override
        public void onCharacteristicRead(final BluetoothGatt gatt, final BluetoothGattCharacteristic characteristic, final int status) {
            Log.d(TAG, "onCharacteristicRead :"+status);

            if(status == BluetoothGatt.GATT_SUCCESS && null != characteristic.getValue()){
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onReadAttribute(gatt.getDevice().getAddress(), characteristic.getUuid().toString().toUpperCase(), characteristic.getValue());
                    }
                });
                thread.start();
            }
            else{
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onGattError(gatt.getDevice().getAddress(), status); //TODO: add read error
                    }
                });
                thread.start();
            }
        }

        @Override
        public void onCharacteristicWrite(final BluetoothGatt gatt, final BluetoothGattCharacteristic characteristic, final int status) {
            Log.d(TAG, "onCharacteristicWrite :"+ status);
//            Log.v(getClass().getSimpleName(), "onWrite " + characteristic.getUuid().toString() + "status " + status);
            if(status == BluetoothGatt.GATT_SUCCESS ){
                //set UUID to upper case to be conformed to native c++
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onWriteAttribute(gatt.getDevice().getAddress(), characteristic.getUuid().toString().toUpperCase());
                    }
                });
                thread.start();

            }
            else{
                Thread thread = new Thread(new Runnable() {
                    @Override
                    public void run() {
                        gattListener.onGattError(gatt.getDevice().getAddress(), status);
                    }
                });
                thread.start();
            }
        }

        @Override
        public void onCharacteristicChanged(BluetoothGatt gatt, BluetoothGattCharacteristic characteristic) {
            super.onCharacteristicChanged(gatt, characteristic);
        }

        @Override
        public void onDescriptorRead(BluetoothGatt gatt, BluetoothGattDescriptor descriptor, int status) {
            super.onDescriptorRead(gatt, descriptor, status);
        }

        @Override
        public void onDescriptorWrite(BluetoothGatt gatt, BluetoothGattDescriptor descriptor, int status) {
            super.onDescriptorWrite(gatt, descriptor, status);
        }

        @Override
        public void onReliableWriteCompleted(BluetoothGatt gatt, int status) {
            super.onReliableWriteCompleted(gatt, status);
        }

        @Override
        public void onReadRemoteRssi(BluetoothGatt gatt, int rssi, int status) {
            super.onReadRemoteRssi(gatt, rssi, status);
        }

        @Override
        public void onMtuChanged(BluetoothGatt gatt, int mtu, int status) {
            super.onMtuChanged(gatt, mtu, status);
        }
    };


    /**
     * Class from RXAndroidBle
     */
    private class BleConnectionCompat {

        private final Context context;

        public BleConnectionCompat(Context context) {
            this.context = context;
        }

        public BluetoothGatt connectGatt(BluetoothDevice remoteDevice, boolean autoConnect, BluetoothGattCallback bluetoothGattCallback) {

            if (remoteDevice == null) {
                return null;
            }

            /**
             * Issue that caused a race condition mentioned below was fixed in 7.0.0_r1
             * https://android.googlesource.com/platform/frameworks/base/+/android-7.0.0_r1/core/java/android/bluetooth/BluetoothGatt.java#649
             * compared to
             * https://android.googlesource.com/platform/frameworks/base/+/android-6.0.1_r72/core/java/android/bluetooth/BluetoothGatt.java#739
             * issue: https://android.googlesource.com/platform/frameworks/base/+/d35167adcaa40cb54df8e392379dfdfe98bcdba2%5E%21/#F0
             */
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N || !autoConnect) {
                return connectGattCompat(bluetoothGattCallback, remoteDevice, autoConnect);
            }

            /**
             * Some implementations of Bluetooth Stack have a race condition where autoConnect flag
             * is not properly set before calling connectGatt. That's the reason for using reflection
             * to set the flag manually.
             */

            try {
                Log.v(getClass().getSimpleName(),"Trying to connectGatt using reflection.");
                Object iBluetoothGatt = getIBluetoothGatt(getIBluetoothManager());

                if (iBluetoothGatt == null) {
                    Log.w(getClass().getSimpleName(),"Couldn't get iBluetoothGatt object");
                    return connectGattCompat(bluetoothGattCallback, remoteDevice, true);
                }

                BluetoothGatt bluetoothGatt = createBluetoothGatt(iBluetoothGatt, remoteDevice);

                if (bluetoothGatt == null) {
                    Log.w(getClass().getSimpleName(),"Couldn't create BluetoothGatt object");
                    return connectGattCompat(bluetoothGattCallback, remoteDevice, true);
                }

                boolean connectedSuccessfully = connectUsingReflection(bluetoothGatt, bluetoothGattCallback, true);

                if (!connectedSuccessfully) {
                    Log.w(getClass().getSimpleName(),"Connection using reflection failed, closing gatt");
                    bluetoothGatt.close();
                }

                return bluetoothGatt;
            } catch (NoSuchMethodException
                    | IllegalAccessException
                    | IllegalArgumentException
                    | InvocationTargetException
                    | InstantiationException
                    | NoSuchFieldException exception) {
                Log.w(getClass().getSimpleName(), "Error during reflection" + exception);
                return connectGattCompat(bluetoothGattCallback, remoteDevice, true);
            }
        }

        private BluetoothGatt connectGattCompat(BluetoothGattCallback bluetoothGattCallback, BluetoothDevice device, boolean autoConnect) {
            Log.v(getClass().getSimpleName(),"Connecting without reflection");

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                return device.connectGatt(context, autoConnect, bluetoothGattCallback, TRANSPORT_LE);
            } else {
                return device.connectGatt(context, autoConnect, bluetoothGattCallback);
            }
        }

        private boolean connectUsingReflection(BluetoothGatt bluetoothGatt, BluetoothGattCallback bluetoothGattCallback, boolean autoConnect)
                throws NoSuchMethodException, InvocationTargetException, IllegalAccessException, NoSuchFieldException {
            Log.v(getClass().getSimpleName(),"Connecting using reflection");
            setAutoConnectValue(bluetoothGatt, autoConnect);
            Method connectMethod = bluetoothGatt.getClass().getDeclaredMethod("connect", Boolean.class, BluetoothGattCallback.class);
            connectMethod.setAccessible(true);
            return (Boolean) (connectMethod.invoke(bluetoothGatt, true, bluetoothGattCallback));
        }

        @TargetApi(Build.VERSION_CODES.M)
        private BluetoothGatt createBluetoothGatt(Object iBluetoothGatt, BluetoothDevice remoteDevice)
                throws IllegalAccessException, InvocationTargetException, InstantiationException {
            Constructor bluetoothGattConstructor = BluetoothGatt.class.getDeclaredConstructors()[0];
            bluetoothGattConstructor.setAccessible(true);
            Log.v(getClass().getSimpleName(),"Found constructor with args count = " + bluetoothGattConstructor.getParameterTypes().length);

            if (bluetoothGattConstructor.getParameterTypes().length == 4) {
                return (BluetoothGatt) (bluetoothGattConstructor.newInstance(context, iBluetoothGatt, remoteDevice, TRANSPORT_LE));
            } else {
                return (BluetoothGatt) (bluetoothGattConstructor.newInstance(context, iBluetoothGatt, remoteDevice));
            }
        }

        private Object getIBluetoothGatt(Object iBluetoothManager)
                throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

            if (iBluetoothManager == null) {
                return null;
            }

            Method getBluetoothGattMethod = getMethodFromClass(iBluetoothManager.getClass(), "getBluetoothGatt");
            return getBluetoothGattMethod.invoke(iBluetoothManager);
        }

        private Object getIBluetoothManager() throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

            BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();

            if (bluetoothAdapter == null) {
                return null;
            }

            Method getBluetoothManagerMethod = getMethodFromClass(bluetoothAdapter.getClass(), "getBluetoothManager");
            return getBluetoothManagerMethod.invoke(bluetoothAdapter);
        }

        private Method getMethodFromClass(Class<?> cls, String methodName) throws NoSuchMethodException {
            Method method = cls.getDeclaredMethod(methodName);
            method.setAccessible(true);
            return method;
        }

        private void setAutoConnectValue(BluetoothGatt bluetoothGatt, boolean autoConnect) throws NoSuchFieldException, IllegalAccessException {
            Field autoConnectField = bluetoothGatt.getClass().getDeclaredField("mAutoConnect");
            autoConnectField.setAccessible(true);
            autoConnectField.setBoolean(bluetoothGatt, autoConnect);
        }
    }

}


