//______________________________________________________________________________________
//
//  KmlPlayer.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________

package com.polestar.naosdk.emulators;
import java.util.TimerTask;


import android.content.ContextWrapper;
import android.location.Location;

import com.polestar.helpers.KmlPath;
import com.polestar.helpers.Log;
import com.polestar.naosdk.api.INAOLocationClient;
import com.polestar.naosdk.api.LocationFix;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.TNAOFIXSTATUS;

/**
 * Emulates Nao Positioning, providing positions from a KMl file<br/>
 *    
 */
public class KmlPlayer extends Emulator<INAOLocationClient>
{
	/** period between consecutive position outputs in milliseconds */
	public static final int OUTPUT_INITIAL_DELAY_MILLIS = 1000;
	public static final int OUTPUT_PERIOD_MILLIS = 1000;
	
	private KmlPath mReplayedPath;
	

	/**
	 * empty constructor
	 */
	public KmlPlayer(INAOLocationClient listener, ContextWrapper context) {
		super(listener, context);
	}

    @Override
    public int getOutputInitialDelayMillis() {
        return OUTPUT_INITIAL_DELAY_MILLIS;
    }
    @Override
    public int getOutputPeriodMillis() {
        return OUTPUT_PERIOD_MILLIS;
    }


    public String getEmulatorFileName()
	{
		return "nao.kml";
	}

	/**
	 * Open the file to replay and start the periodic output of measurements.<br/>
	 * Has no effect if the file is already open (user must use stop() first).
	 * 
	 * @return true if successful, false else.
	 */
	public boolean start() 
	{
		// open the file
		mReplayedPath = new KmlPath();
		if (mReplayedPath.readFromFile(mFileName) == false) 
		{
			mListener.onError(NAOERRORCODE.GENERIC_ERROR, "ERROR : cannot read KML file");
			Log.alwaysError(this.getClass().getName(),"ERROR : cannot read KML file");
			return false;
		}

		if (bIsRunning)
		{
			mListener.onError(NAOERRORCODE.GENERIC_ERROR, "ERROR : cannot start replay (object already in use)");
			Log.alwaysError(this.getClass().getName(),"ERROR : cannot start replay (object already in use)");
			return false;
		}
		
		mReplayIndex = 0;
				
		// start periodic timer
		if (!startPeriodicReading()) {
			mReplayIndex = -1;
			mReplayedPath = null;
			return false;
		}
		
		bIsRunning = true;
		Log.restricted(this.getClass().getName(),"Replay of file "+mFileName+" has started");
		return true;
	}	

	/**
	 * stop periodically reading measurements from the binary log file and close the file.<br/>
	 * if start is called afterwards, it starts back from the first measurement in the file.
	 */
	public void stop() 
	{
		stopPeriodicReading();
		bIsRunning = false;
		mReplayIndex = -1;
		mReplayedPath = null;
	}

    protected TimerTask createPeriodicFileReader()
    {
        return new PeriodicFileReader();
    }

    /**
     * a class to schedule measurement reading+output on a timer event
     */
    public class PeriodicFileReader extends TimerTask
    {
        @Override
        public void run()
        {
            if (mReplayIndex == 0){
                mListener.onStatusChanged(TNAOFIXSTATUS.NAO_FIX_AVAILABLE);
            }

            if ((mReplayIndex >= 0) && (mReplayIndex < mReplayedPath.size())) {
                if (mListener != null) {
                    Location l = mReplayedPath.get(mReplayIndex++);
                    l.setTime(System.currentTimeMillis());
                    com.polestar.helpers.Log.restricted("Nao location", "(lat,lon,alt) = " + l.getLatitude() + " , " + l.getLongitude() + " , " + l.getAltitude());
                    LocationFix locFix = new LocationFix(l.getTime(), l.getLongitude(), l.getLatitude(), l.getAltitude(),
                            l.getAccuracy(), l.getBearing(), true);
                    mListener.onNewLocation(locFix);
                }
            } else if (mReplayIndex >= 0) {
                // return to beginning of path
                mReplayIndex = 0;
            } else {
                if (mListener != null) {
                    mListener.onStatusChanged(TNAOFIXSTATUS.NAO_FIX_UNAVAILABLE);
                }
                // Now stop file reading / periodic notifications:
                stop();
            }
        }
    }


	/**
	 * pause the periodic reading of measurements
	 */
    @Override
	protected void stopPeriodicReading() {

        super.stopPeriodicReading();

		if (mListener != null) {
			mListener.onStatusChanged(TNAOFIXSTATUS.NAO_FIX_UNAVAILABLE);
		}
	}
}