package com.polestar.naosdk.emulators;

import android.content.ContextWrapper;

import com.polestar.naosdk.api.INAOServiceManager;
import com.polestar.naosdk.api.external.NaoAlert;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOGeofencingListener;

import java.util.ArrayList;
import java.util.List;
import java.util.TimerTask;

public class AlertsPlayer extends Emulator<NAOGeofencingListener> {

	public static final String EMULATOR_FILE_NAME = "alerts_emulator.json";
	public static final int OUTPUT_INITIAL_DELAY_MILLIS = 3000;
	public static final int OUTPUT_PERIOD_MILLIS = 3000;


	private List<NaoAlert> mReplayedPath;

	public AlertsPlayer(NAOGeofencingListener listener, ContextWrapper context) {
        super(listener, context);
	}

    @Override
	public int getOutputInitialDelayMillis() {
		return OUTPUT_INITIAL_DELAY_MILLIS;
	}
    @Override
	public int getOutputPeriodMillis() {
		return OUTPUT_PERIOD_MILLIS;
	}


	public String getEmulatorFileName() {
		return EMULATOR_FILE_NAME;
	}


	public boolean readFromFile() {
		try {

			mReplayedPath = INAOServiceManager.parseAlertsFromFile(mFileName);

		} catch (Exception e) {
			e.printStackTrace();
			return  false;
		}
		return true;
	}

	/**
	 * Open the file to replay and start the periodic output of measurements.<br/>
	 * Has no effect if the file is already open (user must use stop() first).
	 *
	 * @return true if successful, false else.
	 */

	public boolean start()
	{
        mReplayedPath = new ArrayList<>();

        readFromFile();

		if (bIsRunning) {
			onError("ERROR : cannot start replay (object already in use)");
			com.polestar.helpers.Log.alwaysError(this.getClass().getName(), "ERROR : cannot start replay (object already in use)");
			return false;
		}

		mReplayIndex = 0;

		// start periodic timer
		if (!startPeriodicReading()) {
			mReplayIndex = -1;
            mReplayedPath = null;
			return false;
		}

		bIsRunning = true;
		com.polestar.helpers.Log.restricted(this.getClass().getName(), "Replay of file " + mFileName + " has started");
		return true;
	}

	/**
	 * stop periodically reading measurements from the binary log file and close the file.<br/>
	 * if start is called afterwards, it starts back from the first measurement in the file.
	 */

	public void stop()
	{
		stopPeriodicReading();
		bIsRunning = false;
		mReplayIndex = -1;
        mReplayedPath = null;
	}

    public int getSize() {
        return mReplayedPath.size();
    }

	public void onError(final String message) {
		Runnable myRunnable = new Runnable() {
			@Override
			public void run() {
				if(mListener != null){
					mListener.onError(NAOERRORCODE.GENERIC_ERROR, message);
				}
			}
		};
		mainHandler.post(myRunnable);

		com.polestar.helpers.Log.alwaysError(this.getClass().getName(), message);
	}

	public void onRun() {
		if (mListener != null) {
			final NaoAlert a = mReplayedPath.get(mReplayIndex++);
			com.polestar.helpers.Log.restricted("Fire alert ", a.getName());
			Runnable myRunnable = new Runnable() {
				@Override
				public void run() {
					if(mListener != null){
						mListener.onFireNaoAlert(a);
					}
				}
			};
			mainHandler.post(myRunnable);
		}
	}


	/**
	 * a class to schedule measurement reading+output on a timer event
	 */
	public class PeriodicFileReader extends TimerTask
	{
		@Override
		public void run()
		{

			if ((mReplayIndex >= 0) && (mReplayIndex < getSize())) {
				onRun();
			} else if (mReplayIndex >= 0) {
				// return to beginning of path
				mReplayIndex = 0;
			} else {
				// Now stop file reading / periodic notifications:
				stop();
			}
		}
	}


    protected TimerTask createPeriodicFileReader()
    {
        return new PeriodicFileReader();
    }
}
