package com.polestar.naosdk.controllers;

import android.Manifest;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;

import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingClient;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.polestar.Constants;
import com.polestar.helpers.AndroidDeviceInfo;
import com.polestar.helpers.Log;
import com.polestar.helpers.PrefHelper;
import com.polestar.models.GeofenceTransition;
import com.polestar.naosdk.api.GeofencePDB;
import com.polestar.naosdk.api.IGPSGeofencingManager;
import com.polestar.naosdk.api.external.NAOWakeUpRegistrationListener;
import com.polestar.naosdk.api.external.TNAOWAKEUP_REGISTER_STATUS;
import com.polestar.naosdk.managers.NaoContext;

import java.util.ArrayList;
import java.util.List;

import static com.polestar.helpers.NaoUtils.isServiceAvailable;

/**
 * Created by ASimonigh on 07/11/2017.
 */

public class GeofencingOSController extends IGPSGeofencingManager {

    private Context context;
    private GeofencingClient mGeofencingClient;
    private ArrayList<Geofence> geofenceList;
    NAOWakeUpRegistrationListener mRegistrationListener;


    public GeofencingOSController(Context context) {
        this.context = context.getApplicationContext();
        geofenceList = new ArrayList<>();
        mGeofencingClient = LocationServices.getGeofencingClient(context);
    }

    /**
     * Register called from native after a synchronize success
     * @param geofences
     */
    @Override
    public void registerGPSGeofences(ArrayList<GeofencePDB> geofences) {
        registerGeofencesWithList(geofences);
    }

    public void registerGeofencesWithKey(String apiKey){
        if (apiKey != null && apiKey.length() > 2) {
            List<GeofencePDB> geofences = getGeofencesListFromKey(apiKey);

            if (!geofences.isEmpty()) {
                addGeofencesList(geofences);
                registerGeofences();

            } else {
                String message = "No GPS geofences to add found, app.json might be missing";
                notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
                Log.alwaysWarn(getClass().getName(), message);
            }
        }
    }

    public void registerGeofencesWithList(List<GeofencePDB> geofencesToRegister){
        if(geofencesToRegister.isEmpty()){
            String message = "No GPS Geofences to register ...";
            notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
            com.polestar.helpers.Log.alwaysWarn(getClass().getName(), message);
            return;
        }
        addGeofencesList(geofencesToRegister);
        registerGeofences();
    }

    public void unregisterGeofencesWithKey(String apiKey){
        if (apiKey != null && apiKey.length() > 2) {
            List<GeofencePDB> geofences = getGeofencesListFromKey(apiKey);

            if (!geofences.isEmpty()) {
                removeGeofences(geofences);
            }
            else{
                String message = "No GPS geofences to delete found , app.json might be missing";
                notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.UNREGISTER_ERROR, message);
                com.polestar.helpers.Log.alwaysWarn(getClass().getName(),message );
            }
        }
    }

    public void removeGeofences(List<GeofencePDB> geofences){
        //get id from geofence to remove
        List<String> removeIds = new ArrayList<>();
        for (GeofencePDB geofence : geofences) {
            //name is used as geofence id
            removeIds.add(geofence.getName());
        }
        mGeofencingClient.removeGeofences(removeIds)
                .addOnSuccessListener(new OnSuccessListener<Void>() {
                    @Override
                    public void onSuccess(Void aVoid) {
                        PrefHelper.put(context, PrefHelper.PREF_GEOFENCE_REGISTERED, false);

                        String message = "Geofences removed";
                        notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.UNREGISTERED, message);
                        Log.alwaysWarn(getClass().getName(), message);
                    }
                })
                .addOnFailureListener(new OnFailureListener() {
                    @Override
                    public void onFailure(@NonNull Exception e) {
                        String message = "Removing geofencesGPS failed !";
                        notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.UNREGISTER_ERROR, message);
                        Log.alwaysError(getClass().getName(), message);
                    }
                });
    }


    public void setListener(NAOWakeUpRegistrationListener  client){
        mRegistrationListener = client;
    }

    public boolean isEnabled(){
        return isServiceAvailable(context,"com.polestar.naosdk.controllers.AndroidGeofencingService");
    }

    public boolean isRegistered() {
        return PrefHelper.get(context, PrefHelper.PREF_GEOFENCE_REGISTERED, false);
    }

    /**
     *************************************
     *  Private methods
     ************************************
     */


    private void registerGeofences() {
        if(!hasPermission()){
            String message = "Geofencing registration needs ACCESS_FINE_LOCATION permission ";
            com.polestar.helpers.Log.alwaysWarn(getClass().getName(), message);
            notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
            return;
        }

        if(!isEnabled()){
            String message = "AndroidGeofencingService must be declared in the application Manifest.xml";
            com.polestar.helpers.Log.alwaysWarn(getClass().getName(), message);
            notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
            return;
        }

        if(geofenceList.isEmpty()){
            String message = "No GPS Geofences to register ...";
            notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
            com.polestar.helpers.Log.alwaysWarn(getClass().getName(),message );
        }

        mGeofencingClient.addGeofences(getGeofencingRequest(), getGeofencePendingIntent())
                .addOnSuccessListener(new OnSuccessListener<Void>() {
                    @Override
                    public void onSuccess(Void aVoid) {
                        PrefHelper.put(context, PrefHelper.PREF_GEOFENCE_REGISTERED, true);

                        String message = "Android geofencing registered!";
                        notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTERED, message);
                        com.polestar.helpers.Log.alwaysWarn(getClass().getName(), message);
                    }
                })
                .addOnFailureListener(new OnFailureListener() {
                    @Override
                    public void onFailure(@NonNull Exception e) {
                        String message = "Registering failed: " + e.getMessage();
                        notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS.REGISTER_ERROR, message);
                        com.polestar.helpers.Log.alwaysError(getClass().getName(), message);
                    }
                });
    }

    private void addGeofence(String id, double lat, double lng, float radius) {
        geofenceList.add(new Geofence.Builder()
                .setRequestId(id)
                .setCircularRegion(lat, lng, radius)
                .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER | Geofence.GEOFENCE_TRANSITION_EXIT)
                .setExpirationDuration(Geofence.NEVER_EXPIRE)
                .build());
    }

    private void addGeofencesList(List<GeofencePDB> listToAdd){
        for(GeofencePDB fence : listToAdd){
            geofenceList.add(new Geofence.Builder()
                    .setRequestId(fence.getName())
                    .setCircularRegion(fence.getLat(), fence.getLon(), fence.getRadius())
                    .setTransitionTypes(Geofence.GEOFENCE_TRANSITION_ENTER | Geofence.GEOFENCE_TRANSITION_EXIT)
                    .setExpirationDuration(Geofence.NEVER_EXPIRE)
                    .build());
        }
    }

    private void notifyRegistrationListener(TNAOWAKEUP_REGISTER_STATUS status, String message){
        if(mRegistrationListener != null){
            mRegistrationListener.onStatusChanged(status, message);
        }
    }

    private PendingIntent getGeofencePendingIntent(){
        Intent intent = new Intent(context, GeofenceTransition.class);
        return PendingIntent.getService(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);

    }

    private GeofencingRequest getGeofencingRequest(){
        GeofencingRequest.Builder builder  = new GeofencingRequest.Builder();
        builder.setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_ENTER)
                .addGeofences(geofenceList);

        return builder.build();
    }

    private List<GeofencePDB> getGeofencesListFromKey(String apiKey){

        List<GeofencePDB> geofences = new ArrayList<>();
        if (apiKey != null && apiKey.length() > 2) {

            try {
                if( context.getExternalFilesDir(Constants.NAO_ROOT_DIRECTORY)!=null) {
                    geofences.addAll(NaoContext.getGeofenceGPS(
                            context.getExternalFilesDir(Constants.NAO_ROOT_DIRECTORY).getAbsolutePath()
                            , apiKey));
                }
            } catch (UnsatisfiedLinkError  e) {
                com.polestar.helpers.Log.alwaysWarn(this.getClass().getName(),"UnsatisfiedLinkError   getGeofenceGPS");
            } catch (Exception e) {
                com.polestar.helpers.Log.alwaysWarn(this.getClass().getName(),"Exception thrown on context.getExternalFilesDir(Constants.NAO_ROOT_DIRECTORY).getAbsolutePath()");
            }
        }
        return geofences;
    }

    private boolean hasPermission(){
        if(AndroidDeviceInfo.checkPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)){
            return true;
        }
        else{
            return  false;
        }
    }
}
