package com.polestar.naosdk.api.external;

import com.polestar.helpers.Log;
import com.polestar.helpers.NaoUtils;
import com.polestar.helpers.PrefHelper;
import com.polestar.naosdk.api.INAOServiceProvider;
import com.polestar.naosdk.controllers.GeofencingOSController;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
* NAO Service Handle base class
*/
public abstract class NAOServiceHandle<Provider extends INAOServiceProvider>  {

    /**
     * @exclude - Pole Star internal use only
     */
    protected Provider myProvider;
    private ExecutorService serialExecutor = Executors.newSingleThreadExecutor();

    /**
     * Synchronizes NAO service data in your application with the data available on NAO Cloud or on your own server.
     * The data will not be downloaded if it is up-to-date.
     * The size of the configuration file depends of the size or your number of geofences but should be no more than 0,5 Mo.
     * @param listener listener implementing callbacks to be notified of synchronization success/failure
     */
    public void synchronizeData(final NAOSyncListener listener) {
        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    PrefHelper.put(myProvider.getApplicationContext(), PrefHelper.PREF_APP_KEY, myProvider.getApiKey());
                    myProvider.synchronizeData(listener);
                } else {
                    myProvider.notifyError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }

    /**
     * Registers handle with the service to start receiving updates.
     * The service will start if not already running.
     */
    public boolean start() {
        Log.alwaysWarn(this.getClass().getName(), this.getClass().getName() + " starting");

        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    PrefHelper.put(myProvider.getApplicationContext(), PrefHelper.PREF_APP_KEY, myProvider.getApiKey());
                    myProvider.start();
                } else {
                    myProvider.notifyError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });
        return true ;
    }

    private void registerOSGeofences() {
        if(myProvider.getApplicationContext() != null) {
            GeofencingOSController geofencingOSController = new GeofencingOSController(myProvider.getApplicationContext());
            geofencingOSController.setListener(null);
            geofencingOSController.registerGeofencesWithKey(myProvider.getApiKey());
        }
    }

    /**
     * Closes connection to the service, the associated listener will stop receiving updates.
     * If there are no more handles in running state, service will shut down.
     */
    public void stop() {
        Log.alwaysWarn(this.getClass().getName(), this.getClass().getName() + " stopping.");

        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    myProvider.stop();
                } else {
                    myProvider.notifyError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });
    }

    /**
     * Gets API key used by the service.
     * @return api key
     */
    public String getApiKey() {
        return myProvider.getApiKey();
    }

    /**
     * Sets the service power mode.
     * @param power: power mode
     */
    public void setPowerMode(final TPOWERMODE power) {
        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    myProvider.setPowerMode(power);
                } else {
                    myProvider.notifyError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }

    /**
     * Gets the power mode the service is running on.
     * @return service power mode
     */
    public TPOWERMODE getPowerMode(){
        return myProvider.getPowerMode();
    }
}
