package com.polestar.naosdk.api;

import android.content.Context;
import android.content.ContextWrapper;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.polestar.helpers.AssetCopier;
import com.polestar.helpers.LocalPathHelper;
import com.polestar.helpers.Log;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOErrorListener;
import com.polestar.naosdk.api.external.NAOSensorsListener;
import com.polestar.naosdk.api.external.NAOSyncListener;
import com.polestar.naosdk.api.external.TPOWERMODE;
import com.polestar.naosdk.emulators.AlertsPlayer;
import com.polestar.naosdk.emulators.Emulator;
import com.polestar.naosdk.managers.ISensorRequestListenerImpl;
import com.polestar.naosdk.managers.NaoServiceManager;
import com.polestar.naosync.NAOSyncListenerWrapper;

import java.io.File;

public abstract class INAOServiceProvider<ExternalListener extends NAOErrorListener, InternalServiceClient> {

    final Handler mainHandler;

    protected final ContextWrapper myContext;
    protected final LocalPathHelper mPathHelper;
    protected final String myApiKey;
    protected final ExternalListener myExternalListener;
    protected final ISensorRequestListenerImpl myISensorRequestListener;

    private Emulator myEmulator;

    protected InternalServiceClient mInternalClient;

    protected TPOWERMODE myPowerMode = TPOWERMODE.HIGH;


    public INAOServiceProvider(Context context, Class<?> cls, @NonNull String apiKey, @NonNull ExternalListener client, @NonNull NAOSensorsListener sensorsListener) {
        myApiKey = apiKey;
        myExternalListener = client;

        myContext = new ContextWrapper(context);

        mainHandler = new Handler(myContext.getApplicationContext().getMainLooper());
        myISensorRequestListener = new ISensorRequestListenerImpl(mainHandler, sensorsListener, myContext);
        mPathHelper = new LocalPathHelper(context);

        NaoServiceManager.startService(context, cls);

        if (myApiKey.equalsIgnoreCase(AlertsPlayer.EMULATOR_KEY)) {
            Log.alwaysWarn(this.getClass().getName(), "Using EMULATOR_KEY");
        }
        mInternalClient = createInternalClient();
    }

    protected INAOServiceManager getServiceManager(){ return NaoServiceManager.getService().getNaoContext().naoServiceManager; }
    protected Handler getMainHandler() {
        return mainHandler;
    }
    public Context getApplicationContext() {
        return myContext.getApplicationContext();
    }

    protected abstract InternalServiceClient createInternalClient();

    protected Emulator createEmulator(){
        return null;
    }

    void copyAssetDirectory() {
        AssetCopier assetCopier = new AssetCopier(myContext);
        File tmpAppFile = new File(getServiceManager().getRootDir() + File.separator + LocalPathHelper.APP_JSON_FILE_NAME);
        //copy app.json file to tmp dircetory or return if not exist
        if(!assetCopier.copyAssetFile(tmpAppFile.getName(), tmpAppFile.getParent()))
            return;

        //check if need to copy assets
        String appFileDir = getServiceManager().getAppLocalDir(myApiKey);
        String appFilePath = getServiceManager().getAppFilePath(myApiKey);
        boolean overwriteAssets = INAOServiceManager.needsToCopyAssets(appFilePath,tmpAppFile.getAbsolutePath());
        assetCopier.copyAssetDirectory(LocalPathHelper.NAO_RELATIVE_PDBS_DIRECTORY, appFileDir, overwriteAssets);

        File appJsonFile = new File(appFilePath);
        if(overwriteAssets || !appJsonFile.exists()) {
            assetCopier.copyAssetFile(appJsonFile.getName(), appFileDir);
        }

        //delete app.json from tmp directory
        try {
            tmpAppFile.delete();
        } catch (Exception e){ /**/ }

    }

    protected abstract boolean registerClient();
    protected abstract void unregisterClient();

    protected void executeInHandler(Runnable runnable){
        mainHandler.post(runnable);
    }

    public boolean start(){

        copyAssetDirectory();

        if (myApiKey.equalsIgnoreCase(Emulator.EMULATOR_KEY)) {
            if (myEmulator == null) {
                myEmulator = createEmulator();
                if (myEmulator == null) {
                    // no Emulator supported - TODO: throw error ?
                    return false;
                } else{
                    AssetCopier assetCopier = new AssetCopier(myContext);
                    assetCopier.copyAssetFile(myEmulator.getEmulatorFileName(), mPathHelper.localDir());
                }
            }

            myEmulator.start();
            return true;

        }else{

            boolean res =  registerClient();
            setPowerMode(myPowerMode);
            return res;
        }
    }

    public final void stop() {
        if (myEmulator != null){
            myEmulator.stop();
        }else{
            unregisterClient();
        }
    }

    public void synchronizeData(final NAOSyncListener listener) {

        NAOSyncListenerWrapper listenerWrapper = new NAOSyncListenerWrapper(myContext, myApiKey, listener);

        if (!myApiKey.equalsIgnoreCase(Emulator.EMULATOR_KEY)) {
            getServiceManager().synchronizeData(myApiKey, listenerWrapper);
        }else{
            listenerWrapper.onSyncSuccess();
            Log.alwaysWarn(this.getClass().getName(), "onSynchronizationSuccess...");
        }
    }

    public TPOWERMODE getPowerMode(){
        return myPowerMode;
    }

    public void setPowerMode(@NonNull TPOWERMODE mode) {
        myPowerMode = mode;
        setPowerModeInternal(mode);
        Log.restricted(this.getClass().getName(), "Set Power Mode to " + myPowerMode.name());
    }

    protected abstract void setPowerModeInternal(TPOWERMODE power);

    public String getApiKey() {
        return myApiKey;
    }

    public void notifyError(NAOERRORCODE code, String msg) {
        if(myExternalListener!=null) {
            myExternalListener.onError(code,msg);
        }

    }

}
