package com.polestar.models;

import android.os.Parcel;
import android.os.ParcelUuid;

import com.polestar.advertisement.ByteUtilsHelper;
import com.polestar.helpers.AssignedUuid;
import com.polestar.helpers.Log;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Vector;


public class BleMeasurement extends Measurement  {
	
	// available data keys
	public static final int BLE_MEAS_DATA_UNKNOWN = 0;
	public static final int BLE_MEAS_DATA_RSSI = 1; 
	public static final int BLE_MEAS_DATA_LOCAL_NAME = 2;
	public static final int BLE_MEAS_DATA_SERVICE_UUIDS = 3;
	public static final int BLE_MEAS_DATA_POWER_LEVEL = 4;
	public static final int BLE_MEAS_DATA_MANUFACTURER= 5;
	public static final int BLE_MEAS_DATA_SERVICE_DATA = 6;
	public static final int BLE_MEAS_DATA_ISCONNECTABLE = 7; // Not available on Android
	public static final int BLE_MEAS_DATA_CHANNEL = 8; 	// Not available on Android
	public static final int BLE_MEAS_DATA_ADDRESS = 9; // Not available on ios
	
	public static final int BLE_MEAS_DATA_IBEACON_UUID = 10; // Not available on Android
	public static final int BLE_MEAS_DATA_IBEACON_MAJOR = 11; // Not available on Android
	public static final int BLE_MEAS_DATA_IBEACON_MINOR = 12; // Not available on Android
	public static final int BLE_MEAS_DATA_IBEACON_THRES = 13; // Not available on Android
	
	public static final int BLE_MEAS_DATA_SERVICE_DATA_16 = 6;
	public static final int BLE_MEAS_DATA_SERVICE_DATA_32 = 14;
	public static final int BLE_MEAS_DATA_SERVICE_DATA_128 = 15;
	
	public static final int BLE_MEAS_DATA_RAW = 255;

    public static final int BLE_MEAS_LENGTH_SERVICE_UUID_16=2;
    public static final int BLE_MEAS_LENGTH_SERVICE_UUID_32=4;
    public static final int BLE_MEAS_LENGTH_SERVICE_UUID_128=16;

	private long mTimeStamp;    // timestamps of the last bleData added
	private int mLength;
	private Vector<BleData> mBleData;

	public BleMeasurement() {
	   reset();
	}

	public void reset(){
		mLength = 0;
		mTimeStamp = 0;
		mBleData = new Vector<>();
	}

	/**
	 * Add a BleData object
	 * we suppose the data is added in chronologic order
	 * @param data
	 */
	public void addBleData(BleData data){
		mBleData.add(data);
		mLength+=1;
		mTimeStamp = data.mTimestamp;
	}
	
	public int computeLength(){
		  int len = 0;
		  int dataLen=0;
		    // Add length
		    len += Short.SIZE/8; // + 2 bytes
		    // Add size of number of meas (nbMeas)
		    len += Short.SIZE/8; // + 2 bytes
		    
		    for (Iterator<BleData> iterator = mBleData.iterator(); iterator.hasNext();) {
				BleData ap = (BleData) iterator.next();
		    
				// For each measured AP, 
				len+= Long.SIZE/8; // + 8 bytes (timestamp)
		        len+= 1; // + 1 bytes (nb data)     
		        //......................................
			        //Key #1 : BLE_MEAS_DATA_RSSI 
			        len+= 1;// + 1 bytes (key)
			        len+= Short.SIZE/8; // + 2 bytes (Length)
			        len+= Short.SIZE/8; // + RSSI 2 bytes (value)
	                if ((ap.mAdress != null) ) {
	                	// Key #9 BLE_MEAS_DATA_ADDRESS 
	                	len+= 1;// + 1 bytes (key)
	                	len+=Short.SIZE/8; // + 2 bytes (Length)
	                	dataLen = ap.mAdress.length();
	                	len+=dataLen; // + L bytes (value)
	                }
	                /* ToDo: Add under debug flag */
	                /*
	                if ((ap.mRawData != null)) {
		                //Key #255 : BLE_MEAS_DATA_RAW
				        len+= 1;// + 1 bytes (key)
		                len+=Short.SIZE/8; // + 2 bytes (Length)
		                dataLen = ap.mRawData.length;
		                len+=dataLen; // + L bytes (value)   
			        }*/
			        
			        if ((ap.mName != null) && (!ap.mName.equalsIgnoreCase(""))) {
			        	//Key #2 : BLE_MEAS_DATA_LOCAL_NAME
			        	len+= 1;// + 1 bytes (key)
			        	len+=Short.SIZE/8; // + 2 bytes (Length)
			        	dataLen = ap.mName.length();
			        	len+=dataLen; // + L bytes (value)
			        }
			        if ((ap.mServiceUUIDs != null)) {
                        //Key #3 : BLE_MEAS_DATA_SERVICE_UUID
                        for (ParcelUuid uuid : ap.mServiceUUIDs) {
                            AssignedUuid auid = new AssignedUuid(uuid);
                            int uLen = auid.uuidLength();
                            if(uLen>0) {
                                len += 1;// + 1 bytes (key)
                                len += Short.SIZE / 8; // + 2 bytes (Length)
                                dataLen = uLen;
                                len += dataLen; // + L bytes (value)
                            }
                        }
			        } 
	                if ((ap.mTxPowerLevel != 0) ) {
	                	//Key #4 : BLE_MEAS_DATA_POWER_LEVEL
	                	len+= 1;// + 1 bytes (key)
	                	len+=Short.SIZE/8; // + 2 bytes (Length)
	                	dataLen = 1;
	                	len+=dataLen; // + L bytes (value)
	                }
	                if ((ap.mManufacturerSpecificData != null) ) {
	                	//Key #5 : BLE_MEAS_DATA_MANUFACTURER
                        for (int index=0;index<ap.mManufacturerSpecificData.size();index++) {
                            int mLen = Short.SIZE / 8;
                            int dLen = ap.mManufacturerSpecificData.valueAt(index).length;

                            len += 1;// + 1 bytes (key)
                            len += Short.SIZE / 8; // + 2 bytes (Length)
                            dataLen = mLen+dLen;

                            len += dataLen; // + L bytes (value)
                        }
	                }
					if ((ap.mServiceData != null) && (ap.mServiceData.keySet()!=null) ) {
						//Key #6 : BLE_MEAS_DATA_SERVICE_DATA
                        for (ParcelUuid uuid : ap.mServiceData.keySet()) {
                            AssignedUuid auid = new AssignedUuid(uuid);
                            int uLen = auid.uuidLength();
                            int dLen = ap.mServiceData.get(uuid).length;

                            len+= 1;// + 1 bytes (key)
                            len+=Short.SIZE/8; // + 2 bytes (Length)
                            dataLen = uLen + dLen;
                            len+=dataLen; // + L bytes (value)
                        }
					}
                }
	                // Key #7 : BLE_MEAS_DATA_ISCONNECTABLE - > Not available 
	                // Key #8 : BLE_MEAS_DATA_CHANNEL - > Not available 

		return len;
	}

	@Override
	public byte[] toByteArray() {
		// compute array length :
		int serializedBytes = this.computeLength();
		// create new output stream with default size
		ByteArrayOutputStream bos = new ByteArrayOutputStream(serializedBytes);
		DataOutputStream dos = new DataOutputStream(bos);
		

		// Writing in the buffer
		try {
			dos.writeShort(serializedBytes); // 2 Bytes
			dos.writeShort(getNbMeas()); // 2 Bytes
			
			for (Iterator<BleData> iterator = mBleData.iterator(); iterator.hasNext();) {
				BleData ap = iterator.next();
				// add date=uint64 (8)
		        long timeStamp = (long) ap.mTimestamp;
		        dos.writeLong(timeStamp);

		        // count following data keys
		        int nbKeys = 1; // rssi
		        if ((ap.mAdress != null)) {
		        	nbKeys++;
		        }
		        /* ToDo: Add under debug flag */
                /*
		        if(ap.mRawData!=null){
		        	nbKeys++;
		        }
		        */
		        if ((ap.mName != null) && (!ap.mName.equalsIgnoreCase(""))) {
		        	nbKeys++;
		        }
		        if ((ap.mServiceUUIDs != null) ) {
		        	nbKeys+=ap.mServiceUUIDs.size();
		        }
		        if ((ap.mTxPowerLevel != 0) ) {
		        	nbKeys++;
		        }
		        if ((ap.mServiceData != null)) {
		        	nbKeys+=ap.mServiceData.size();
		        }
		        if ((ap.mManufacturerSpecificData != null)) {
		        	nbKeys+=ap.mManufacturerSpecificData.size();
		        }
		        
		        dos.writeByte(nbKeys);
		        
		        // add Key #1 : BLE_MEAS_DATA_RSSI uint16 (2)
		        dos.writeByte(BLE_MEAS_DATA_RSSI); // key ID : uint8 (1) 
		        dos.writeShort(Short.SIZE/8); // key value length : uint16 (2) 
		        dos.writeShort(Math.abs(ap.mRssi));
		        
		        if ((ap.mAdress != null)   && (!ap.mAdress.equalsIgnoreCase("")) ) {
                	//Key #9 : BLE_MEAS_DATA_ADDRESS
		        	dos.writeByte(BLE_MEAS_DATA_ADDRESS); // key ID : uint8 (1) 
			        dos.writeShort(ap.mAdress.length()); // key value length :char[] (N) 
			        dos.writeBytes(ap.mAdress);
                } 
		        /* ToDo: Add under debug flag */
                /*
		        if ((ap.mRawData != null)) {
		        	// add Key #255 : BLE_MEAS_RAW char[] (N)
		        	dos.writeByte(BLE_MEAS_DATA_RAW); // key ID : uint8 (1) 
		        	dos.writeShort(ap.mRawData.length); // key value length :char[] (N) 
		        	dos.write(ap.mRawData);
		        }	
				*/
		        if ((ap.mName != null) && (!ap.mName.equalsIgnoreCase(""))) {
		        	// add Key #2 : BLE_MEAS_DATA_LOCAL_NAME char[] (N)
			        dos.writeByte(BLE_MEAS_DATA_LOCAL_NAME); // key ID : uint8 (1) 
			        dos.writeShort(ap.mName.length()); // key value length :char[] (N) 
			        dos.writeBytes(ap.mName);
		        }
				//Key #3 : BLE_MEAS_DATA_SERVICE_UUID
				for (ParcelUuid uuid : ap.mServiceUUIDs) {
					AssignedUuid auid = new AssignedUuid(uuid);
					int uLen = auid.uuidLength();

					dos.writeByte(BLE_MEAS_DATA_SERVICE_UUIDS); // key ID : uint8 (1)
					dos.writeShort(uLen); // key value length :char[] (N)
					dos.write(auid.uuidBytes());
				}

		        if ((ap.mTxPowerLevel != 0) ) {
		        	// add Key #4 : BLE_MEAS_DATA_POWER_LEVEL int8
		        	dos.writeByte(BLE_MEAS_DATA_POWER_LEVEL); // key ID : uint8 (1) 
			        dos.writeShort(1); // key value length :int8 (N) 
			        dos.writeByte((byte) ap.mTxPowerLevel);
		        }

				// add Key #5 : BLE_MEAS_DATA_MANUFACTURER char[] (N)
				for (int index=0;index<ap.mManufacturerSpecificData.size();index++) {
					byte [] manufacturerId = ByteUtilsHelper.getBytesFromShort((short) (ap.mManufacturerSpecificData.keyAt(index)));
					byte [] manufacturerData = ap.mManufacturerSpecificData.valueAt(index);
					ByteUtilsHelper.invertArray(manufacturerId);
					dos.writeByte(BLE_MEAS_DATA_MANUFACTURER); // key ID : uint8 (1)
					dos.writeShort(manufacturerId.length + manufacturerData.length); // key value length :char[] (N)
					dos.write(manufacturerId);
					dos.write(manufacturerData);
				}

                if ((ap.mServiceData != null) && (ap.mServiceData.keySet()!=null)) {
                    // add Key #6 : BLE_MEAS_DATA_SERVICE_DATA
                    for (ParcelUuid uuid : ap.mServiceData.keySet()) {
                        // for each available Service Data associated to its UUID
                        AssignedUuid auid = new AssignedUuid(uuid);
                        int uLen = auid.uuidLength();
                        int dLen = ap.mServiceData.get(uuid).length;
                        // add key enumerator
                        dos.writeByte(BleMeasurement.serviceDataKeyForUuidLength(uLen)); // key ID : uint8 (1)
                        // add value len (uuid len + data len)
                        dos.writeShort(uLen+dLen); // key value length (uuid + data) :char[] (U+D)
                        // add value (uuid...)
                        dos.write(auid.uuidBytes());
                        // add value (...data)
                        byte [] data = ap.mServiceData.get(uuid);
                        dos.write(data);
                    }
                }
			}
			if (dos.size() != serializedBytes){
				Log.alwaysWarn(this.getClass().getName(), " We only send " + dos.size() + "/" + serializedBytes + " bytes, something's wrong!");
			} else {
				/*DBG*///Log.alwaysWarn(this.getClass().getName()," We  send " + dos.size() + "/" + serializedBytes + " bytes, everything ok !");
			}
			                   
			dos.flush();	
			dos.close();	
		} catch (IOException ie) {
			// error : impossible to write in the stream
			return null;
		}
		
		mLength = serializedBytes;
		
		return bos.toByteArray();
		
	}
public static int serviceDataKeyForUuidLength(int length){
        switch (length)
        {
            case BLE_MEAS_LENGTH_SERVICE_UUID_16:
                return BLE_MEAS_DATA_SERVICE_DATA_16;
            case BLE_MEAS_LENGTH_SERVICE_UUID_32:
                return BLE_MEAS_DATA_SERVICE_DATA_32;
            case BLE_MEAS_LENGTH_SERVICE_UUID_128:
                return BLE_MEAS_DATA_SERVICE_DATA_128;
            default:
                break;

        }
        return BLE_MEAS_DATA_SERVICE_DATA;
    }

	@Override
	public String toHumanString() {
		return "BleMeasurement :: nbMEas " + getNbMeas();
	}

	
	/********************************************
	 * SETTERS & GETTERS
	 ********************************************/
	
	public int getNbMeas(){
		return (mBleData == null) ? 0 : mBleData.size();
	}

	public long getmTimeStamp() {
		return mTimeStamp;
	}

	public void setmTimeStamp(long mTimeStamp) {
		this.mTimeStamp = mTimeStamp;
	}

	public int getmLength() {
		return mLength;
	}

	public int getLength() {
		return mLength;
	}

	public void setmLength(int len) {
		this.mLength = (short) len;
	}


	public Vector<BleData> getBleData() {
		return mBleData;
	}

	public void setBleData(Vector<BleData> bleData) {
		this.mBleData = bleData;
	}
	
	public void writeToParcel(Parcel dest, int flags) {
		dest.writeLong(this.mTimeStamp);
		dest.writeInt(mBleData.size());
		
	
		for (BleData a:this.mBleData) {
			dest.writeString(a.mAdress);
			dest.writeInt(a.mBabid);
			dest.writeInt(a.mRssi);			
		}
	}
	
	
	private void readFromParcel(Parcel src) {

		// clean up first
		this.mBleData.clear();
		
		//read timstamp
		this.mTimeStamp = src.readLong(); 
		// read the number of AP stored
		int nAP = src.readInt();

		// read the list of bluespot
		for(int i = 0; i < nAP; i++) {
			BleData a = new BleData();
			a.mAdress = src.readString();
			a.mBabid = src.readInt();
			a.mRssi = src.readInt();
			this.mBleData.add(a);
		}

	}
}
