package com.polestar.models;

import android.os.ParcelUuid;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.bluetooth.le.ScanRecord;
import android.util.SparseArray;

import com.polestar.advertisement.Advertisement;
import com.polestar.advertisement.AdvertisementData;
import com.polestar.advertisement.AdvertisementHelper;

import com.polestar.advertisement.ByteUtilsHelper;
import com.polestar.helpers.AssignedUuid;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class BleData {
    public double mPeriod;
    public long mTimestamp;
    public String mAdress;
    public byte[] mRawData;
    public int mRssi; //received signal strength
    // AD data dictionary
    public String mName;
	public SparseArray<byte[]> mManufacturerSpecificData;
	public Map<ParcelUuid, byte[]> mServiceData;
	public List<ParcelUuid> mServiceUUIDs;

   	public int mTxPowerLevel;
   	public boolean mIsConectable; // Currently Not Available on Android
    public int mChannel; // Currently Not Available on Android
    
    public int mBabid;
    
	public BleData() {
	    super();
        mPeriod = 0.0;
	    mTimestamp = 0;
	    mAdress = "";
	    mRawData = null;
	    mRssi= 0;
	    // AD data dictionary
	    mName="";
		mManufacturerSpecificData = new SparseArray();
		mServiceData = new HashMap<>();
		mServiceUUIDs= new ArrayList<>();
	    mTxPowerLevel = 0;

	    // not available with Android
	    mIsConectable = false;
	    mChannel = 0;
		mBabid = 0;
	}
	
	public BleData(BleData aBleData){
        //mPeriod = aBleData.mPeriod;
		mTimestamp = aBleData.mTimestamp;
	    mAdress = aBleData.mAdress;
		mRawData = aBleData.mRawData;
	    mRssi = aBleData.mRssi;
	    // AD data dictionary
	    mName = aBleData.mName;
	    mManufacturerSpecificData = aBleData.mManufacturerSpecificData;
	    mServiceData = aBleData.mServiceData;
		mServiceUUIDs = aBleData.mServiceUUIDs;
		mTxPowerLevel = aBleData.mTxPowerLevel;
	    
	    // Not available with Android
	    mChannel = aBleData.mChannel;
	    mIsConectable = aBleData.mIsConectable;

		mBabid = aBleData.mBabid;
	}

	@RequiresApi(21)
	public BleData(long timestamp,String adress,ScanRecord scanRecord, int rssi){
        super();
		this.mTimestamp = timestamp;
		this.mAdress = adress;
		this.mRawData = scanRecord.getBytes();
		this.mRssi = rssi;

        mManufacturerSpecificData = new SparseArray();
        mServiceData = new HashMap<>();
        mServiceUUIDs= new ArrayList<>();

        // Retrieve additional data (API >=21)
		this.parseBleDataFromScanRecord(scanRecord);

	}

	public BleData(long timestamp,String adress,byte [] scanRecord, int rssi){
        super();
		this.mTimestamp = timestamp;
		this.mAdress = adress;
		this.mRawData = scanRecord;
		this.mRssi = rssi;

        mManufacturerSpecificData = new SparseArray();
        mServiceData = new HashMap<>();
        mServiceUUIDs= new ArrayList<>();

		// Retrieve additional data (API<21)
		this.parseBleDataFromByteArray(this.mRawData);
		
	}

	private void parseBleDataFromByteArray(byte [] rawData){
		try {
			Advertisement advDatas = AdvertisementHelper.parseScanRecord(rawData);
			// Retrieve local name if present
			AdvertisementData namData = AdvertisementHelper.getADrecord(advDatas,AdvertisementData.TYPE_LOCAL_NAME_COMPLETE);
			if (namData!=null){
				this.mName = new String(namData.getData());
			}
			// Retrieve power level if present
			AdvertisementData txpData =AdvertisementHelper.getADrecord(advDatas,AdvertisementData.TYPE_TX_POWER_LEVEL);
			if (txpData != null) {	
				this.mTxPowerLevel =  (int) (txpData.getData()[0]);
			}
			// Retrieve manufacturer data if present
            Advertisement manDatas = new Advertisement();
            manDatas.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_MANUFACTURER_SPECIFIC_DATA));
           	for(AdvertisementData manData:manDatas) {
				// ToDo: Quick'n'Dirt
                byte [] data = manData.getData().clone();
                if(data.length>1) {
                    int manuf = data[0] + ((int) data[1] << 8);
                    this.mManufacturerSpecificData.append(manuf, Arrays.copyOfRange(data, 2, data.length));
                }
			}
			// Retrieve service uuid list if present
            Advertisement srvUuids = new Advertisement();
			srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID16));
			srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID16_INC));
            srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID32));
			srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID32_INC));
            srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID128));
			srvUuids.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_UUID128_INC));
			for (AdvertisementData srvUuid:srvUuids){
                byte[] dui = Arrays.copyOfRange(srvUuid.getData(), 0,AdvertisementData.bytesForUuid(srvUuid.getType()));
                ByteUtilsHelper.invertArray(dui);
                this.mServiceUUIDs.add(AssignedUuid.uuidFomByteArray(dui));
			}

			// Retrieve service data if present
			Advertisement srvDatas = new Advertisement();
            srvDatas.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_SERVICE_DATA16));
            srvDatas.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_SERVICE_DATA32));
            srvDatas.addAll(AdvertisementHelper.filterAdvertisement(advDatas,AdvertisementData.TYPE_SERVICE_DATA128));
            for (AdvertisementData srvData:srvDatas)
            {   if(srvData.getLength()>AdvertisementData.bytesForUuid(srvData.getType())) {
                    byte[] dui = Arrays.copyOfRange(srvData.getData(), 0,AdvertisementData.bytesForUuid(srvData.getType()));
                    ByteUtilsHelper.invertArray(dui);
                    ParcelUuid uid = AssignedUuid.uuidFomByteArray(dui);
                    this.mServiceData.put(uid, Arrays.copyOfRange(srvData.getData(), AdvertisementData.bytesForUuid(srvData.getType()), srvData.getData().length));
                }
            }
        } catch(Exception e) {
			Log.w("parseBleDataFromArray:",e.toString());//,e.toString());

		}
	}

	@RequiresApi(21)
	private void parseBleDataFromScanRecord(ScanRecord scanRecord){
		try {
			// Retrieve local name if present
			this.mName = scanRecord.getDeviceName();
			// Retrieve power level if present
			this.mTxPowerLevel =  scanRecord.getTxPowerLevel();

			// Retrieve manufacturer data if present
            this.mManufacturerSpecificData =scanRecord.getManufacturerSpecificData();

			// Retrieve service data if present
			this.mServiceData = scanRecord.getServiceData();

			// Retrieve service uuid list if present
			this.mServiceUUIDs = scanRecord.getServiceUuids();

		} catch(Exception e) {
			Log.w("parseBleDataFromRecord:",e.toString());//,e.toString());

		}
	}
}
