package com.polestar.advertisement;

import java.nio.ByteBuffer;

public class ByteUtilsHelper {

    /**
     * The Constant HEXES.
     */
    private static final String HEXDIGITS = "0123456789ABCDEF";

    private ByteUtilsHelper(){
        // TO AVOID INSTANTIATION
    }

    /**
     * Gets a representation of a Byte Array as a HEX String.
     * <p>
     * ex : [01,30,FF,AA]
     *
     * @param array the array
     * @return the string
     */
    public static String byteArrayToHexString(final byte[] array) {
        final StringBuilder sb = new StringBuilder();
        boolean firstEntry = true;
        sb.append('[');

        for (final byte b : array) {
            if (!firstEntry) {
                sb.append(",");
            }
            sb.append(HEXDIGITS.charAt((b & 0xF0) >> 4));
            sb.append(HEXDIGITS.charAt((b & 0x0F)));
            firstEntry = false;
        }

        sb.append(']');
        return sb.toString();
    }
    /**
     * Interprets an Hex string as a Byte Array
     * <p>
     * ex : 0130FFAA
     *
     * @param s the string
     * @return the byte array
     */
    public static byte[] hexStringToByteArray(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
                    + Character.digit(s.charAt(i+1), 16));
        }
        return data;
    }

    /**
     * Checks to see if a byte array starts with another byte array.
     *
     * @param array  the array
     * @param prefix the prefix
     * @return true, if successful
     */
    public static boolean doesArrayBeginWith(final byte[] array, final byte[] prefix) {
        if (array.length < prefix.length) {
            return false;
        }

        for (int i = 0; i < prefix.length; i++) {
            if (array[i] != prefix[i]) {
                return false;
            }
        }

        return true;
    }

    /**
     * Converts a byte array with a length of 2 into an int
     *
     * @param input the input
     * @return the int from the array
     */
    public static int getIntFrom2ByteArray(final byte[] input) {
        final byte[] result = new byte[4];

        result[0] = 0;
        result[1] = 0;
        result[2] = input[0];
        result[3] = input[1];

        return ByteUtilsHelper.getIntFromByteArray(result);
    }

    /**
     * Converts a byte to an int, preserving the sign.
     * <p>
     * For example, FF will be converted to 255 and not -1.
     *
     * @param bite the byte
     * @return the int from byte
     */
    public static int getIntFromByte(final byte bite) {
        return bite & 0xFF;
    }
    /**
     * Converts a byte array to an short.
     *
     * @param bytes the bytes
     * @return the short from byte array
     */
    public static short getShortFromByteArray(final byte[] bytes) {
        return ByteBuffer.wrap(bytes).getShort();
    }
    /**
     * Converts a short to a byte array.
     *
     * @param x the short
     * @return the byte array
     */
    public static byte[] getBytesFromShort(short x) {
        ByteBuffer buffer = ByteBuffer.allocate(Short.SIZE/8);
        buffer.putShort(x);
        return buffer.array();
    }

    /**
     * Converts a byte array to an int.
     *
     * @param bytes the bytes
     * @return the int from byte array
     */
    public static int getIntFromByteArray(final byte[] bytes) {
        return ByteBuffer.wrap(bytes).getInt();
    }
    /**
     * Converts an integer to a byte array.
     *
     * @param x the int
     * @return the byte array
     */
    public static byte[] getBytesFromInt(int x) {
        ByteBuffer buffer = ByteBuffer.allocate(Integer.SIZE/8);
        buffer.putInt(x);
        return buffer.array();
    }
    /**
     * Converts a byte array to a long.
     *
     * @param bytes the bytes
     * @return the long from byte array
     */
    public static long getLongFromByteArray(final byte[] bytes) {
        return ByteBuffer.wrap(bytes).getLong();
    }
    /**
     * Converts a long to a byte array.
     *
     * @param x from byte array
     * @return byte array
     */
    public static byte[] getBytesFromLong(long x) {
        ByteBuffer buffer = ByteBuffer.allocate(Long.SIZE/8);
        buffer.putLong(x);
        return buffer.array();
    }
    /**
     * Inverts an byte array in place.
     *
     * @param array the array
     */
    public static void invertArray(final byte[] array) {
        final int size = array.length;
        byte temp;

        for (int i = 0; i < size / 2; i++) {
            temp = array[i];
            array[i] = array[size - 1 - i];
            array[size - 1 - i] = temp;
        }
    }
}
