package com.aitime.android.security;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

/**
 * Tools for RSA encryption and decryption
 */
public class RSA {

    private static final int KEY_SIZE = 1024;
    private static final String FILE_NAME_PUBLIC = "publicKey.key";
    private static final String FILE_NAME_PRIVATE = "privateKey.key";

    /**
     * 字节数据转字符串专用集合
     */
    private static final char[] HEX_CHAR = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

    /**
     * Randomly generated key pair
     */
    public static void generateKeyPair(String filePath) {
        BufferedWriter publicWriter = null;
        BufferedWriter privateWriter = null;
        try {
            KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance("RSA");
            // 初始化密钥对生成器，密钥大小为96-1024位
            keyPairGen.initialize(KEY_SIZE, new SecureRandom());
            // 生成一个密钥对，保存在keyPair中
            KeyPair keyPair = keyPairGen.generateKeyPair();
            // 得到私钥
            RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
            // 得到公钥
            RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
            // 得到公钥字符串
            String publicKeyString = Base64.encodeToString(publicKey.getEncoded(), Base64.DEFAULT);
            // 得到私钥字符串
            String privateKeyString = Base64.encodeToString(privateKey.getEncoded(), Base64.DEFAULT);
            // 将密钥对写入到文件
            publicWriter = new BufferedWriter(new FileWriter(new File(filePath, FILE_NAME_PUBLIC)));
            privateWriter = new BufferedWriter(new FileWriter(new File(filePath, FILE_NAME_PRIVATE)));
            publicWriter.write(publicKeyString);
            privateWriter.write(privateKeyString);
            publicWriter.flush();
            privateWriter.flush();
        } catch (NoSuchAlgorithmException | IOException e) {
            e.printStackTrace();
        } finally {
            try {
                if (publicWriter != null) {
                    publicWriter.close();
                }
                if (privateWriter != null) {
                    privateWriter.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

    }

    /**
     * Load public key from input stream in file
     *
     * @param path file path
     * @throws Exception
     */
    public static String loadPublicKeyByFile(String path) throws Exception {
        String readLine;
        StringBuilder builder = new StringBuilder();
        BufferedReader reader = new BufferedReader(new FileReader(new File(path, FILE_NAME_PUBLIC)));
        while ((readLine = reader.readLine()) != null) {
            builder.append(readLine);
        }
        reader.close();
        return builder.toString();
    }

    /**
     * Load public key from string
     *
     * @param publicKeyStr Public key data string
     * @throws Exception
     */
    public static RSAPublicKey loadPublicKeyByString(String publicKeyStr) throws Exception {
        byte[] buffer = Base64.decode(publicKeyStr, Base64.DEFAULT);
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(buffer);
        return (RSAPublicKey) keyFactory.generatePublic(keySpec);
    }

    /**
     * Load private key from input stream in file
     *
     * @param path file path
     */
    public static String loadPrivateKeyByFile(String path) throws Exception {
        String readLine;
        StringBuilder builder = new StringBuilder();
        BufferedReader reader = new BufferedReader(new FileReader(new File(path, FILE_NAME_PRIVATE)));
        while ((readLine = reader.readLine()) != null) {
            builder.append(readLine);
        }
        reader.close();
        return builder.toString();
    }

    /**
     * Load private key from string
     *
     * @param privateKeyStr Private key data string
     * @throws Exception
     */
    public static RSAPrivateKey loadPrivateKeyByString(String privateKeyStr)
            throws Exception {
        byte[] buffer = Base64.decode(privateKeyStr, Base64.DEFAULT);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(buffer);
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        return (RSAPrivateKey) keyFactory.generatePrivate(keySpec);
    }

    /**
     * Public key encryption
     *
     * @param publicKey Public key
     * @param content   content to be encrypted
     * @throws Exception
     */
    public static String encrypt(RSAPublicKey publicKey, byte[] content) throws Exception {
        if (publicKey == null) {
            throw new IllegalArgumentException("The publicKey is null");
        }
        try {
            Cipher cipher = Cipher.getInstance("RSA/None/PKCS1Padding");
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            byte[] output = cipher.doFinal(content);
            return Base64.encodeToString(output, Base64.DEFAULT);
        } catch (NoSuchAlgorithmException | NoSuchPaddingException | InvalidKeyException | IllegalBlockSizeException | BadPaddingException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Private key decryption
     *
     * @param privateKey Private key
     * @param content    content to be decrypted
     * @throws Exception
     */
    public static byte[] decrypt(RSAPrivateKey privateKey, byte[] content) throws Exception {
        if (privateKey == null) {
            throw new IllegalArgumentException("The privateKey is null");
        }
        try {
            Cipher cipher = Cipher.getInstance("RSA");
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            return cipher.doFinal(content);
        } catch (NoSuchAlgorithmException | NoSuchPaddingException | InvalidKeyException | IllegalBlockSizeException | BadPaddingException e) {
            e.printStackTrace();
        }
        return null;
    }

}
