package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.util.Log;
import android.widget.Toast;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.razorpay.R;
import com.razorpay.Checkout;

import org.json.JSONObject;

import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

/**
 * Razorpay payment
 */
@Payment(channel = PayChannel.RAZORPAY)
public class Razorpay implements IPayment {

    private static final String TAG = "Razorpay";

    private Activity mActivity;

    public Razorpay(Activity activity) {
        this.mActivity = activity;
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {
        // Preload payment resources
        Checkout.preload(application);
    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code razorpayKey} and {@code razorId})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        String razorId = getParameter(params, "razorId");
        String razorpayKey = getParameter(params, "razorpayKey");
        int iconRes = 0;
        int labelRes = 0;
        try {
            Application context = mActivity.getApplication();
            PackageManager packageManager = context.getPackageManager();
            PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0);
            iconRes = packageInfo.applicationInfo.icon;
            labelRes = packageInfo.applicationInfo.labelRes;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        // Instantiate Checkout
        Checkout checkout = new Checkout();
        // Initialize SDK
        checkout.setKeyID(razorpayKey);
        // Set your logo here
        checkout.setImage(iconRes);
        // Reference to current activity

        // Pass your payment options to the Razorpay Checkout as a JSONObject
        try {
            String themeColor = mActivity.getString(R.string.payment_razorpay_theme_color);
            JSONObject options = new JSONObject();
            // Merchant Name eg: ACME Corp || HasGeek etc.
            options.put("name", mActivity.getString(labelRes));
            options.put("description", info.getPayName());

            // Description can be anything
            options.put("order_id", razorId);

            // Amount is always passed in currency subunits
            options.put("amount", info.getPayMoney());
            options.put("currency", info.getCurrency());

            options.put("prefill.name", info.getName());
            options.put("prefill.email", info.getEmail());
            options.put("prefill.contact", info.getContact());

            options.put("theme.color", themeColor);

            checkout.open(mActivity, options);
        } catch (Exception e) {
            Log.e(TAG, "Error in starting Razorpay Checkout", e);
        }
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        if ("0".equals(status)) {
            int code = getNumber(bundle.getString("code"));
            if (code == Checkout.PAYMENT_CANCELED) {
                showToast(R.string.payment_status_cancelled);
                return false;
            }
            if (code == Checkout.NETWORK_ERROR) {
                showToast(R.string.payment_network_error);
                return false;
            }
            if (code == Checkout.INVALID_OPTIONS) {
                showToast(R.string.payment_invalid_options);
                return false;
            }
        }
        return true;
    }

    @Override
    public void release() {
        Checkout.clearUserData(mActivity.getApplicationContext());
        mActivity = null;
    }

    private int getNumber(String value) {
        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException | NullPointerException e) {
            return -1;
        }
    }

    private void showToast(@StringRes int resId) {
        Toast.makeText(mActivity, resId, Toast.LENGTH_SHORT).show();
    }

}
