package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.widget.Toast;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.core.PayCode;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.paytm.R;
import com.paytm.pgsdk.PaytmOrder;
import com.paytm.pgsdk.PaytmPaymentTransactionCallback;
import com.paytm.pgsdk.TransactionManager;

import java.lang.ref.WeakReference;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

/**
 * Paytm payment
 */
@Payment(channel = PayChannel.PAYTM)
public class Paytm implements IPayment {

    private final WeakReference<Activity> mActivity;
    private IPaymentCallback mCallback;

    public Paytm(Activity activity) {
        this.mActivity = new WeakReference<>(activity);
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {
        this.mCallback = callback;
    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code orderId}、{@code mid}、{@code txnToken}、{@code callbackUrl} and {@code showPaymentUrl})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return;
        }
        String orderId = getParameter(params, "orderId");
        String mid = getParameter(params, "mid");
        String txnToken = getParameter(params, "txnToken");
        String callbackUrl = getParameter(params, "callbackUrl");
        String showPaymentUrl = params.get("showPaymentUrl");
        String amount = info.getPayMoney();
        PaytmPaymentTransactionCallback transactionCallback = new PaytmPaymentTransactionCallback() {

            @Override
            public void onTransactionResponse(Bundle bundle) {
                mCallback.onPaymentResult("1", bundle);
            }

            @Override
            public void networkNotAvailable() {
                mCallback.onPaymentResult("-1", new Bundle());
            }

            @Override
            public void onErrorProceed(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-2", bundle);
            }

            @Override
            public void clientAuthenticationFailed(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-3", bundle);
            }

            @Override
            public void someUIErrorOccurred(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-4", bundle);
            }

            @Override
            public void onErrorLoadingWebPage(int i, String s, String s1) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s1);
                mCallback.onPaymentResult("-5", bundle);
            }

            @Override
            public void onBackPressedCancelTransaction() {
                mCallback.onPaymentResult("0", new Bundle());
            }

            @Override
            public void onTransactionCancel(String s, Bundle bundle) {
                mCallback.onPaymentResult("0", bundle);
            }
        };
        PaytmOrder paytmOrder = new PaytmOrder(orderId, mid, txnToken, amount, callbackUrl);
        TransactionManager transactionManager = new TransactionManager(paytmOrder, transactionCallback);
        transactionManager.setAppInvokeEnabled(false);
        if (showPaymentUrl != null && showPaymentUrl.length() > 0) {
            transactionManager.setShowPaymentUrl(showPaymentUrl);
        }
        transactionManager.startTransaction(activity, PayCode.PAYTM);
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return false;
        }
        if ("1".equals(status)) {
            return true;
        }
        if ("0".equals(status)) {
            showToast(activity, R.string.payment_status_cancelled);
            return false;
        } else if ("-1".equals(status)) {
            showToast(activity, R.string.payment_network_error);
            return false;
        } else {
            String message = bundle.getString("message");
            if (message == null || message.length() == 0) {
                showToast(activity, R.string.payment_status_failed);
            } else {
                showToast(activity, activity.getString(R.string.payment_status_failed) + "[" + message + "]");
            }
            return false;
        }
    }

    @Override
    public void release() {
        mActivity.clear();
    }

    private void showToast(Activity activity, @StringRes int resId) {
        Toast.makeText(activity, resId, Toast.LENGTH_SHORT).show();
    }

    private void showToast(Activity activity, String text) {
        Toast.makeText(activity, text, Toast.LENGTH_SHORT).show();
    }

}