package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.widget.Toast;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.core.PayCode;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.paytm.R;
import com.paytm.pgsdk.PaytmOrder;
import com.paytm.pgsdk.PaytmPaymentTransactionCallback;
import com.paytm.pgsdk.TransactionManager;

import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

/**
 * Paytm payment
 */
@Payment(channel = PayChannel.PAYTM)
public class Paytm implements IPayment {

    private Activity mActivity;
    private IPaymentCallback mCallback;

    public Paytm(Activity activity) {
        this.mActivity = activity;
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {
        this.mCallback = callback;
    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code orderId}、{@code mid}、{@code txnToken}、{@code callbackUrl} and {@code showPaymentUrl})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        String orderId = getParameter(params, "orderId");
        String mid = getParameter(params, "mid");
        String txnToken = getParameter(params, "txnToken");
        String callbackUrl = getParameter(params, "callbackUrl");
        String showPaymentUrl = params.get("showPaymentUrl");
        String amount = info.getPayMoney();
        PaytmPaymentTransactionCallback transactionCallback = new PaytmPaymentTransactionCallback() {

            @Override
            public void onTransactionResponse(Bundle bundle) {
                mCallback.onPaymentResult("1", bundle);
            }

            @Override
            public void networkNotAvailable() {
                mCallback.onPaymentResult("-1", new Bundle());
            }

            @Override
            public void onErrorProceed(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-2", bundle);
            }

            @Override
            public void clientAuthenticationFailed(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-3", bundle);
            }

            @Override
            public void someUIErrorOccurred(String s) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s);
                mCallback.onPaymentResult("-4", bundle);
            }

            @Override
            public void onErrorLoadingWebPage(int i, String s, String s1) {
                Bundle bundle = new Bundle();
                bundle.putString("message", s1);
                mCallback.onPaymentResult("-5", bundle);
            }

            @Override
            public void onBackPressedCancelTransaction() {
                mCallback.onPaymentResult("0", new Bundle());
            }

            @Override
            public void onTransactionCancel(String s, Bundle bundle) {
                mCallback.onPaymentResult("0", bundle);
            }
        };
        PaytmOrder paytmOrder = new PaytmOrder(orderId, mid, txnToken, amount, callbackUrl);
        TransactionManager transactionManager = new TransactionManager(paytmOrder, transactionCallback);
        if (showPaymentUrl != null && showPaymentUrl.length() > 0) {
            transactionManager.setShowPaymentUrl(showPaymentUrl);
        }
        transactionManager.startTransaction(mActivity, PayCode.PAYTM);
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        if ("1".equals(status)) {
            return true;
        }
        if ("0".equals(status)) {
            showToast(R.string.payment_status_cancelled);
            return false;
        } else if ("-1".equals(status)) {
            showToast(R.string.payment_network_error);
            return false;
        } else {
            String message = bundle.getString("message");
            if (message == null || message.length() == 0) {
                showToast(R.string.payment_status_failed);
            } else {
                showToast(mActivity.getString(R.string.payment_status_failed) + "[" + message + "]");
            }
            return false;
        }
    }

    @Override
    public void release() {

    }

    private void showToast(@StringRes int resId) {
        Toast.makeText(mActivity, resId, Toast.LENGTH_SHORT).show();
    }

    private void showToast(String text) {
        Toast.makeText(mActivity, text, Toast.LENGTH_SHORT).show();
    }

}