package com.aitime.android.payment;

import android.app.Activity;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.util.ArrayMap;

import com.aitime.android.payment.core.IPayment;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * An utility class that provides {@link IPayment} for a scope.
 */
public final class PaymentFactory {

    private static final String PACKAGE_PAYMENT = "com.aitime.android.payment";

    private Map<String, IPayment> payments;
    private Map<String, Class<? extends IPayment>> channels;
    private volatile static boolean hasRegister = false;

    private static class SingletonHolder {
        private static final PaymentFactory INSTANCE = new PaymentFactory();
    }

    /**
     * Get instance of PaymentFactory
     */
    public static PaymentFactory getInstance() {
        return SingletonHolder.INSTANCE;
    }

    private PaymentFactory() {
        payments = new ArrayMap<>();
        channels = new ArrayMap<>();
    }

    /**
     * Register all payment channel
     */
    public void register(@NonNull Context context) {
        if (hasRegister) {
            return;
        }
        try {
            Context application = context.getApplicationContext();
            PackageManager packageManager = application.getPackageManager();
            ApplicationInfo appInfo = packageManager.getApplicationInfo(application.getPackageName(), PackageManager.GET_META_DATA);
            Bundle metaData = appInfo.metaData;
            if (metaData == null || metaData.isEmpty()) {
                return;
            }
            Set<String> keySet = metaData.keySet();
            for (String key : keySet) {
                if (key == null || !key.startsWith(PACKAGE_PAYMENT)) {
                    continue;
                }
                String className = metaData.getString(key);
                Class<? extends IPayment> paymentClass = getPaymentClass(className);
                if (paymentClass == null) {
                    continue;
                }
                Payment annotation = paymentClass.getAnnotation(Payment.class);
                channels.put(annotation == null ? null : annotation.channel(), paymentClass);
            }
            hasRegister = true;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Create payment channel by payChannel
     */
    @Nullable
    public IPayment create(@NonNull Activity activity, @NonNull String payChannel) {
        if (!hasRegister) {
            register(activity);
        }
        IPayment payment = payments.get(payChannel);
        if (payment != null) {
            return payment;
        }
        Class<? extends IPayment> paymentClass = channels.get(payChannel);
        if (paymentClass != null) {
            payment = generatePayment(activity, paymentClass);
            payments.put(payChannel, payment);
        }
        return payment;
    }

    /**
     * Release resource
     */
    public void clear() {
        if (payments != null && !payments.isEmpty()) {
            for (IPayment payment : payments.values()) {
                payment.release();
            }
            payments.clear();
        }
    }

    @Nullable
    @SuppressWarnings("unchecked")
    private Class<? extends IPayment> getPaymentClass(String className) {
        try {
            Class<?> paymentClass = Class.forName(className);
            if (IPayment.class.isAssignableFrom(paymentClass)) {
                return (Class<? extends IPayment>) paymentClass;
            }
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    private IPayment generatePayment(@NonNull Activity activity, Class<? extends IPayment> clazz) {
        IPayment payment = null;
        try {
            payment = clazz.newInstance();
        } catch (IllegalAccessException | InstantiationException e) {
            try {
                Constructor<? extends IPayment> constructor = clazz.getDeclaredConstructor(Activity.class);
                constructor.setAccessible(true);
                payment = constructor.newInstance(activity);
            } catch (NoSuchMethodException | IllegalAccessException | InstantiationException | InvocationTargetException ex) {
                ex.printStackTrace();
            }
        }
        return payment;
    }

}
