package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.widget.Toast;

import com.adyen.checkout.afterpay.AfterPayConfiguration;
import com.adyen.checkout.base.model.PaymentMethodsApiResponse;
import com.adyen.checkout.base.model.payments.Amount;
import com.adyen.checkout.base.util.CheckoutCurrency;
import com.adyen.checkout.bcmc.BcmcConfiguration;
import com.adyen.checkout.card.CardConfiguration;
import com.adyen.checkout.core.api.Environment;
import com.adyen.checkout.core.exception.CheckoutException;
import com.adyen.checkout.core.log.Logger;
import com.adyen.checkout.dropin.DropIn;
import com.adyen.checkout.dropin.DropInConfiguration;
import com.adyen.checkout.googlepay.GooglePayConfiguration;
import com.aitime.android.payment.adyen.AdyenStorage;
import com.aitime.android.payment.adyen.R;
import com.aitime.android.payment.adyen.SimpleDropInService;
import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.Locale;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Adyen payment
 */
@Payment(channel = PayChannel.ADYEN)
public class Adyen implements IPayment {

    private static final String TAG = "Adyen";

    private static Class<? extends SimpleDropInService> dropInServiceClass;

    private final WeakReference<Activity> mActivity;

    public Adyen(Activity activity) {
        this.mActivity = new WeakReference<>(activity);
    }

    public static void setDropInServiceClass(Class<? extends SimpleDropInService> dropInServiceClass) {
        Adyen.dropInServiceClass = dropInServiceClass;
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code environmentName}、{@code publicKey}、{@code countryCode}、{@code merchantAccount}、{@code paymentMethods} and {@code shopperLocale})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return;
        }
        if (dropInServiceClass == null) {
            throw new IllegalArgumentException("The dropInServiceClass is required.");
        }
        String environmentName = getParameter(params, "environmentName");
        String publicKey = getParameter(params, "publicKey");
        String countryCode = getParameter(params, "countryCode");
        String merchantAccount = getParameter(params, "merchantAccount");
        String paymentMethods = getParameter(params, "paymentMethods");
        String shopperLocale = getParameter(params, "shopperLocale");
        Environment environment;
        if ("TEST".equalsIgnoreCase(environmentName)) {
            environment = Environment.TEST;
        } else if ("EUROPE".equalsIgnoreCase(environmentName)) {
            environment = Environment.EUROPE;
        } else if ("UNITED_STATES".equalsIgnoreCase(environmentName)) {
            environment = Environment.UNITED_STATES;
        } else if ("AUSTRALIA".equalsIgnoreCase(environmentName)) {
            environment = Environment.AUSTRALIA;
        } else {
            throw new IllegalArgumentException("The environment is not supported.");
        }

        AdyenStorage.setReference(info.getOrderId());
        AdyenStorage.setCountryCode(countryCode);
        AdyenStorage.setMerchantAccount(merchantAccount);
        AdyenStorage.setShopperReference(info.getPayName());

        String[] split = shopperLocale.split("_");
        Locale locale = new Locale(split[0], split[1]);

        GooglePayConfiguration googlePayConfig = new GooglePayConfiguration.Builder(
                activity, merchantAccount)
                .setCountryCode(countryCode)
                .build();

        CardConfiguration cardConfiguration = new CardConfiguration.Builder(activity, publicKey)
                .setShopperReference(info.getPayName())
                .setShopperLocale(locale)
                .build();

        AfterPayConfiguration afterPayConfiguration = new AfterPayConfiguration.Builder(
                activity, AfterPayConfiguration.CountryCode.NL)
                .setShopperLocale(locale)
                .build();

        BcmcConfiguration bcmcConfiguration = new BcmcConfiguration.Builder(activity, publicKey)
                .setShopperLocale(locale)
                .build();

        Intent resultIntent = new Intent(activity, activity.getClass());
        resultIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);

        DropInConfiguration.Builder configurationBuilder = new DropInConfiguration.Builder(
                activity, resultIntent, dropInServiceClass)
                .setEnvironment(environment)
                .setShopperLocale(locale)
                .addCardConfiguration(cardConfiguration)
                .addAfterPayConfiguration(afterPayConfiguration)
                .addBcmcConfiguration(bcmcConfiguration)
                .addGooglePayConfiguration(googlePayConfig);

        try {
            String currency = info.getCurrency();
            int value = Integer.parseInt(info.getPayMoney());
            final CheckoutCurrency checkoutCurrency = CheckoutCurrency.find(currency);
            Amount amount = new Amount();
            amount.setValue(value * (int) Math.pow(10, checkoutCurrency.getFractionDigits()));
            amount.setCurrency(currency);
            AdyenStorage.setAmount(amount);
            configurationBuilder.setAmount(amount);
        } catch (CheckoutException e) {
            Logger.e(TAG, "Amount $amount not valid", e);
        }
        try {
            PaymentMethodsApiResponse paymentMethodsApiResponse = PaymentMethodsApiResponse.SERIALIZER
                    .deserialize(new JSONObject(paymentMethods));
            DropIn.startPayment(activity, paymentMethodsApiResponse, configurationBuilder.build());
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return false;
        }
        if ("Refused".equalsIgnoreCase(status)) {
            showToast(activity, activity.getString(R.string.payment_status_failed));
            return false;
        }
        if ("Cancelled".equalsIgnoreCase(status)) {
            showToast(activity, activity.getString(R.string.payment_status_cancelled));
            return false;
        }
        if ("Error".equalsIgnoreCase(status)) {
            String refusalReason = bundle.getString("refusalReason");
            if (TextUtils.isEmpty(refusalReason)) {
                refusalReason = activity.getString(R.string.payment_status_failed);
            }
            showToast(activity, refusalReason);
            return false;
        }
        return "Authorised".equalsIgnoreCase(status) || "Received".equalsIgnoreCase(status);
    }

    @Override
    public void release() {
        mActivity.clear();
        dropInServiceClass = null;
    }

    private void showToast(Activity activity, String message) {
        Toast.makeText(activity, message, Toast.LENGTH_SHORT).show();
    }

}
