package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.widget.Toast;

import com.adyen.checkout.afterpay.AfterPayConfiguration;
import com.adyen.checkout.base.model.PaymentMethodsApiResponse;
import com.adyen.checkout.base.model.payments.Amount;
import com.adyen.checkout.bcmc.BcmcConfiguration;
import com.adyen.checkout.card.CardConfiguration;
import com.adyen.checkout.core.api.Environment;
import com.adyen.checkout.core.exception.CheckoutException;
import com.adyen.checkout.core.log.Logger;
import com.adyen.checkout.dropin.DropIn;
import com.adyen.checkout.dropin.DropInConfiguration;
import com.adyen.checkout.dropin.service.DropInService;
import com.adyen.checkout.googlepay.GooglePayConfiguration;
import com.aitime.android.payment.adyen.AdyenStorage;
import com.aitime.android.payment.adyen.R;
import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Locale;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Adyen payment
 */
@Payment(channel = PayChannel.ADYEN)
public class Adyen implements IPayment {

    private static final String TAG = "Adyen";

    private static Class<? extends DropInService> dropInServiceClass;

    private Activity mActivity;

    public Adyen(Activity activity) {
        this.mActivity = activity;
    }

    public static void setDropInServiceClass(Class<? extends DropInService> dropInServiceClass) {
        Adyen.dropInServiceClass = dropInServiceClass;
    }

    @Override
    public void callback(IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {

    }

    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        if (dropInServiceClass == null) {
            throw new IllegalArgumentException("The dropInServiceClass is required.");
        }
        String environmentName = params.get("environmentName");
        if (environmentName == null || environmentName.length() == 0) {
            throw new IllegalArgumentException("The environmentName is required.");
        }
        String publicKey = params.get("publicKey");
        if (publicKey == null || publicKey.length() == 0) {
            throw new IllegalArgumentException("The publicKey is required.");
        }
        String countryCode = params.get("countryCode");
        if (countryCode == null || countryCode.length() == 0) {
            throw new IllegalArgumentException("The countryCode is required.");
        }
        String merchantAccount = params.get("merchantAccount");
        if (merchantAccount == null || merchantAccount.length() == 0) {
            throw new IllegalArgumentException("The merchantAccount is required.");
        }
        String paymentMethods = params.get("paymentMethods");
        if (paymentMethods == null || paymentMethods.length() == 0) {
            throw new IllegalArgumentException("The paymentMethods is required.");
        }
        String shopperLocale = params.get("shopperLocale");
        if (shopperLocale == null || shopperLocale.length() == 0) {
            throw new IllegalArgumentException("The shopperLocale is required.");
        }
        Environment environment;
        if ("TEST".equalsIgnoreCase(environmentName)) {
            environment = Environment.TEST;
        } else if ("EUROPE".equalsIgnoreCase(environmentName)) {
            environment = Environment.EUROPE;
        } else if ("UNITED_STATES".equalsIgnoreCase(environmentName)) {
            environment = Environment.UNITED_STATES;
        } else if ("AUSTRALIA".equalsIgnoreCase(environmentName)) {
            environment = Environment.AUSTRALIA;
        } else {
            throw new IllegalArgumentException("The environment is not supported.");
        }

        AdyenStorage.setReference(info.getOrderId());
        AdyenStorage.setCountryCode(countryCode);
        AdyenStorage.setMerchantAccount(merchantAccount);
        AdyenStorage.setShopperReference(info.getPayName());

        String[] split = shopperLocale.split("_");
        Locale locale = new Locale(split[0], split[1]);

        GooglePayConfiguration googlePayConfig = new GooglePayConfiguration.Builder(
                mActivity, merchantAccount)
                .setCountryCode(countryCode)
                .build();

        CardConfiguration cardConfiguration = new CardConfiguration.Builder(mActivity, publicKey)
                .setShopperReference(info.getPayName())
                .setShopperLocale(locale)
                .build();

        AfterPayConfiguration afterPayConfiguration = new AfterPayConfiguration.Builder(
                mActivity, AfterPayConfiguration.CountryCode.NL)
                .setShopperLocale(locale)
                .build();

        BcmcConfiguration bcmcConfiguration = new BcmcConfiguration.Builder(mActivity, publicKey)
                .setShopperLocale(locale)
                .build();

        Intent resultIntent = new Intent(mActivity, mActivity.getClass());
        resultIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);

        DropInConfiguration.Builder configurationBuilder = new DropInConfiguration.Builder(
                mActivity, resultIntent, dropInServiceClass)
                .setEnvironment(environment)
                .setShopperLocale(locale)
                .addCardConfiguration(cardConfiguration)
                .addAfterPayConfiguration(afterPayConfiguration)
                .addBcmcConfiguration(bcmcConfiguration)
                .addGooglePayConfiguration(googlePayConfig);

        try {
            Amount amount = new Amount();
            amount.setValue(Integer.parseInt(info.getPayMoney()));
            amount.setCurrency(info.getCurrency());
            AdyenStorage.setAmount(amount);
            configurationBuilder.setAmount(amount);
        } catch (CheckoutException e) {
            Logger.e(TAG, "Amount $amount not valid", e);
        }
        try {
            PaymentMethodsApiResponse paymentMethodsApiResponse = PaymentMethodsApiResponse.SERIALIZER
                    .deserialize(new JSONObject(paymentMethods));
            DropIn.startPayment(mActivity, paymentMethodsApiResponse, configurationBuilder.build());
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        if ("Refused".equalsIgnoreCase(status)) {
            showToast(mActivity.getString(R.string.payment_status_failed));
            return false;
        }
        if ("Cancelled".equalsIgnoreCase(status)) {
            showToast(mActivity.getString(R.string.payment_status_cancelled));
            return false;
        }
        if ("Error".equalsIgnoreCase(status)) {
            String refusalReason = bundle.getString("refusalReason");
            if (TextUtils.isEmpty(refusalReason)) {
                refusalReason = mActivity.getString(R.string.payment_status_failed);
            }
            showToast(refusalReason);
            return false;
        }
        return true;
    }

    @Override
    public void release() {
        dropInServiceClass = null;
    }

    private void showToast(String message) {
        Toast.makeText(mActivity, message, Toast.LENGTH_SHORT).show();
    }

}
