/*
   Copyright (c) 2014,2015,2016 Ahome' Innovation Technologies. All rights reserved.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

package com.ait.lienzo.client.core.image;

import com.google.gwt.dom.client.ImageElement;
import com.google.gwt.event.dom.client.ErrorEvent;
import com.google.gwt.event.dom.client.ErrorHandler;
import com.google.gwt.event.dom.client.LoadEvent;
import com.google.gwt.event.dom.client.LoadHandler;
import com.google.gwt.resources.client.ImageResource;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.RootPanel;

public abstract class ImageLoader
{
    public ImageLoader(final String url)
    {
        final Image image = new Image();

        image.setVisible(false);

        if (isValidDataURL(url))
        {
            RootPanel.get().add(image);

            ImageElement.as(image.getElement()).setSrc(url);

            onImageElementLoad(ImageElement.as(image.getElement()));
        }
        else
        {
            /*
            if (url.startsWith("http:") || (url.startsWith("https:")))
            {
                setCrossOrigin(ImageElement.as(m_image.getElement()), "anonymous");
            }
            */
            image.addLoadHandler(new LoadHandler()
            {
                @Override
                public final void onLoad(final LoadEvent event)
                {
                    doImageElementLoadAndRetry(ImageElement.as(image.getElement()), image, url);
                }
            });
            image.addErrorHandler(new ErrorHandler()
            {
                @Override
                public final void onError(final ErrorEvent event)
                {
                    RootPanel.get().remove(image);

                    onImageElementError("Image " + url + " failed to load");
                }
            });
            RootPanel.get().add(image);

            image.setUrl(url);
        }
    }

    private final void doImageElementLoadAndRetry(final ImageElement elem, final Image image, final String url)
    {
        final int w = Math.max(image.getWidth(), elem.getWidth());

        final int h = Math.max(image.getHeight(), elem.getHeight());

        if ((w < 1) || (h < 1))
        {
            load(url, "", new JSImageCallback()
            {
                @Override
                public void onSuccess(final ImageElement e)
                {
                    onImageElementLoad(e);
                }

                @Override
                public void onFailure()
                {
                    RootPanel.get().remove(image);

                    onImageElementError("Image " + url + " failed to load");
                }
            });
        }
        else
        {
            elem.setWidth(w);

            elem.setHeight(h);

            onImageElementLoad(elem);
        }
    }

    public boolean isValidDataURL(final String url)
    {
        if ((url.startsWith("data:")) && (url.length() > 6) && (false == ("data:,".equals(url))))
        {
            return true;
        }
        return false;
    }

    public ImageLoader(final ImageResource resource)
    {
        final Image image = new Image();

        image.setVisible(false);

        image.addLoadHandler(new LoadHandler()
        {
            @Override
            public final void onLoad(final LoadEvent event)
            {
                onImageElementLoad(ImageElement.as(image.getElement()));
            }
        });
        image.addErrorHandler(new ErrorHandler()
        {
            @Override
            public final void onError(final ErrorEvent event)
            {
                RootPanel.get().remove(image);

                onImageElementError("Resource " + resource.getName() + " failed to load");
            }
        });
        image.setResource(resource);

        RootPanel.get().add(image);
    }

    private final native void load(String url, String orig, JSImageCallback self)
    /*-{
		var image = new $wnd.Image();
		image.onload = function() {
			if ('naturalHeight' in image) {
				if (image.naturalHeight + image.naturalWidth == 0) {
					self.@com.ait.lienzo.client.core.image.ImageLoader.JSImageCallback::onFailure()();
					return;
				}
			} else if (image.width + image.height == 0) {
				self.@com.ait.lienzo.client.core.image.ImageLoader.JSImageCallback::onFailure()();
				return;
			}
			self.@com.ait.lienzo.client.core.image.ImageLoader.JSImageCallback::onSuccess(Lcom/google/gwt/dom/client/ImageElement;)(image);
		};
		image.onerror = function() {
			self.@com.ait.lienzo.client.core.image.ImageLoader.JSImageCallback::onFailure()();
		};
		image.crossOrigin = orig;
		image.src = url;
    }-*/;

    private final native void setCrossOrigin(ImageElement element, String value)
    /*-{
		element.crossOrigin = value;
    }-*/;

    public abstract void onImageElementLoad(ImageElement elem);

    public abstract void onImageElementError(String message);

    private interface JSImageCallback
    {
        public void onSuccess(ImageElement e);

        public void onFailure();
    }
}
