/*
   Copyright (c) 2014,2015,2016 Ahome' Innovation Technologies. All rights reserved.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */
// TODO - review DSJ

package com.ait.lienzo.client.core.shape;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.ait.lienzo.client.core.Context2D;
import com.ait.lienzo.client.core.event.AttributesChangedEvent;
import com.ait.lienzo.client.core.event.AttributesChangedHandler;
import com.ait.lienzo.client.core.event.NodeDragEndEvent;
import com.ait.lienzo.client.core.event.NodeDragEndHandler;
import com.ait.lienzo.client.core.event.NodeDragMoveEvent;
import com.ait.lienzo.client.core.event.NodeDragMoveHandler;
import com.ait.lienzo.client.core.event.NodeDragStartEvent;
import com.ait.lienzo.client.core.event.NodeDragStartHandler;
import com.ait.lienzo.client.core.shape.json.validators.ValidationContext;
import com.ait.lienzo.client.core.shape.json.validators.ValidationException;
import com.ait.lienzo.client.core.shape.wires.AbstractControlHandle;
import com.ait.lienzo.client.core.shape.wires.ControlHandleList;
import com.ait.lienzo.client.core.shape.wires.IControlHandle;
import com.ait.lienzo.client.core.shape.wires.IControlHandle.ControlHandleType;
import com.ait.lienzo.client.core.shape.wires.IControlHandleFactory;
import com.ait.lienzo.client.core.shape.wires.IControlHandleList;
import com.ait.lienzo.client.core.types.BoundingBox;
import com.ait.lienzo.client.core.types.PathPartEntryJSO;
import com.ait.lienzo.client.core.types.PathPartList;
import com.ait.lienzo.client.core.types.Point2D;
import com.ait.lienzo.client.core.types.Point2DArray;
import com.ait.lienzo.shared.core.types.ColorName;
import com.ait.lienzo.shared.core.types.DragMode;
import com.ait.lienzo.shared.core.types.ShapeType;
import com.ait.tooling.nativetools.client.collection.NFastArrayList;
import com.ait.tooling.nativetools.client.collection.NFastDoubleArrayJSO;
import com.ait.tooling.nativetools.client.event.HandlerRegistrationManager;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.json.client.JSONObject;

public abstract class AbstractMultiPathPartShape<T extends AbstractMultiPathPartShape<T>> extends Shape<T>
{
    private final NFastArrayList<PathPartList> m_list = new NFastArrayList<PathPartList>();

    protected AbstractMultiPathPartShape(final ShapeType type)
    {
        super(type);
    }

    protected AbstractMultiPathPartShape(final ShapeType type, final JSONObject node, final ValidationContext ctx) throws ValidationException
    {
        super(type, node, ctx);
    }

    @Override
    public BoundingBox getBoundingBox()
    {
        final int size = m_list.size();

        if (size < 1)
        {
            return new BoundingBox(0, 0, 0, 0);
        }
        final BoundingBox bbox = new BoundingBox();

        for (int i = 0; i < size; i++)
        {
            bbox.add(m_list.get(i).getBoundingBox());
        }
        return bbox;
    }

    @Override
    public T refresh()
    {
        return clear();
    }

    public T clear()
    {
        final int size = m_list.size();

        for (int i = 0; i < size; i++)
        {
            m_list.get(i).clear();
        }
        m_list.clear();

        return cast();
    }

    protected final void add(PathPartList list)
    {
        m_list.add(list);
    }

    public final NFastArrayList<PathPartList> getPathPartListArray()
    {
        return m_list;
    }

    @Override
    protected void drawWithoutTransforms(final Context2D context, double alpha, BoundingBox bounds)
    {
        final Attributes attr = getAttributes();

        if ((context.isSelection()) && (false == attr.isListening()))
        {
            return;
        }
        alpha = alpha * attr.getAlpha();

        if (alpha <= 0)
        {
            return;
        }
        if (prepare(context, attr, alpha))
        {
            final int size = m_list.size();

            if (size < 1)
            {
                return;
            }
            for (int i = 0; i < size; i++)
            {
                setAppliedShadow(false);

                setWasFilledFlag(false);

                final PathPartList list = m_list.get(i);

                if (list.size() > 1)
                {
                    if (context.path(list))
                    {
                        fill(context, attr, alpha);
                    }
                    stroke(context, attr, alpha);
                }
            }
        }
    }

    public IControlHandleFactory getControlHandleFactory()
    {
        IControlHandleFactory factory = super.getControlHandleFactory();

        if (null != factory)
        {
            return factory;
        }
        return new DefaultMultiPathShapeHandleFactory(m_list, this);
    }

    public static class OnDragMoveIControlHandleList implements AttributesChangedHandler, NodeDragStartHandler, NodeDragMoveHandler, NodeDragEndHandler
    {
        private Shape<?>            m_shape;

        private IControlHandleList  m_chlist;

        private double[]            m_startPoints;

        private HandlerRegistration m_nodeDragStartHandlerReg;

        private HandlerRegistration m_nodeDragMoveHandlerReg;

        public OnDragMoveIControlHandleList(Shape<?> shape, IControlHandleList chlist)
        {
            m_shape = shape;

            m_chlist = chlist;

            HandlerRegistrationManager regManager = m_chlist.getHandlerRegistrationManager();

            m_nodeDragStartHandlerReg = m_shape.addNodeDragStartHandler(this);

            m_nodeDragMoveHandlerReg = m_shape.addNodeDragMoveHandler(this);

            regManager.register(m_nodeDragStartHandlerReg);

            regManager.register(m_nodeDragMoveHandlerReg);
        }

        @Override
        public void onAttributesChanged(AttributesChangedEvent event)
        {
            //event.
        }

        @Override
        public void onNodeDragStart(NodeDragStartEvent event)
        {
            int size = m_chlist.size();

            m_startPoints = new double[size * 2];

            int i = 0;

            for (IControlHandle handle : m_chlist)
            {
                m_startPoints[i] = handle.getControl().getX();

                m_startPoints[i + 1] = handle.getControl().getY();

                i = i + 2;
            }
        }

        @Override
        public void onNodeDragMove(NodeDragMoveEvent event)
        {
            int i = 0;

            for (IControlHandle handle : m_chlist)
            {
                IPrimitive<?> prim = handle.getControl();

                prim.setX(m_startPoints[i] + event.getDragContext().getDx());

                prim.setY(m_startPoints[i + 1] + event.getDragContext().getDy());

                i = i + 2;
            }
            m_shape.getLayer().draw();
        }

        @Override
        public void onNodeDragEnd(NodeDragEndEvent event)
        {
            m_startPoints = null;
        }
    }

    public static final class DefaultMultiPathShapeHandleFactory implements IControlHandleFactory
    {
        private final NFastArrayList<PathPartList> m_listOfPaths;

        private final Shape<?>                     m_shape;

        private DragMode                           m_dmode = DragMode.SAME_LAYER;

        public DefaultMultiPathShapeHandleFactory(NFastArrayList<PathPartList> listOfPaths, Shape<?> shape)
        {
            m_listOfPaths = listOfPaths;

            m_shape = shape;
        }

        @Override
        public Map<ControlHandleType, IControlHandleList> getControlHandles(ControlHandleType... types)
        {
            return getControlHandles(Arrays.asList(types));
        }

        @Override
        public Map<ControlHandleType, IControlHandleList> getControlHandles(List<ControlHandleType> types)
        {
            if ((null == types) || (types.isEmpty()))
            {
                return null;
            }

            HashMap<ControlHandleType, IControlHandleList> map = new HashMap<ControlHandleType, IControlHandleList>();

            for (ControlHandleType type : types)
            {
                if (type == IControlHandle.ControlHandleStandardType.RESIZE)
                {
                    IControlHandleList chList = getResizeHandles(m_shape, m_listOfPaths, m_dmode);

                    map.put(IControlHandle.ControlHandleStandardType.RESIZE, chList);
                }
                else if (type == IControlHandle.ControlHandleStandardType.POINT)
                {
                    IControlHandleList chList = getPointHandles();

                    map.put(IControlHandle.ControlHandleStandardType.POINT, chList);
                }
            }
            return map;
        }

        public IControlHandleList getPointHandles()
        {
            ControlHandleList chlist = new ControlHandleList();

            NFastArrayList<Point2DArray> allPoints = new NFastArrayList<Point2DArray>();

            int pathIndex = 0;

            for (PathPartList path : m_listOfPaths)
            {
                Point2DArray points = path.getPoints();

                allPoints.add(points);

                int entryIndex = 0;

                for (Point2D point : points)
                {
                    Circle prim = getControlPrimitive(point.getX(), point.getY(), m_shape, m_dmode);

                    PointControlHandle pointHandle = new PointControlHandle(prim, pathIndex, entryIndex++, m_shape, m_listOfPaths, path, chlist);

                    chlist.add(pointHandle);
                }
                pathIndex++;
            }
            new OnDragMoveIControlHandleList(m_shape, chlist);

            return chlist;
        }

        public static IControlHandleList getResizeHandles(Shape<?> shape, NFastArrayList<PathPartList> listOfPaths, DragMode dragMode)
        {
            // FIXME This isn't quite right yet, do not release  (mdp, um what did I mean here?)

            ControlHandleList chlist = new ControlHandleList();

            BoundingBox box = shape.getBoundingBox();

            final Point2D tl = new Point2D(box.getX(), box.getY());

            final Point2D tr = new Point2D(box.getX() + box.getWidth(), box.getY());

            final Point2D bl = new Point2D(box.getX(), box.getHeight() + box.getY());

            final Point2D br = new Point2D(box.getX() + box.getWidth(), box.getHeight() + box.getY());

            ArrayList<ResizeControlHandle> orderedChList = new ArrayList<ResizeControlHandle>();

            Circle prim = getControlPrimitive(tl.getX(), tl.getY(), shape, dragMode);

            ResizeControlHandle topLeft = new ResizeControlHandle(prim, chlist, orderedChList, shape, listOfPaths, 0);

            chlist.add(topLeft);

            orderedChList.add(topLeft);

            prim = getControlPrimitive(tr.getX(), tr.getY(), shape, dragMode);

            ResizeControlHandle topRight = new ResizeControlHandle(prim, chlist, orderedChList, shape, listOfPaths, 1);

            chlist.add(topRight);

            orderedChList.add(topRight);

            prim = getControlPrimitive(br.getX(), br.getY(), shape, dragMode);

            ResizeControlHandle bottomRight = new ResizeControlHandle(prim, chlist, orderedChList, shape, listOfPaths, 2);

            chlist.add(bottomRight);

            orderedChList.add(bottomRight);

            prim = getControlPrimitive(bl.getX(), bl.getY(), shape, dragMode);

            ResizeControlHandle bottomLeft = new ResizeControlHandle(prim, chlist, orderedChList, shape, listOfPaths, 3);

            chlist.add(bottomLeft);

            orderedChList.add(bottomLeft);

            new OnDragMoveIControlHandleList(shape, chlist);

            return chlist;
        }

        private static Circle getControlPrimitive(double x, double y, Shape<?> shape, DragMode dragMode)
        {
            return new Circle(9).setFillColor(ColorName.RED).setFillAlpha(0.4).setX(x + shape.getX()).setY(y + shape.getY()).setDraggable(true).setDragMode(dragMode).setStrokeColor(ColorName.BLACK).setStrokeWidth(2);
        }
    }

    private static class PointControlHandle extends AbstractControlHandle
    {
        private static final long                  serialVersionUID = -5223995638527030291L;

        private final Shape<?>                     m_shape;

        private final NFastArrayList<PathPartList> m_listOfPaths;

        private final IControlHandleList           m_chlist;

        private final Shape<?>                     m_prim;

        private final int                          m_pathIndex;

        private final int                          m_entryIndex;

        public PointControlHandle(Shape<?> prim, int pathIndex, int entryIndex, Shape<?> shape, NFastArrayList<PathPartList> listOfPaths, PathPartList plist, IControlHandleList hlist)
        {
            m_shape = shape;

            m_listOfPaths = listOfPaths;

            m_chlist = hlist;

            m_prim = prim;

            m_pathIndex = pathIndex;

            m_entryIndex = entryIndex;

            init();
        }

        public void init()
        {
            PointHandleDragHandler topRightHandler = new PointHandleDragHandler(m_shape, m_listOfPaths, m_chlist, m_prim, this);

            register(m_prim.addNodeDragMoveHandler(topRightHandler));

            register(m_prim.addNodeDragStartHandler(topRightHandler));

            register(m_prim.addNodeDragEndHandler(topRightHandler));
        }

        public int getPathIndex()
        {
            return m_pathIndex;
        }

        public int getEntryIndex()
        {
            return m_entryIndex;
        }

        @Override
        public IPrimitive<?> getControl()
        {
            return m_prim;
        }

        @Override
        public void destroy()
        {
            super.destroy();
        }

        @Override
        public final ControlHandleType getType()
        {
            return ControlHandleStandardType.POINT;
        }
    }

    public static class PointHandleDragHandler implements NodeDragStartHandler, NodeDragMoveHandler, NodeDragEndHandler
    {
        protected final Shape<?>                      m_shape;

        private final NFastArrayList<PathPartList>    m_listOfPaths;

        protected final IControlHandleList            m_chlist;

        protected final Shape<?>                      m_prim;

        protected final PointControlHandle            m_handle;

        protected NFastArrayList<NFastDoubleArrayJSO> m_entries;

        public PointHandleDragHandler(Shape<?> shape, NFastArrayList<PathPartList> listOfPaths, IControlHandleList chlist, Shape<?> prim, PointControlHandle handle)
        {
            m_shape = shape;

            m_listOfPaths = listOfPaths;

            m_chlist = chlist;

            m_prim = prim;

            m_handle = handle;
        }

        @Override
        public void onNodeDragStart(NodeDragStartEvent event)
        {
            copyDoubles();

            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                m_prim.setFillColor(ColorName.GREEN);

                m_prim.getLayer().draw();
            }
        }

        public void onNodeDragMove(NodeDragMoveEvent event)
        {
            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                double dx = event.getDragContext().getDx();

                double dy = event.getDragContext().getDy();

                PathPartList list = (PathPartList) m_listOfPaths.get(m_handle.getPathIndex());

                PathPartEntryJSO entry = list.get(m_handle.getEntryIndex());

                NFastDoubleArrayJSO points = entry.getPoints();

                switch (entry.getCommand())
                {
                    case PathPartEntryJSO.MOVETO_ABSOLUTE:
                    case PathPartEntryJSO.LINETO_ABSOLUTE:
                    {
                        NFastDoubleArrayJSO doubles = m_entries.get(m_handle.getEntryIndex());

                        double x = doubles.get(0);

                        double y = doubles.get(1);

                        points.set(0, x + dx);

                        points.set(1, y + dy);

                        break;
                    }
                }
                m_shape.refresh();

                m_shape.getLayer().batch();
            }
        }

        @Override
        public void onNodeDragEnd(NodeDragEndEvent event)
        {
            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                NFastArrayList<PathPartList> lists = m_listOfPaths;

                for (PathPartList list : lists)
                {
                    list.resetBoundingBox();
                }
                m_prim.setFillColor(ColorName.RED);

                m_prim.getLayer().draw();
            }
        }

        private void copyDoubles()
        {
            m_entries = new NFastArrayList<NFastDoubleArrayJSO>();

            NFastArrayList<PathPartList> lists = m_listOfPaths;

            for (PathPartList list : lists)
            {
                for (int i = 0; i < list.size(); i++)
                {
                    PathPartEntryJSO entry = list.get(i);

                    NFastDoubleArrayJSO points = entry.getPoints();

                    switch (entry.getCommand())
                    {
                        case PathPartEntryJSO.MOVETO_ABSOLUTE:
                        case PathPartEntryJSO.LINETO_ABSOLUTE:
                        {
                            double x = points.get(0);

                            double y = points.get(1);

                            NFastDoubleArrayJSO doubles = NFastDoubleArrayJSO.make(x, y);

                            m_entries.push(doubles);

                            break;
                        }
                    }
                }
            }
        }
    }

    private static class ResizeControlHandle extends AbstractControlHandle
    {
        private static final long                    serialVersionUID = 1L;

        private final Shape<?>                       m_shape;

        private final NFastArrayList<PathPartList>   m_listOfPaths;

        private final IControlHandleList             m_chlist;

        private final ArrayList<ResizeControlHandle> m_orderedChList;

        private final Shape<?>                       m_prim;

        private int                                  m_position;

        public ResizeControlHandle(Circle prim, IControlHandleList hlist, ArrayList<ResizeControlHandle> orderedChList, Shape<?> shape, NFastArrayList<PathPartList> listOfPaths, int position)
        {
            m_prim = prim;

            m_chlist = hlist;

            m_shape = shape;

            m_position = position;

            m_orderedChList = orderedChList;

            m_listOfPaths = listOfPaths;

            init();
        }

        public void init()
        {
            ResizeHandleDragHandler topRightHandler = new ResizeHandleDragHandler(m_shape, m_listOfPaths, m_chlist, m_prim, this);

            register(m_prim.addNodeDragMoveHandler(topRightHandler));

            register(m_prim.addNodeDragStartHandler(topRightHandler));

            register(m_prim.addNodeDragEndHandler(topRightHandler));
        }

        public int getPosition()
        {
            return m_position;
        }

        public void setPosition(int position)
        {
            this.m_position = position;
        }

        @Override
        public IPrimitive<?> getControl()
        {
            return m_prim;
        }

        @Override
        public void destroy()
        {
            super.destroy();
        }

        @Override
        public ControlHandleType getType()
        {
            return ControlHandleStandardType.RESIZE;
        }

        public Shape<?> getPrimitive()
        {
            return m_prim;
        }

        public double getX(double startTopLeftX, double startTopLeftY, double startW, double startH, double x, double dx)
        {
            double newX = 0;

            switch (m_position)
            {
                case 0:
                case 3:
                    newX = getLeft(startTopLeftX, startTopLeftY, startW, startH, x, dx);
                    break;
                case 1:
                case 2:
                    newX = getRight(startTopLeftX, startTopLeftY, startW, startH, x, dx);
                    break;
            }
            return newX;
        }

        public double getY(double startTopLeftX, double startTopLeftY, double startW, double startH, double y, double dy)
        {
            double newY = 0;

            switch (m_position)
            {
                case 0:
                case 1:
                    newY = getTop(startTopLeftX, startTopLeftY, startW, startH, y, dy);
                    break;
                case 2:
                case 3:
                    newY = getBottom(startTopLeftX, startTopLeftY, startW, startH, y, dy);
                    break;
            }
            return newY;
        }

        void updateOtherHandles(double dx, double dy, double offsetX, double offsetY, double boxStartX, double boxStartY, double boxStartWidth, double boxStartHeight)
        {
            switch (m_position)
            {
                case 0:
                {
                    IControlHandle topRight = m_orderedChList.get(1);
                    topRight.getControl().setY(boxStartY + dy + offsetY);

                    IControlHandle bottomLeft = m_orderedChList.get(3);
                    bottomLeft.getControl().setX(boxStartX + dx + offsetX);
                    break;
                }
                case 1:
                {
                    IControlHandle topLeft = m_orderedChList.get(0);
                    topLeft.getControl().setY(boxStartY + dy + offsetY);

                    IControlHandle bottomRight = m_orderedChList.get(2);
                    bottomRight.getControl().setX(boxStartX + boxStartWidth + dx + offsetX);
                    break;
                }
                case 2:
                {
                    IControlHandle topRight = m_orderedChList.get(1);
                    topRight.getControl().setX(boxStartX + boxStartWidth + dx + offsetX);

                    IControlHandle bottomLeft = m_orderedChList.get(3);
                    bottomLeft.getControl().setY(boxStartY + boxStartHeight + dy + offsetY);
                    break;
                }
                case 3:
                {
                    IControlHandle topLeft = m_orderedChList.get(0);
                    topLeft.getControl().setX(boxStartX + dx + offsetX);

                    IControlHandle bottomRight = m_orderedChList.get(2);
                    bottomRight.getControl().setY(boxStartY + boxStartHeight + dy + offsetY);
                    break;
                }
            }
        }

        double getLeft(double startTopLeftX, double startTopLeftY, double startW, double startH, double x, double dx)
        {
            double wpc = ((100 / startW) * ((startTopLeftX + startW) - x)) / 100;
            double newX = x + (dx * wpc);
            return newX;
        }

        double getRight(double startTopLeftX, double startTopLeftY, double startW, double startH, double x, double dx)
        {
            double wpc = ((100 / startW) * (x - startTopLeftX)) / 100;
            double newX = x + (dx * wpc);
            return newX;
        }

        double getTop(double startTopLeftX, double startTopLeftY, double startW, double startH, double y, double dy)
        {
            double hpc = ((100 / startH) * ((startTopLeftY + startH) - y)) / 100;
            double newY = y + (dy * hpc);
            return newY;
        }

        double getBottom(double startTopLeftX, double startTopLeftY, double startW, double startH, double y, double dy)
        {
            double hpc = ((100 / startH) * (y - startTopLeftY)) / 100;
            double newY = y + (dy * hpc);
            return newY;
        }
    }

    public static class ResizeHandleDragHandler implements NodeDragStartHandler, NodeDragMoveHandler, NodeDragEndHandler
    {
        private final Shape<?>                      m_shape;

        private final NFastArrayList<PathPartList>  m_listOfPaths;

        private final IControlHandleList            m_chlist;

        private final Shape<?>                      m_prim;

        private final ResizeControlHandle           m_handle;

        private double                              m_boxStartX;

        private double                              m_boxStartY;

        private double                              m_boxStartWidth;

        private double                              m_boxStartHeight;

        private NFastArrayList<NFastDoubleArrayJSO> m_entries;

        private double                              m_offsetX;

        private double                              m_offsetY;

        public ResizeHandleDragHandler(Shape<?> shape, NFastArrayList<PathPartList> listOfPaths, IControlHandleList chlist, Shape<?> prim, ResizeControlHandle handle)
        {
            m_shape = shape;
            m_listOfPaths = listOfPaths;
            m_chlist = chlist;
            m_prim = prim;
            m_handle = handle;
        }

        @Override
        public void onNodeDragStart(NodeDragStartEvent event)
        {
            BoundingBox box = m_shape.getBoundingBox();

            m_boxStartX = box.getX();
            m_boxStartY = box.getY();
            m_boxStartWidth = box.getWidth();
            m_boxStartHeight = box.getHeight();

            m_offsetX = m_shape.getX();
            m_offsetY = m_shape.getY();

            repositionAndResortHandles(box);

            copyDoubles();

            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                m_prim.setFillColor(ColorName.GREEN);

                m_prim.getLayer().draw();
            }
        }

        /**
         * If the handles are flip horizontally or vertically, they must be re-aligned with the correct box corner.
         * @param box
         */
        private void repositionAndResortHandles(BoundingBox box)
        {
            double x = m_handle.m_orderedChList.get(0).getPrimitive().getX();

            double y = m_handle.m_orderedChList.get(0).getPrimitive().getY();

            ResizeControlHandle topLeft = m_handle.m_orderedChList.get(0);

            for (ResizeControlHandle handle : m_handle.m_orderedChList)
            {
                if (handle.getPrimitive().getX() <= x && handle.getPrimitive().getY() <= y)
                {
                    x = handle.getPrimitive().getX();

                    y = handle.getPrimitive().getY();

                    topLeft = handle;
                }
            }
            topLeft.setPosition(0);

            x = m_handle.m_orderedChList.get(0).getPrimitive().getX();
            y = m_handle.m_orderedChList.get(0).getPrimitive().getY();
            ResizeControlHandle topRight = m_handle.m_orderedChList.get(0);
            for (ResizeControlHandle handle : m_handle.m_orderedChList)
            {
                if (handle.getPrimitive().getX() >= x && handle.getPrimitive().getY() <= y)
                {
                    x = handle.getPrimitive().getX();
                    y = handle.getPrimitive().getY();
                    topRight = handle;
                }
            }
            topRight.setPosition(1);

            x = m_handle.m_orderedChList.get(0).getPrimitive().getX();
            y = m_handle.m_orderedChList.get(0).getPrimitive().getY();
            ResizeControlHandle bottomRight = m_handle.m_orderedChList.get(0);
            for (ResizeControlHandle handle : m_handle.m_orderedChList)
            {
                if (handle.getPrimitive().getX() >= x && handle.getPrimitive().getY() >= y)
                {
                    x = handle.getPrimitive().getX();
                    y = handle.getPrimitive().getY();
                    bottomRight = handle;
                }
            }
            bottomRight.setPosition(2);

            x = m_handle.m_orderedChList.get(0).getPrimitive().getX();
            y = m_handle.m_orderedChList.get(0).getPrimitive().getY();
            ResizeControlHandle bottomLeft = m_handle.m_orderedChList.get(0);
            for (ResizeControlHandle handle : m_handle.m_orderedChList)
            {
                if (handle.getPrimitive().getX() <= x && handle.getPrimitive().getY() >= y)
                {
                    x = handle.getPrimitive().getX();
                    y = handle.getPrimitive().getY();
                    bottomLeft = handle;
                }
            }
            bottomLeft.setPosition(3);

            Collections.sort(m_handle.m_orderedChList, new Comparator<ResizeControlHandle>()
            {
                @Override
                public int compare(ResizeControlHandle h1, ResizeControlHandle h2)
                {
                    if (h1.getPosition() > h2.getPosition())
                    {
                        return 1;
                    }
                    if (h1.getPosition() < h2.getPosition())
                    {
                        return -1;
                    }
                    else
                    {
                        return 0;
                    }
                }
            });
        }

        public void onNodeDragMove(NodeDragMoveEvent event)
        {
            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                double dx = event.getDragContext().getDx();
                double dy = event.getDragContext().getDy();

                for (PathPartList list : m_listOfPaths)
                {
                    for (int i = 0; i < list.size(); i++)
                    {
                        PathPartEntryJSO entry = list.get(i);
                        NFastDoubleArrayJSO points = entry.getPoints();
                        switch (entry.getCommand())
                        {
                            case PathPartEntryJSO.MOVETO_ABSOLUTE:
                            case PathPartEntryJSO.LINETO_ABSOLUTE:
                            {
                                NFastDoubleArrayJSO doubles = m_entries.get(i);
                                double x = doubles.get(0);
                                double newX = m_handle.getX(m_boxStartX, m_boxStartY, m_boxStartWidth, m_boxStartHeight, x, dx);

                                double y = doubles.get(1);
                                double newY = m_handle.getY(m_boxStartX, m_boxStartY, m_boxStartWidth, m_boxStartHeight, y, dy);

                                points.set(0, newX);
                                points.set(1, newY);
                                break;
                            }
                        }
                    }
                }
                m_handle.updateOtherHandles(dx, dy, m_offsetX, m_offsetY, m_boxStartX, m_boxStartY, m_boxStartWidth, m_boxStartHeight);

                m_shape.refresh();

                m_shape.getLayer().batch();
            }
        }

        @Override
        public void onNodeDragEnd(NodeDragEndEvent event)
        {
            if ((m_handle.isActive()) && (m_chlist.isActive()))
            {
                for (PathPartList list : m_listOfPaths)
                {
                    list.resetBoundingBox();
                }
                m_prim.setFillColor(ColorName.RED);
                m_prim.getLayer().draw();
            }
        }

        private void copyDoubles()
        {
            m_entries = new NFastArrayList<NFastDoubleArrayJSO>();

            for (PathPartList list : m_listOfPaths)
            {
                for (int i = 0; i < list.size(); i++)
                {
                    PathPartEntryJSO entry = list.get(i);
                    NFastDoubleArrayJSO points = entry.getPoints();
                    switch (entry.getCommand())
                    {
                        case PathPartEntryJSO.MOVETO_ABSOLUTE:
                        case PathPartEntryJSO.LINETO_ABSOLUTE:
                        {
                            double x = points.get(0);
                            double y = points.get(1);
                            NFastDoubleArrayJSO doubles = NFastDoubleArrayJSO.make(x, y);
                            m_entries.push(doubles);
                            break;
                        }
                    }
                }
            }
        }
    }
}
