/* Copyright 2018 AHEAD iTec, s.r.o

   Permission to use, copy, modify, and/or distribute this software
   for any purpose with or without fee is hereby granted, provided
   that the above copyright notice and this permission notice appear
   in all copies.

   There is NO WARRANTY for this software.  See LICENSE for
   details. */
package com.aheaditec.sensitiveuserinputview;

import android.os.Parcel;
import android.os.Parcelable;

import androidx.annotation.NonNull;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;

/**
 * @author Milan Jiricek <milan.jiricek@ahead-itec.com>
 * @author Jakub Jerabek <jakub.jerabek@ahead-itec.com>
 */
public class SecureString implements CharSequence, Parcelable {

    private char[] value;

    /**
     * Creates an empty secure string.
     */
    public SecureString() {
        this(0);
    }

    /**
     * Constructs a new SecureString which controls the passed in char array.
     *
     * @param length the length of this secure string.
     */
    public SecureString(int length) {
        if (length < 0) {
            throw new IllegalArgumentException("Length must be a positive value!");
        }
        this.value = new char[length];
    }

    /**
     * Constructs a new SecureString which controls the passed in char array.
     *
     * @param value char[]
     */
    public SecureString(@NonNull char[] value) {
        if (value == null) {
            throw new IllegalArgumentException("Value must not be null!");
        }
        this.value = value;
    }

    private SecureString(@NonNull char[] value, int start, int end) {
        if (value == null) {
            throw new IllegalArgumentException("Value must not be null!");
        }
        this.value = new char[end - start + 1];
        System.arraycopy(value, start, this.value, 0, this.value.length);
    }

    /**
     * Get length of current input.
     *
     * @return the length of the sequence of characters represented by this object.
     */
    @Override
    public int length() {
        return value.length;
    }

    /**
     * Get char at given position.
     *
     * @param index the index of the {@code char} value.
     * @return the {@code char} value at the specified index of this object.
     * @throws ArrayIndexOutOfBoundsException if the {@code index} argument is negative or not less than the length of this secure string.
     */
    @Override
    public char charAt(int index) {
        return value[index];
    }

    /**
     * Returns a character sequence that is a subsequence of this sequence.
     *
     * @param start the begin index, inclusive.
     * @param end   the end index, exclusive.
     * @return the specified subsequence.
     * @throws NegativeArraySizeException     if {@code beginIndex} is greater than {@code endIndex}
     * @throws ArrayIndexOutOfBoundsException if {@code beginIndex} or {@code endIndex} is negative,
     *                                        if {@code endIndex} is greater than {@code length()}
     */
    @Override
    public SecureString subSequence(int start, int end) {
        return new SecureString(this.value, start, end);
    }

    @NonNull
    @Override
    public String toString() {
        throw new UnsupportedOperationException("This method is not supported.");
    }

    //#############
    //#### Help methods
    //#############

    /**
     * Manually clear the underlying char array.
     */
    public void clear() {
        Arrays.fill(value, '\u0000');
    }

    /**
     * Converts this secure string to a new character array.
     *
     * @return a newly allocated character array whose length is the length
     * of this secure string and whose contents are initialized to contain
     * the character sequence represented by this secure string.
     */
    @NonNull
    public char[] toCharArray() {
        return Arrays.copyOf(value, value.length);
    }

    /**
     * Encodes this {@link SecureString} into a sequence of bytes using the
     * platform's default charset, storing the result into a new byte array.
     *
     * @param charset The {@linkplain java.nio.charset.Charset} to be used to encode
     *                the {@link SecureString}
     * @return a newly allocated byte array whose length is the length
     * of this secure string and whose contents are initialized to contain
     * the byte sequence represented by this secure string.
     */
    @NonNull
    public byte[] toByteArray(@NonNull Charset charset) {
        if (charset == null) {
            throw new IllegalArgumentException("Charset must not be null!");
        }

        CharBuffer charBuffer = CharBuffer.wrap(value);
        ByteBuffer byteBuffer = charset.encode(charBuffer);
        return Arrays.copyOfRange(byteBuffer.array(),
                byteBuffer.position(), byteBuffer.limit());
    }

    /**
     * Encodes this {@link SecureString} into a sequence of bytes using the
     * platform's default charset, storing the result into a new byte array.
     *
     * @return a newly allocated byte array whose length is the length
     * of this secure string and whose contents are initialized to contain
     * the byte sequence represented by this secure string.
     */
    @NonNull
    public byte[] toByteArray() {
        return toByteArray(Charset.defaultCharset());
    }

    /**
     * This is a dangerous operation as the array may be modified while it is being used by other threads
     * or a consumer may modify the values in the array. For safety, it is preferable to use {@link #toCharArray()}.
     *
     * @return the underlying char array.
     */
    public char[] getChars() {
        return value;
    }

    /**
     * The character at the specified position is set to newChar.
     * The index argument must be greater than or equal to 0, and less than the length of this sequence.
     *
     * @param position the position of the character to modify.
     * @param newChar  the new character.
     * @throws ArrayIndexOutOfBoundsException if index is negative or greater than or equal to length().
     */
    public void setCharAt(int position, char newChar) throws ArrayIndexOutOfBoundsException {
        value[position] = newChar;
    }

    /**
     * The character at the specified position will be cleared.
     */
    public void clearCharAt(int position) {
        setCharAt(position, '\u0000');
    }

    /**
     * Appends a character to the secure string.
     *
     * @param ch the character to append
     */
    public void appendChar(char ch) {
        final int n = length();
        char[] valueCopy = Arrays.copyOf(value, n + 1);
        clear();
        value = valueCopy;
        value[n] = ch;
    }

    public void removeLastChar() {
        final int length = length();
        if (length > 0) {
            char[] valueCopy = Arrays.copyOfRange(value, 0, length - 1);
            clear();
            value = valueCopy;
        }
    }

    @Override
    public SecureString clone() {
        return new SecureString(toCharArray());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SecureString that = (SecureString) o;
        return Arrays.equals(value, that.value);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(value);
    }

    //#############
    //#### Parcelable implementation
    //#############

    private SecureString(Parcel in) {
        value = in.createCharArray();
    }

    public static final Creator<SecureString> CREATOR = new Creator<SecureString>() {
        @Override
        public SecureString createFromParcel(Parcel in) {
            return new SecureString(in);
        }

        @Override
        public SecureString[] newArray(int size) {
            return new SecureString[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeCharArray(value);
    }
}
