/* Copyright 2018 AHEAD iTec, s.r.o

   Permission to use, copy, modify, and/or distribute this software
   for any purpose with or without fee is hereby granted, provided
   that the above copyright notice and this permission notice appear
   in all copies.

   There is NO WARRANTY for this software.  See LICENSE for
   details. */
package com.aheaditec.sensitiveuserinputview;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.ColorInt;
import androidx.annotation.FloatRange;
import androidx.annotation.IntDef;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.aheaditec.core.BaseCompoundView;
import com.aheaditec.core.utils.AttrHelper;
import com.aheaditec.core.utils.DrawableHelper;
import com.aheaditec.sensitiveuserinputview.interfaces.OnPinEnteredListener;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * A PIN entry view widget for Android
 *
 * @author Milan Jiricek <milan.jiricek@ahead-itec.com>
 */

public class SensitiveUserInputView extends BaseCompoundView {

    private static final String TAG = SensitiveUserInputView.class.getSimpleName();

    private static final boolean DBG = false;

    public static final int VISIBLE_PIN = 4;
    public static final int HIDDEN_PIN = 5;

    @Retention(RetentionPolicy.SOURCE)
    @IntDef({VISIBLE_PIN, HIDDEN_PIN})
    private @interface PinNumberVisibility {
    }

    private boolean newlyEntered;

    /**
     * Number of digits
     */
    private int digits;

    /**
     * Pin colors
     */
    @ColorInt
    private int pinFocusColor, pinDotColor, pinTextColor, remainingDigitsColor;

    @FloatRange(from = 0.0, to = 1.0)
    private float remainingDigitsAlpha;

    @IntRange(from = 0, to = Integer.MAX_VALUE)
    private int cursorPosition;

    @PinNumberVisibility
    private int pinNumberVisibility = HIDDEN_PIN;

    private AnimatorSet animatorSet, showHideAnimatorSet, resetAnimatorSet;
    private OnPinEnteredListener pinEnteredListener;

    public SensitiveUserInputView(Context context) {
        this(context, null);
    }

    public SensitiveUserInputView(Context context, @Nullable AttributeSet set) {
        this(context, set, 0);
    }

    public SensitiveUserInputView(Context context, @Nullable AttributeSet set, int defStyleAttr) {
        super(context, set, defStyleAttr);

        if (set == null) {
            return;
        }

        digits = AttrHelper.resolveInt(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_digits);
        pinFocusColor = AttrHelper.resolveColor(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_pinFocusColor, android.R.color.white);
        pinDotColor = AttrHelper.resolveColor(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_pinDotColor);
        pinTextColor = AttrHelper.resolveColor(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_pinTextColor);
        remainingDigitsColor = AttrHelper.resolveColor(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_remainingDigitsColor);
        remainingDigitsAlpha = AttrHelper.resolveFloat(context, set, R.styleable.SensitiveUserInputView, R.styleable.SensitiveUserInputView_remainingDigitsAlpha);

        addPinDigitsView(context);
    }

    @Override
    protected void onLayoutInflated() {
        if (DBG) {
            Log.d(TAG, "Layout inflation has completed!");
        }

        if (digits == AttrHelper.VALUE_NOT_SET) {
            return;
        }

        setupAttributes();
        setUpCursor();
    }

    //#############
    //#### Help methods
    //#############

    private void addPinDigitsView(Context context) {
        for (int i = 0; i < digits; i++) {
            if (DBG) {
                Log.d(TAG, "Add view number " + i);
            }

            addView(context, R.layout.pin_layout);
        }
    }

    private void setupPinDigitsView(Context context) {
        int numberOfChildren = getChildCount();
        if (numberOfChildren == digits) {
            return;
        }

        this.cursorPosition = 0;
        if (getChildCount() > 0) {
            removeAllViews();
        }
        addPinDigitsView(context);
    }

    private void setupAttributes() {
        setupTextAndFocusColor();
    }

    private void setupTextAndFocusColor() {
        boolean defaultPinFocusColor = pinFocusColor == AttrHelper.VALUE_NOT_SET,
                defaultPinTextColor = pinTextColor == AttrHelper.VALUE_NOT_SET,
                defaultPinDotColor = pinDotColor == AttrHelper.VALUE_NOT_SET,
                defaultRemainingDigitsColor = remainingDigitsColor == AttrHelper.VALUE_NOT_SET;

        if (defaultPinFocusColor && defaultPinTextColor
                && defaultRemainingDigitsColor && defaultPinDotColor) {
            return;
        }

        Context context = getContext();
        for (int i = 0; i < getChildCount(); i++) { // set the final colors

            View pinView = getChildAt(i);
            ImageView imgPinInput = pinView.findViewById(R.id.imgPinInput);
            if (i > cursorPosition) {
                if (!defaultRemainingDigitsColor) {
                    setupVectorDrawableColor(context, imgPinInput, remainingDigitsColor);
                } else if (!defaultPinFocusColor) {
                    setupVectorDrawableColor(context, imgPinInput, pinFocusColor);
                }

                imgPinInput.setAlpha(remainingDigitsAlpha);
            } else {
                if (!defaultPinFocusColor) {
                    setupVectorDrawableColor(context, imgPinInput, pinFocusColor);
                }
            }

            if (!defaultPinTextColor) {
                ((TextView) pinView.findViewById(R.id.txtPinInput)).setTextColor(pinTextColor);
            }
            if (!defaultPinDotColor) {
                setupVectorDrawableColor(context, (ImageView) pinView.findViewById(R.id.imgPinDot), pinDotColor);
            }
        }
    }

    private void setupVectorDrawableColor(Context context, @NonNull ImageView imageView, @ColorInt int color) {
        DrawableHelper.newInstance(context)
                .setColor(color)
                .setDrawable(imageView.getDrawable())
                .tint()
                .applyTo(imageView);
    }

    /**
     * Resets alpha channel (imgPinInput) to 100 %
     */
    private void resetAlpha() {
        for (int i = 0; i < getChildCount(); i++) {

            View rootItemView = getChildAt(i);
            ImageView imgPinInput = rootItemView.findViewById(R.id.imgPinInput);
            if (i > cursorPosition) {
                imgPinInput.setAlpha(remainingDigitsAlpha);
            } else {
                imgPinInput.setAlpha(1.0f);

                TextView txtCurrentPinInput = rootItemView.findViewById(R.id.txtPinInput);
                ImageView imgCurrentPinDot = rootItemView.findViewById(R.id.imgPinDot);
                if (pinNumberVisibility == HIDDEN_PIN) {
                    txtCurrentPinInput.setAlpha(0.0f);
                    imgCurrentPinDot.setAlpha(1.0f);
                } else if (pinNumberVisibility == VISIBLE_PIN) {
                    txtCurrentPinInput.setAlpha(1.0f);
                    imgCurrentPinDot.setAlpha(0.0f);
                }
            }
        }
    }

    /**
     * Set the cursor to the current position
     */
    private void setUpCursor() {
        endAnimations();

        ImageView imgNextFocused = getChildAt(cursorPosition).findViewById(R.id.imgPinInput);
        animatorSet = startBlinkAnimation(imgNextFocused);
    }

    private void setUpCursor(@NonNull ImageView imageView) {
        animatorSet = startBlinkAnimation(imageView);
    }

    private void endAnimations() {
        if (animatorSet != null) {
            animatorSet.end();
            animatorSet = null;
        }
        if (resetAnimatorSet != null) {
            resetAnimatorSet.end();
            resetAnimatorSet = null;
        }
    }

    //#############
    //#### Animations
    //#############

    /**
     * Runs animation when input is entered
     *
     * @param txtCurrentPinInput text view with entered number
     * @param imgCurrentPinDot   PIN dot with focus
     * @param imgNextFocused     next focused image view
     * @return animator set
     */
    @NonNull
    private AnimatorSet startEnteredAnimation(@NonNull TextView txtCurrentPinInput, @NonNull ImageView imgCurrentPinDot, @Nullable ImageView imgNextFocused) {

        if (pinNumberVisibility == VISIBLE_PIN) {
            return startVisibleAnimation(txtCurrentPinInput, imgNextFocused);
        }

        return startHiddenAnimation(txtCurrentPinInput, imgCurrentPinDot, imgNextFocused);
    }

    private AnimatorSet startVisibleAnimation(@NonNull final TextView txtCurrentPinInput, @Nullable final ImageView imgNextFocused) {
        AnimatorSet animatorSet = new AnimatorSet();

        ObjectAnimator dotFadeIn = ObjectAnimator.ofFloat(txtCurrentPinInput, View.ALPHA, 0, 1).setDuration(400);
        animatorSet.play(dotFadeIn);

        if (imgNextFocused != null) {
            final ObjectAnimator blink = ObjectAnimator.ofFloat(imgNextFocused, View.ALPHA, 1, 1, 0.5f, 0.5f, 0.5f).setDuration(500);

            blink.setRepeatMode(ValueAnimator.REVERSE);
            blink.setRepeatCount(ValueAnimator.INFINITE);

            animatorSet.play(blink).after(750);
        }

        dotFadeIn.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationStart(Animator animation) {
                super.onAnimationStart(animation);
                if (pinNumberVisibility == VISIBLE_PIN) {
                    txtCurrentPinInput.setVisibility(VISIBLE);
                }
            }
        });

        setUpResetAnimationListener(animatorSet);
        animatorSet.start();

        return animatorSet;
    }

    private AnimatorSet startHiddenAnimation(@NonNull final TextView txtCurrentPinInput, @NonNull final ImageView imgCurrentPinDot, @Nullable final ImageView imgNextFocused) {
        txtCurrentPinInput.setVisibility(VISIBLE);
        txtCurrentPinInput.setAlpha(1.0f);

        ObjectAnimator fadeOut = ObjectAnimator.ofFloat(txtCurrentPinInput, View.ALPHA, 1, 0).setDuration(200);
        ObjectAnimator dotFadeIn = ObjectAnimator.ofFloat(imgCurrentPinDot, View.ALPHA, 0, 1).setDuration(400);

        final AnimatorSet animatorSet = new AnimatorSet();
        animatorSet.play(fadeOut).after(400);
        animatorSet.play(dotFadeIn).after(400);

        if (imgNextFocused != null) {
            final ObjectAnimator blink = ObjectAnimator.ofFloat(imgNextFocused, View.ALPHA, 1, 1, 0.5f, 0.5f, 0.5f).setDuration(500);

            blink.setRepeatMode(ValueAnimator.REVERSE);
            blink.setRepeatCount(ValueAnimator.INFINITE);

            animatorSet.play(fadeOut).before(blink);
            animatorSet.play(blink).after(750);
        }

        dotFadeIn.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationStart(Animator animation) {
                super.onAnimationStart(animation);
                if (pinNumberVisibility == HIDDEN_PIN) {
                    imgCurrentPinDot.setVisibility(VISIBLE);
                }
            }
        });

        setUpResetAnimationListener(animatorSet);
        animatorSet.start();

        return animatorSet;
    }

    /**
     * Sets actions to do when animation ends
     *
     * @param animatorSet animator set
     */
    private void setUpResetAnimationListener(@NonNull AnimatorSet animatorSet) {
        animatorSet.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationEnd(Animator animation) {
                super.onAnimationEnd(animation);
                resetAlpha();

                if (cursorPosition >= digits) {
                    if (DBG) {
                        Log.d(TAG, "Input is complete.");
                    }

                    if (pinEnteredListener != null && newlyEntered) {
                        newlyEntered = false;
                        pinEnteredListener.onPinEntered(getInputSecureString());
                    }
                }
            }
        });
    }

    /**
     * Starts blinking animation on focused input
     *
     * @param imgFocused focused image view
     * @return animator set @{@link AnimatorSet}
     */
    private AnimatorSet startBlinkAnimation(@NonNull final ImageView imgFocused) {
        final ObjectAnimator blink = ObjectAnimator.ofFloat(imgFocused, View.ALPHA, 1, 1, 0.5f, 0.5f, 0.5f).setDuration(500);
        blink.setRepeatMode(ValueAnimator.REVERSE);
        blink.setRepeatCount(ValueAnimator.INFINITE);

        final AnimatorSet animatorSet = new AnimatorSet();
        animatorSet.play(blink).after(400);

        setUpResetAnimationListener(animatorSet);
        animatorSet.start();

        return animatorSet;
    }

    /**
     * Clears all animation
     *
     * @param currentImageView     current image view
     * @param nextFocusedTextView  next focused text view
     * @param nextFocusedImageView next focused image view
     * @param nextFocusedDot       next focused dot
     */
    private void clearAllAnimation(@NonNull ImageView currentImageView, @NonNull TextView nextFocusedTextView, @NonNull ImageView nextFocusedImageView, @NonNull ImageView nextFocusedDot) {
        clearAllAnimations(nextFocusedTextView, nextFocusedImageView, nextFocusedDot);
        currentImageView.clearAnimation();
    }

    /**
     * Clears all animation
     *
     * @param nextFocusedTextView  next focused text view
     * @param nextFocusedImageView next focused image view
     * @param nextFocusedDot       next focused dot
     */
    private void clearAllAnimations(@NonNull TextView nextFocusedTextView, @NonNull ImageView nextFocusedImageView, @NonNull ImageView nextFocusedDot) {
        nextFocusedDot.clearAnimation();
        nextFocusedTextView.clearAnimation();
        nextFocusedImageView.clearAnimation();
    }

    /**
     * Runs animation when input is deleted
     *
     * @param focusedImageView     image view with focus (blinking cursor)
     * @param nextFocusedTextView  text view which will have focus
     * @param nextFocusedImageView image view which will have focus
     * @param nextFocusedDot       PIN dot for next focused element
     */
    private void setupDeletePress(@Nullable ImageView focusedImageView, @NonNull TextView nextFocusedTextView, @NonNull ImageView nextFocusedImageView, @NonNull ImageView nextFocusedDot) {
        if (focusedImageView == null) {
            clearAllAnimations(nextFocusedTextView, nextFocusedImageView, nextFocusedDot);
        } else {
            clearAllAnimation(focusedImageView, nextFocusedTextView, nextFocusedImageView, nextFocusedDot);
            focusedImageView.setAlpha(remainingDigitsAlpha);
            if (remainingDigitsColor != AttrHelper.VALUE_NOT_SET) {
                setupVectorDrawableColor(getContext(), focusedImageView, remainingDigitsColor);
            }
        }

        setText(' ', nextFocusedTextView);
        nextFocusedTextView.setVisibility(View.INVISIBLE);
        nextFocusedDot.setVisibility(View.INVISIBLE);
        animatorSet = startBlinkAnimation(nextFocusedImageView);
    }

    //#############
    //#### Public methods
    //#############

    /**
     * Inserts given number to the first empty position
     *
     * @param number Character with number
     */
    public void insertNumber(char number) {
        if (digits == AttrHelper.VALUE_NOT_SET || getChildCount() == 0) { // Check to make sure the length is correct
            throw new IllegalArgumentException("PIN length must be greater than 0. PIN length = " + digits);
        }

        if (cursorPosition < digits) {
            newlyEntered = true;
            endAnimations();

            View pinRootView = getChildAt(cursorPosition);
            TextView txtPinInput = pinRootView.findViewById(R.id.txtPinInput);
            ImageView imgPinDot = pinRootView.findViewById(R.id.imgPinDot);
            ImageView imgPinInput = pinRootView.findViewById(R.id.imgPinInput);
            if (pinFocusColor != AttrHelper.VALUE_NOT_SET) {
                setupVectorDrawableColor(getContext(), imgPinInput, pinFocusColor);
            }

            cursorPosition++;
            ImageView imgAfterPinInput = null;
            if (cursorPosition < digits) {
                View afterPinRootView = getChildAt(cursorPosition);
                imgAfterPinInput = afterPinRootView.findViewById(R.id.imgPinInput);
            }

            setText(number, txtPinInput);
            animatorSet = startEnteredAnimation(txtPinInput, imgPinDot, imgAfterPinInput);
        } else {
            if (DBG) {
                Log.d(TAG, "The PIN is entered");
            }
        }
    }

    private void setText(char character, @NonNull TextView txtPinInput) {
        txtPinInput.setText(new char[]{character}, 0, 1);
    }

    /**
     * Deletes last number in input
     */
    public void deleteLastNumber() {
        if (cursorPosition <= 0) {
            if (DBG) {
                Log.d(TAG, "There is nothing to delete. cursorPosition = " + cursorPosition);
            }
            return;
        }

        endAnimations();
        if (cursorPosition <= 0) { // A resetInput method can be called
            return;
        }

        ImageView imgFocusedPinInput = null;
        if (cursorPosition < digits) {
            View focusedPinRootView = getChildAt(cursorPosition);
            imgFocusedPinInput = focusedPinRootView.findViewById(R.id.imgPinInput);
        }

        cursorPosition--;

        View nextFocusedPinRootView = getChildAt(cursorPosition);
        TextView txtNextFocusedPinInput = nextFocusedPinRootView.findViewById(R.id.txtPinInput);
        ImageView imgNextFocusedPinInput = nextFocusedPinRootView.findViewById(R.id.imgPinInput);
        ImageView imgNextFocusedPinDot = nextFocusedPinRootView.findViewById(R.id.imgPinDot);

        setupDeletePress(imgFocusedPinInput, txtNextFocusedPinInput, imgNextFocusedPinInput, imgNextFocusedPinDot);
    }

    /**
     * Returns current user input
     *
     * @return current input
     */
    @NonNull
    public SecureString getInputSecureString() {
        return new SecureString(getInputCharArray());
    }

    /**
     * Returns current user input
     *
     * @return current input
     */
    @NonNull
    public char[] getInputCharArray() {
        char[] input = new char[cursorPosition];

        for (int i = 0; i < cursorPosition; i++) {
            TextView txtPinInput = getChildAt(i).findViewById(R.id.txtPinInput);
            input[i] = txtPinInput.getText().charAt(0);
        }

        return input;
    }

    /**
     * Converts the @{@link CharSequence} to a char using the first character, throwing
     * an exception on empty CharSequence.
     *
     * @param charSequence the character to convert
     * @return the char value of the first letter of the CharSequence
     * @throws IllegalArgumentException if the CharSequence is empty
     */
    private static char toChar(@NonNull CharSequence charSequence) {
        if (TextUtils.isEmpty(charSequence)) {
            throw new IllegalArgumentException("The character must not be null!");
        }
        return charSequence.charAt(0);
    }

    /**
     * Returns current user input
     *
     * @return current input
     */
    @Deprecated
    @NonNull
    public String[] getInputStringArray() {
        String[] input = new String[cursorPosition];

        for (int i = 0; i < cursorPosition; i++) {
            TextView txtPinInput = getChildAt(i).findViewById(R.id.txtPinInput);
            input[i] = txtPinInput.getText().toString();
        }

        return input;
    }

    /**
     * Returns current user input
     *
     * @return current input
     */
    @Deprecated
    @NonNull
    public String getInputString() {
        StringBuilder builder = new StringBuilder();

        for (int i = 0; i < cursorPosition; i++) {
            TextView txtPinInput = getChildAt(i).findViewById(R.id.txtPinInput);
            builder.append(txtPinInput.getText());
        }

        return builder.toString();
    }

    /**
     * Sets the length of the pin code.
     *
     * @param pinLength the length of the pin code
     */
    public SensitiveUserInputView setPinLength(@IntRange(from = 1, to = Integer.MAX_VALUE) int pinLength) {
        this.digits = pinLength;
        return this;
    }

    /**
     * Shows input to user
     */
    public void showInput() {
        this.pinNumberVisibility = VISIBLE_PIN;

        if (cursorPosition <= 0) {
            return;
        }

        endAnimations();
        if (showHideAnimatorSet != null) {
            showHideAnimatorSet.end();
        }

        ObjectAnimator[] animatorsDot = new ObjectAnimator[cursorPosition];
        ObjectAnimator[] animatorsText = new ObjectAnimator[cursorPosition];
        for (int i = 0; i < cursorPosition; i++) {
            View pinRootView = getChildAt(i);

            ImageView imgPinDot = pinRootView.findViewById(R.id.imgPinDot);
            animatorsDot[i] = ObjectAnimator.ofFloat(imgPinDot, View.ALPHA, 1, 0).setDuration(300);

            final TextView txtPinInput = pinRootView.findViewById(R.id.txtPinInput);
            animatorsText[i] = ObjectAnimator.ofFloat(txtPinInput, View.ALPHA, 0, 1).setDuration(800);
            animatorsText[i].addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationStart(Animator animation) {
                    super.onAnimationStart(animation);
                    txtPinInput.setVisibility(VISIBLE);
                }
            });
        }

        showHideAnimatorSet = new AnimatorSet();
        showHideAnimatorSet.playTogether(animatorsDot);
        showHideAnimatorSet.playTogether(animatorsText);

        if (cursorPosition < digits) {
            setUpCursor();
        }

        showHideAnimatorSet.start();
    }

    /**
     * Hides input to user
     */
    public void hideInput() {
        this.pinNumberVisibility = HIDDEN_PIN;

        if (cursorPosition <= 0) {
            return;
        }

        endAnimations();
        if (showHideAnimatorSet != null) {
            showHideAnimatorSet.end();
        }

        ObjectAnimator[] animatorsDot = new ObjectAnimator[cursorPosition];
        ObjectAnimator[] animatorsText = new ObjectAnimator[cursorPosition];
        for (int i = 0; i < cursorPosition; i++) {
            View pinRootView = getChildAt(i);

            final ImageView imgPinDot = pinRootView.findViewById(R.id.imgPinDot);
            animatorsDot[i] = ObjectAnimator.ofFloat(imgPinDot, View.ALPHA, 0, 1).setDuration(500);
            animatorsDot[i].addListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationStart(Animator animation) {
                    super.onAnimationStart(animation);
                    imgPinDot.setVisibility(VISIBLE);
                }
            });

            TextView txtPinInput = pinRootView.findViewById(R.id.txtPinInput);
            animatorsText[i] = ObjectAnimator.ofFloat(txtPinInput, View.ALPHA, 1, 0).setDuration(300);
        }

        showHideAnimatorSet = new AnimatorSet();
        showHideAnimatorSet.playTogether(animatorsDot);
        showHideAnimatorSet.playTogether(animatorsText);

        if (cursorPosition < digits) {
            setUpCursor();
        }

        showHideAnimatorSet.start();
    }

    /**
     * Set text to the @{@link SensitiveUserInputView} view
     *
     * @param input               pin to be displayed
     * @param pinNumberVisibility One of {@link #VISIBLE_PIN}, {@link #HIDDEN_PIN}
     */
    public void setupInput(@NonNull char[] input, @PinNumberVisibility int pinNumberVisibility) {
        int inputLength = input.length;
        if (inputLength > digits) {
            throw new IllegalArgumentException("Input length is out of bounds! Maximum is " + digits);
        }

        this.pinNumberVisibility = pinNumberVisibility;

        endAnimations();
        boolean defaultPinFocusColor = pinFocusColor == AttrHelper.VALUE_NOT_SET;
        for (int i = 0; i < inputLength; i++) {
            View pinRootView = getChildAt(i);
            TextView txtPinInput = pinRootView.findViewById(R.id.txtPinInput);
            ImageView imgPinDot = pinRootView.findViewById(R.id.imgPinDot);
            ImageView imgPinInput = pinRootView.findViewById(R.id.imgPinInput);

            setText(input[i], txtPinInput);

            if (i == inputLength - 1) {
                cursorPosition = inputLength;
                if (cursorPosition < digits) {
                    setUpCursor();
                }
            }

            if (!defaultPinFocusColor) {
                setupVectorDrawableColor(getContext(), imgPinInput, pinFocusColor);
            }

            imgPinInput.setAlpha(1f);

            if (pinNumberVisibility == VISIBLE_PIN) {
                txtPinInput.setVisibility(VISIBLE);
            } else {
                imgPinDot.setVisibility(VISIBLE);
            }
        }

        if (inputLength == 0) {
            setUpCursor();
        }
    }

    /**
     * Delete current input and set proper images
     */
    public void resetInput() {
        if (cursorPosition <= 0) {
            return;
        }

        endAnimations();
        if (cursorPosition <= 0) {
            return;
        }

        boolean defaultPinFocusColor = pinFocusColor == AttrHelper.VALUE_NOT_SET,
                defaultRemainingDigitsColor = remainingDigitsColor == AttrHelper.VALUE_NOT_SET;

        Context context = getContext();
        ObjectAnimator[] animators = new ObjectAnimator[cursorPosition];
        int lineLength = cursorPosition + (cursorPosition < digits ? 1 : 0);
        ObjectAnimator[] animatorsLineAlpha = new ObjectAnimator[lineLength];
        for (int i = 0; i <= cursorPosition; i++) {
            if (i == digits) {
                break;
            }

            View pinRootView = getChildAt(i);

            final TextView txtPinInput = pinRootView.findViewById(R.id.txtPinInput);
            final ImageView imgPinDot = pinRootView.findViewById(R.id.imgPinDot);
            ImageView imgPinInput = pinRootView.findViewById(R.id.imgPinInput);

            if (i < cursorPosition) {
                if (pinNumberVisibility == VISIBLE_PIN) {
                    animators[i] = ObjectAnimator.ofFloat(txtPinInput, View.ALPHA, 1, 0).setDuration(300);
                } else {
                    animators[i] = ObjectAnimator.ofFloat(imgPinDot, View.ALPHA, 1, 0).setDuration(300);
                }

                animators[i].addListener(new AnimatorListenerAdapter() {
                    @Override
                    public void onAnimationEnd(Animator animation) {
                        super.onAnimationEnd(animation);
                        setText(' ', txtPinInput);
                        txtPinInput.setVisibility(INVISIBLE);
                        imgPinDot.setVisibility(INVISIBLE);
                    }
                });
            }

            if (!defaultRemainingDigitsColor && i != 0) {
                setupVectorDrawableColor(context, imgPinInput, remainingDigitsColor);
            } else if (!defaultPinFocusColor) {
                setupVectorDrawableColor(context, imgPinInput, pinFocusColor);
            }
            animatorsLineAlpha[i] = ObjectAnimator.ofFloat(imgPinInput, View.ALPHA, remainingDigitsAlpha).setDuration(300);
        }

        this.cursorPosition = 0;

        setUpCursor();

        resetAnimatorSet = new AnimatorSet();
        resetAnimatorSet.playTogether(animators);
        resetAnimatorSet.playTogether(animatorsLineAlpha);

        resetAnimatorSet.start();
    }

    /**
     * Set text to the @{@link SensitiveUserInputView} view
     *
     * @param input pin to be displayed
     */
    public void setupInput(@NonNull char[] input) {
        setupInput(input, HIDDEN_PIN);
    }

    /**
     * Apply the set changes
     */
    public void apply() {
        setupPinDigitsView(getContext());
        setupTextAndFocusColor();
        if (cursorPosition < digits) {
            setUpCursor();
        }
    }

    //#############
    //#### Getter & setters
    //#############

    /**
     * @return current pin entered listener @{@link OnPinEnteredListener}
     */
    public OnPinEnteredListener getPinEnteredListener() {
        return pinEnteredListener;
    }

    /**
     * Sets @{@link OnPinEnteredListener} listener
     *
     * @param pinEnteredListener @{@link OnPinEnteredListener}
     */
    public SensitiveUserInputView setPinEnteredListener(OnPinEnteredListener pinEnteredListener) {
        this.pinEnteredListener = pinEnteredListener;
        return this;
    }

    /**
     * Returns the PIN visibility status for this view.
     *
     * @return One of {@link #VISIBLE_PIN}, {@link #HIDDEN_PIN}
     */
    @PinNumberVisibility
    public int getPinNumberVisibility() {
        return pinNumberVisibility;
    }

    public SensitiveUserInputView setPinFocusColor(@ColorInt int pinFocusColor) {
        this.pinFocusColor = pinFocusColor;
        return this;
    }

    public SensitiveUserInputView setPinDotColor(@ColorInt int pinDotColor) {
        this.pinDotColor = pinDotColor;
        return this;
    }

    public SensitiveUserInputView setPinTextColor(@ColorInt int pinTextColor) {
        this.pinTextColor = pinTextColor;
        return this;
    }

    /**
     * Set the color of the remaining digits
     *
     * @param remainingDigitsColor AARRGGBB See {@link SensitiveUserInputView#remainingDigitsColor} XML attr
     */
    public SensitiveUserInputView setRemainingDigitsColor(@ColorInt int remainingDigitsColor) {
        this.remainingDigitsColor = remainingDigitsColor;
        return this;
    }

    public SensitiveUserInputView setRemainingDigitsAlpha(@FloatRange(from = 0.0, to = 1.0) float remainingDigitsAlpha) {
        this.remainingDigitsAlpha = remainingDigitsAlpha;
        return this;
    }

    //#############
    //#### Saving state
    //#############

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        Parcelable parcelable = super.onSaveInstanceState();
        SavedState savedState = new SavedState(parcelable);

        savedState.pinNumberVisibility = pinNumberVisibility;
        savedState.pin = getInputCharArray();

        return savedState;
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        SavedState savedState = (SavedState) state;
        super.onRestoreInstanceState(savedState.getSuperState());

        setupInput(savedState.pin, savedState.pinNumberVisibility);
    }

    /**
     * Save state of the @{@link SensitiveUserInputView} view
     */
    private static class SavedState extends BaseSavedState {

        @PinNumberVisibility
        private int pinNumberVisibility;
        private char[] pin;

        private SavedState(Parcel source) {
            super(source);
            pinNumberVisibility = source.readInt();
            pin = source.createCharArray();
        }

        private SavedState(Parcelable superState) {
            super(superState);
        }

        @Override
        public void writeToParcel(Parcel out, int flags) {
            super.writeToParcel(out, flags);
            out.writeInt(pinNumberVisibility);
            out.writeCharArray(pin);
        }

        public static final Parcelable.Creator<SavedState> CREATOR
                = new Parcelable.Creator<SavedState>() {
            public SavedState createFromParcel(Parcel in) {
                return new SavedState(in);
            }

            public SavedState[] newArray(int size) {
                return new SavedState[size];
            }
        };
    }
}
