/* Copyright 2018 AHEAD iTec, s.r.o

   Permission to use, copy, modify, and/or distribute this software
   for any purpose with or without fee is hereby granted, provided
   that the above copyright notice and this permission notice appear
   in all copies.

   There is NO WARRANTY for this software.  See LICENSE for
   details. */
package com.aheaditec.sensitiveuserinputview;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import java.util.Arrays;

/**
 * @author Milan Jiricek <milan.jiricek@ahead-itec.com>
 */
public class SecureString implements CharSequence, Parcelable {

    private final char[] value;

    /**
     * Constructs a new SecureString which controls the passed in char array.
     *
     * @param value char[]
     */
    public SecureString(@NonNull char[] value) {
        if (value == null) {
            throw new IllegalArgumentException("Value must not be null!");
        }
        this.value = value;
    }

    private SecureString(@NonNull char[] value, int start, int end) {
        if (value == null) {
            throw new IllegalArgumentException("Value must not be null!");
        }
        this.value = new char[end - start + 1];
        System.arraycopy(value, start, this.value, 0, this.value.length);
    }

    /**
     * @return the length of the sequence of characters represented by this
     * object.
     */
    @Override
    public int length() {
        return value.length;
    }

    /**
     * @param index the index of the {@code char} value.
     * @return the {@code char} value at the specified index of this object.
     * @throws ArrayIndexOutOfBoundsException if the {@code index}
     *                                        argument is negative or not less than the length of this
     *                                        secure string.
     */
    @Override
    public char charAt(int index) {
        return value[index];
    }

    /**
     * Returns a character sequence that is a subsequence of this sequence.
     *
     * @param start the begin index, inclusive.
     * @param end   the end index, exclusive.
     * @return the specified subsequence.
     * @throws NegativeArraySizeException     if {@code beginIndex} is greater than {@code endIndex}
     * @throws ArrayIndexOutOfBoundsException if {@code beginIndex} or {@code endIndex} is negative,
     *                                        if {@code endIndex} is greater than {@code length()}
     */
    @Override
    public SecureString subSequence(int start, int end) {
        return new SecureString(this.value, start, end);
    }

    //#############
    //#### Help methods
    //#############

    /**
     * Manually clear the underlying char array.
     */
    public void clear() {
        Arrays.fill(value, '0');
    }

    /**
     * Converts this secure string to a new character array.
     *
     * @return a newly allocated character array whose length is the length
     * of this secure string and whose contents are initialized to contain
     * the character sequence represented by this secure string.
     */
    public char[] toCharArray() {
        return Arrays.copyOf(value, value.length);
    }

    /**
     * This is a dangerous operation as the array may be modified while it is being used by other threads
     * or a consumer may modify the values in the array. For safety, it is preferable to use {@link #toCharArray()}.
     *
     * @return the underlying char array.
     */
    public char[] getChars() {
        return value;
    }

    //#############
    //#### Parcelable implementation
    //#############

    private SecureString(Parcel in) {
        value = in.createCharArray();
    }

    public static final Creator<SecureString> CREATOR = new Creator<SecureString>() {
        @Override
        public SecureString createFromParcel(Parcel in) {
            return new SecureString(in);
        }

        @Override
        public SecureString[] newArray(int size) {
            return new SecureString[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeCharArray(value);
    }
}
