package com.adups.trace;

import android.os.Environment;
import android.util.Log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 日志功能：<br/>
 * 日志开关和级别
 * 显示当前打印日志代码位置
 * 将打印的日志写入文件，不需要打印日志代码位置
 * 日志文件可配置路径
 * 日志文件可共享
 * Created by raise.yang on 2016/08/03.
 */
public class Trace {

    public static final int VERBOSE = 2;
    public static final int DEBUG = 3;
    public static final int INFO = 4;
    public static final int WARN = 5;
    public static final int ERROR = 6;
    public static final int NONE = 7; // 关闭日志

    private static int s_level;//日志级别
    private static boolean s_show_code_position; //是否显示 调用位置
    private static boolean s_write_file; //是否写文件

    private static String m_log_path;
    private static int s_log_size = 500;

    static {
        s_level = NONE;
        s_show_code_position = false;
        s_write_file = true;
        m_log_path = Environment.getExternalStorageDirectory() + "/iport_log.txt";
    }

    private static void println(int level, String tag, String message) {
        if (level >= s_level) {//过滤级别
            if (s_show_code_position) {
                message += getCodePosition();
            }
            printAndroidLog(level, "iport/"+tag, message);
            if (s_write_file) {
                write_log(tag, message);
            }
        }
    }

    private static void printAndroidLog(int level, String tag, String message) {
        switch (level) {
            case VERBOSE:
                Log.v(tag, message);
                break;
            case DEBUG:
                Log.d(tag, message);
                break;
            case INFO:
                Log.i(tag, message);
                break;
            case WARN:
                Log.w(tag, message);
                break;
            case ERROR:
                Log.e(tag, message);
                break;
        }
    }

    public static void setLevel(int level) {
        s_level = level;
    }

    /**
     * 是否写日志文件
     *
     * @param write_file
     */
    public static void write_file(boolean write_file) {
        s_write_file = write_file;
    }

    /**
     * 日志文件大小，单位：KB,超过此大小会重新读写文件,默认值 500
     */
    public static void setLog_size(int log_size) {
        s_log_size = log_size;
    }

    /**
     * 设置日志保存路径
     *
     * @param m_log_path
     */
    public static void setLog_path(String m_log_path) {
        Trace.m_log_path = m_log_path;
    }

    public static void v(String tag, String msg) {
        println(2, tag, msg);
    }

    public static void v(String tag, String msg, Throwable tr) {
        println(2, tag, msg + '\n' + getStackTraceString(tr));
    }


    public static void d(String tag, String msg) {
        println(3, tag, msg);
    }

    public static void d(String tag, String msg, Throwable tr) {
        println(3, tag, msg + '\n' + getStackTraceString(tr));
    }

    public static void i(String tag, String msg) {
        println(4, tag, msg);
    }

    public static void i(String tag, String msg, Throwable tr) {
        println(4, tag, msg + '\n' + getStackTraceString(tr));
    }

    public static void w(String tag, String msg) {
        println(5, tag, msg);
    }

    public static void w(String tag, String msg, Throwable tr) {
        println(5, tag, msg + '\n' + getStackTraceString(tr));
    }

    public static void w(String tag, Throwable tr) {
        println(5, tag, getStackTraceString(tr));
    }

    public static void e(String tag, String msg) {
        println(6, tag, msg);
    }

    public static void e(String tag, String msg, Throwable tr) {
        println(6, tag, msg + '\n' + getStackTraceString(tr));
    }

    public static void e(String tag, Throwable tr) {
        println(6, tag, getStackTraceString(tr));
    }


    /**
     * Handy function to get a loggable stack trace from a Throwable
     *
     * @param tr An exception to log
     */
    private static String getStackTraceString(Throwable tr) {
        if (tr == null) {
            return "";
        }

        // This is to reduce the amount of log spew that apps do in the non-error
        // condition of the network being unavailable.
        Throwable t = tr;
        while (t != null) {
            if (t instanceof UnknownHostException) {
                return "";
            }
            t = t.getCause();
        }

        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        tr.printStackTrace(pw);
        pw.flush();
        return sw.toString();
    }

    /**
     * 显示当前打印日志代码位置
     *
     * @return
     */
    private static String getCodePosition() {
        StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();

        int index = 5;
        String className = stackTrace[index].getFileName();
        String methodName = stackTrace[index].getMethodName();
        int lineNumber = stackTrace[index].getLineNumber();
//       return " at "+stackTrace[index].toString();
        return String.format(".(%s:%s) %s()", className, lineNumber, methodName);
    }

    private static boolean check_log_file() {
        File log_file = new File(m_log_path);
        if (!log_file.exists()) {
            if (!log_file.getParentFile().exists()) {
                log_file.getParentFile().mkdirs();//父目录不存在创建
            }
            try {
                log_file.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        } else {
            // 日志文件备份
            if (log_file.length() > 1024 * s_log_size)
                log_file.renameTo(new File(m_log_path + "(1)"));
        }
        return true;
    }

    private static void write_log(String tag, String msg) {
        String text = getFormatLog(tag, msg);
        File log_file = new File(m_log_path);
        if (!check_log_file()) return;

        FileOutputStream fos = null;
        try {
            boolean append = log_file.length() > 1024 * s_log_size ? false : true;
            fos = new FileOutputStream(log_file, append);
            fos.write(text.getBytes());
            fos.write("\n".getBytes());
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (fos != null)
                    fos.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private static String getFormatLog(String tag, String msg) {
        SimpleDateFormat sdf = new SimpleDateFormat("MM-dd HH:mm:ss.SSS");
        String dateStr = sdf.format(new Date());
        return new StringBuilder()
                .append(dateStr)
                .append(" ")
                .append(String.format("%s ", convertThreadId((int) Thread.currentThread().getId())))
                .append(String.format("%s: ", tag))
                .append(msg).toString();
    }

    private static String convertThreadId(int value) {
        int limit = 5;
        String src = String.valueOf(value);
        int i = limit - src.length();
        if (i < 0) {
            src = src.substring(-i, src.length());
        }
        for (; i > 0; i--) {
            src = "0" + src;
        }
        return src;
    }
}
