package com.adups.http_libs;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;

/**
 * Created by fighter_lee on 2017/7/19.
 */

public abstract class HttpListener {

    private static final String TAG = HttpListener.class.getSimpleName();

    private static final int M_START = 1;
    private static final int M_SUCCESS = 2;
    private static final int M_FAILURE = 3;
    private static final int M_READING = 5;
    private static final int M_UPLOADING = 6;
    private static final int M_RETRY = 7;
    private static final int M_REDIRECT = 8;
    private static final int M_END = 9;

    private HttpHandler handler;
    private boolean runOnUiThread = true;
    private boolean readingNotify = false;
    private boolean uploadingNotify = false;
    private HttpListener linkedListener;
    private long delayMillis;

    /**
     * default run on UI thread
     */
    public HttpListener() {
        this(true);
    }

    public HttpListener(long delayMillis) {
        this.delayMillis = delayMillis;
    }

    public HttpListener(boolean runOnUiThread) {
        setRunOnUiThread(runOnUiThread);
    }

    public HttpListener(boolean runOnUiThread, boolean readingNotify, boolean uploadingNotify) {
        this(runOnUiThread);
        this.readingNotify = readingNotify;
        this.uploadingNotify = uploadingNotify;
    }

    public final HttpListener getLinkedListener() {
        return linkedListener;
    }

    public final HttpListener setLinkedListener(HttpListener httpListener) {
        if (this.linkedListener != null) {
            HttpListener temp = this.linkedListener;
            do {
                if (httpListener == temp) {
                    throw new RuntimeException("Circular refrence:  " + httpListener);
                }
            } while ((temp = temp.getLinkedListener()) != null);
        }
        this.linkedListener = httpListener;
        return this;
    }

    public final boolean isRunOnUiThread() {
        return runOnUiThread;
    }

    public final HttpListener setRunOnUiThread(boolean runOnUiThread) {
        this.runOnUiThread = runOnUiThread;
        if (runOnUiThread) {
            handler = new HttpHandler(Looper.getMainLooper());
        } else {
            handler = null;
        }
        return this;
    }

    public final boolean isReadingNotify() {
        return readingNotify;
    }

    public final HttpListener setReadingNotify(boolean readingNotify) {
        this.readingNotify = readingNotify;
        return this;
    }

    public final boolean isUploadingNotify() {
        return uploadingNotify;
    }

    public final HttpListener setUploadingNotify(boolean uploadingNotify) {
        this.uploadingNotify = uploadingNotify;
        return this;
    }

    public long getDelayMillis() {
        return delayMillis;
    }

    public HttpListener setDelayMillis(long delayMillis) {
        this.delayMillis = delayMillis;
        return this;
    }

    /**
     * note: hold an implicit reference to outter class
     */
    private class HttpHandler extends Handler {
        private HttpHandler(Looper looper) {
            super(looper);
        }

        @Override
        @SuppressWarnings("unchecked")
        public void handleMessage(Message msg) {
            if (disableListener()) {
                return;
            }
            Object[] data;
            switch (msg.what) {
                case M_START:
                    onStart((AbstractRequest) msg.obj);
                    break;
                case M_SUCCESS:
                    data = (Object[]) msg.obj;
                    onSuccess((Data) data[0], (Response) data[1]);
                    break;
                case M_FAILURE:
                    data = (Object[]) msg.obj;
                    onFailure((HttpException) data[0], (Response) data[1]);
                    break;
                case M_READING:
                    data = (Object[]) msg.obj;
                    onLoading((AbstractRequest) data[0], (Long) data[1], (Long) data[2]);
                    break;
                case M_UPLOADING:
                    data = (Object[]) msg.obj;
                    onUploading((AbstractRequest) data[0], (Long) data[1], (Long) data[2]);
                    break;
                case M_RETRY:
                    data = (Object[]) msg.obj;
                    onRetry((AbstractRequest) data[0], (Integer) data[1], (Integer) data[2]);
                    break;
                case M_REDIRECT:
                    data = (Object[]) msg.obj;
                    onRedirect((AbstractRequest) data[0], (Integer) data[1], (Integer) data[2]);
                    break;
                case M_END:
                    onEnd((Response) msg.obj);
                    break;
            }
        }
    }

    //____________lite called method ____________
    public final void notifyCallStart(AbstractRequest req) {
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_START);
            msg.obj = req;
            handler.sendMessage(msg);
        } else {
            onStart(req);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallStart(req);
        }
    }

    public final void notifyCallSuccess(Data data, Response response) {
        delayOrNot();
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_SUCCESS);
            msg.obj = new Object[]{data, response};
            handler.sendMessage(msg);
        } else {
            onSuccess(data, response);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallSuccess(data, response);
        }
    }

    public final void notifyCallFailure(HttpException e, Response response) {
        delayOrNot();
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_FAILURE);
            msg.obj = new Object[]{e, response};
            handler.sendMessage(msg);
        } else {
            onFailure(e, response);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallFailure(e, response);
        }
    }


    public final void notifyCallLoading(AbstractRequest req, long total, long len) {
        if (disableListener()) {
            return;
        }
        if (readingNotify) {
            if (runOnUiThread) {
                Message msg = handler.obtainMessage(M_READING);
                msg.obj = new Object[]{req, total, len};
                handler.sendMessage(msg);
            } else {
                onLoading(req, total, len);
            }
        }
        if (linkedListener != null) {
            linkedListener.notifyCallLoading(req, total, len);
        }
    }

    public final void notifyCallUploading(AbstractRequest req, long total, long len) {
        if (disableListener()) {
            return;
        }
        if (uploadingNotify) {
            if (runOnUiThread) {
                Message msg = handler.obtainMessage(M_UPLOADING);
                msg.obj = new Object[]{req, total, len};
                handler.sendMessage(msg);
            } else {
                onUploading(req, total, len);
            }
        }
        if (linkedListener != null) {
            linkedListener.notifyCallUploading(req, total, len);
        }
    }

    public final void notifyCallRetry(AbstractRequest req, int max, int times) {
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_RETRY);
            msg.obj = new Object[]{req, max, times};
            handler.sendMessage(msg);
        } else {
            onRetry(req, max, times);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallRetry(req, max, times);
        }
    }

    public final void notifyCallRedirect(AbstractRequest req, int max, int times) {
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_REDIRECT);
            msg.obj = new Object[]{req, max, times};
            handler.sendMessage(msg);
        } else {
            onRedirect(req, max, times);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallRedirect(req, max, times);
        }
    }

    public final void notifyCallEnd(Response response) {
        if (disableListener()) {
            return;
        }
        if (runOnUiThread) {
            Message msg = handler.obtainMessage(M_END);
            msg.obj = response;
            handler.sendMessage(msg);
        } else {
            onEnd(response);
        }
        if (linkedListener != null) {
            linkedListener.notifyCallEnd(response);
        }
    }

    private boolean delayOrNot() {
        if (delayMillis > 0) {
            try {
                Thread.sleep(delayMillis);
                return true;
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        return false;
    }

    //____________ developer override method ____________
    public boolean disableListener() {
        return false;
    }

    public void onStart(AbstractRequest request) {}

    public void onSuccess(Data data, Response response) {}

    public void onFailure(HttpException e, Response response) {}

    public void onLoading(AbstractRequest request, long total, long len) {}

    public void onUploading(AbstractRequest request, long total, long len) {}

    public void onRetry(AbstractRequest request, int max, int times) {}

    public void onRedirect(AbstractRequest request, int max, int times) {}

    public void onEnd(Response response) {}

}
