package com.adpdigital.push;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.media.AudioAttributes;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

@RequiresApi(api = Build.VERSION_CODES.O)
public class ChabokNotificationChannel {

    //region Properties

    private String sound;
    private Context context;
    private String channelId;
    private String channelName;
    private int lightColor = -1;
    private int importance = NotificationManager.IMPORTANCE_HIGH;

    public Context getContext() {
        return context;
    }

    public ChabokNotificationChannel setContext(Context context) {
        this.context = context;

        return this;
    }

    public String getChannelId() {
        return channelId;
    }

    public ChabokNotificationChannel setChannelId(String channelId) {
        this.channelId = channelId;

        return this;
    }

    public String getSound() {
        return sound;
    }

    public ChabokNotificationChannel setSound(String sound) {
        this.sound = sound;

        return this;
    }

    public int getLightColor() {
        return lightColor;
    }

    public ChabokNotificationChannel setLightColor(int lightColor) {
        this.lightColor = lightColor;

        return this;
    }

    public int getImportance() {
        return importance;
    }

    public ChabokNotificationChannel setImportance(int importance) {
        this.importance = importance;

        return this;
    }

    public String getChannelName() {
        return channelName;
    }

    public ChabokNotificationChannel setChannelName(String channelName) {
        this.channelName = channelName;

        return this;
    }

    //endregion

    private ChabokNotificationChannel() {

    }

    public ChabokNotificationChannel(String channelId, Context context) {
        this.context = context;
        this.setChannelId(channelId);
    }

    private NotificationManager getNotificationManager(Context context) {
        return (NotificationManager) context
                .getSystemService(Context.NOTIFICATION_SERVICE);
    }

    @Nullable
    public NotificationChannel build() {
        return createNotificationChannel(
                this.getChannelId(),
                this.getChannelName(),
                this.getImportance(),
                this.getSound(),
                null,
                this.getLightColor()
        );
    }

    private Uri getDefaultNotificationSoundUri() {
        return RingtoneManager.getActualDefaultRingtoneUri(
                this.getContext(),
                RingtoneManager.TYPE_NOTIFICATION
        );
    }

    @Nullable
    private NotificationChannel createNotificationChannel(String channelId,
                                                          String channelName,
                                                          int importanceLevel,
                                                          String soundName,
                                                          long[] vibrationPattern,
                                                          int lightColor) {
        NotificationManager notificationManager = (NotificationManager) this.getContext()
                .getSystemService(Context.NOTIFICATION_SERVICE);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            NotificationChannel notificationChannel = new NotificationChannel(
                    channelId,
                    channelName.substring(0, 1).toUpperCase() + channelName.substring(1),
                    importanceLevel);

            Uri soundUri = getDefaultNotificationSoundUri();
            AudioAttributes audioAttributes = getAudioAttributesBuilder();
            if (soundName != null && DrawableUtil.isResourceFileExists(soundName, "raw", context)) {
                String soundPathStr = "android.resource://" + context.getPackageName() + "/raw/" +
                        soundName;
                soundUri = Uri.parse(soundPathStr);
            }
            if (audioAttributes != null) {
                if (soundName != null && (soundName.contentEquals("null") || soundName.contentEquals("nil"))){
                    notificationChannel.setSound(null, null);
                } else {
                    notificationChannel.setSound(soundUri, audioAttributes);
                }
            }

            if (vibrationPattern != null) {
                notificationChannel.enableVibration(true);
                notificationChannel.setVibrationPattern(vibrationPattern);
            }

            if (lightColor != -1) {
                notificationChannel.enableLights(true);
                notificationChannel.setLightColor(lightColor);
            }

            if (notificationManager != null) {
                notificationManager.createNotificationChannel(notificationChannel);
            }

            return notificationChannel;
        } else {
            Logger.d(Constants.TAG, "Couldn't create notification channel, because is require api level 26");
        }

        return null;
    }

    @Nullable
    private AudioAttributes getAudioAttributesBuilder() {
        AudioAttributes.Builder builder;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            builder = new AudioAttributes.Builder();

            builder.setUsage(AudioAttributes.USAGE_NOTIFICATION);
            builder.setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION);

            return builder.build();
        }

        return null;
    }
}
