package com.network;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;

import com.adpdigital.push.ChabokCrypto;
import com.adpdigital.push.ChabokLocalStorage;
import com.adpdigital.push.Logger;
import com.adpdigital.push.SecureString;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.HashMap;
import java.util.TimeZone;

public class InstallationModel {

    public static final String DEVICE_TYPE_ANDROID = "android";
    public static final String STATUS_ACTIVE = "Active";

    private static final String CHK_AD_ID = "CHK_AD_ID";
    private static final String PROPERTY_APP_VERSION = "appVersion";
    public static final String PROPERTY_DEVICE_TOKEN = "deviceToken";
    public static final String PROPERTY_INSTALLATION_ID = "installationId";

    Object id;
    String appId;
    String appVersion;
    SecureString userId;
    SecureString deviceToken;
    String[] subscriptions;
    String timeZone;
    String status;
    String adId;
    boolean adIdEnabled;
    String tokenStatus;
    HashMap<String, Object> tokenErr;

    private final Context applicationContext;

    public InstallationModel(Context applicationContext) {

        this.applicationContext = applicationContext;
        fillDefaults();
        loadSharedPreferences();
    }

    /**
     * @return The Installation id as assigned by the LoopBack server.
     */
    public Object getId() {
        return id;
    }

    /**
     * See {@link InstallationModel#getId()}.
     */
    void setId(final Object id) {
        this.id = id;
    }

    /**
     * @return The id of the LoopBack Application object this Installation
     * is bound to.
     */
    public String getAppId() {
        return appId;
    }

    /**
     * See {@link InstallationModel#getAppId()}.
     */
    public void setAppId(final String appId) {
        this.appId = appId;
    }

    /**
     * @return The version of this Android application. The property is pre-filled
     * for you using the value in the Android manifest.
     */
    public String getAppVersion() {
        return appVersion;
    }

    /**
     * See {@link InstallationModel#getAppVersion()}.
     */
    public void setAppVersion(final String appVersion) {
        this.appVersion = appVersion;
    }

    /**
     * @return Id of the user logged in the Android application.
     */
    public SecureString getUserId() {
        return userId;
    }

    /**
     * See {@link InstallationModel#getUserId()}.
     */
    public void setUserId(final SecureString userId) {
        this.userId = userId;
    }

    /**
     * @return One of the device types supported by LoopBack Push Notification
     * plugin. This is a read-only property that returns
     * {@link #DEVICE_TYPE_ANDROID}
     */
    public String getDeviceType() {
        return DEVICE_TYPE_ANDROID;
    }

    /**
     * @return The GCM registration id that can be used by the LoopBack server
     * to push notifications to this application/device.
     */
    public SecureString getDeviceToken() {
        return deviceToken;
    }

    /**
     * See {@link InstallationModel#getDeviceToken()}.
     */
    public void setDeviceToken(final SecureString deviceToken) {
        this.deviceToken = deviceToken;
        saveDeviceToken();
    }

    public void setTokenStatus(final String tokenStatus) {
        this.tokenStatus = tokenStatus;
    }

    public String getTokenStatus() {
        return tokenStatus;
    }

    public void setTokenError(final String errorMessage, final int errorCode) {
        if (errorMessage == null){
            Logger.d(Logger.TAG, "errorMessage in tokenError is null....");
            return;
        }
        HashMap<String, Object> tokenErr = new HashMap<>();
        tokenErr.put("code", errorCode);
        tokenErr.put("message", errorMessage);

        this.tokenErr = tokenErr;
    }

    public HashMap<String, Object> getTokenErr() {
        return tokenErr;
    }

    /**
     * @return List subscriptions.
     */
    public String[] getSubscriptions() {
        return subscriptions;
    }

    /**
     * See {@link InstallationModel#getSubscriptions()}.
     */
    public void setSubscriptions(final String[] subscriptions) {
        this.subscriptions = subscriptions;
    }

    public void setAdId(String adId){
        this.adId = adId;
        getSharedPreferences().edit().putString(CHK_AD_ID, adId).apply();
    }

    public String getAdId(){
        return getSharedPreferences().getString(CHK_AD_ID, this.adId);
    }

    public void setAdIdEnabled(boolean adIdEnabled) {
        this.adIdEnabled = adIdEnabled;
    }

    public boolean isAdIdEnabled() {
        return adIdEnabled;
    }

    /**
     * @return ID of the time zone preferred by the user.
     * Example: America/Los_Angeles, Europe/London
     */
    public String getTimeZone() {
        return timeZone;
    }

    /**
     * See {@link InstallationModel#getTimeZone()}
     */
    public void setTimeZone(String timeZone) {
        this.timeZone = timeZone;
    }

    /**
     * @return Status of this installation, the value is one of the strings
     * recognized by LoopBack Push Notification plugin.
     * See also {@link #STATUS_ACTIVE}
     */
    public String getStatus() {
        return status;
    }

    /**
     * See {@link InstallationModel#getStatus()}.
     */
    public void setStatus(final String status) {
        this.status = status;
    }

    public HashMap<String, Object> getRequireParams() {
        HashMap<String, Object> params = new HashMap<>();

        if (this.getId() != null) {
            params.put("id", this.getId());
        }

        if (this.getAppId() != null){
            params.put("appId", this.getAppId());
        }

        if (this.getAdId() != null){
            params.put("adId", this.getAdId());
        }

        if (this.getAppVersion() != null){
            params.put("appVersion", this.getAppVersion());
        }

        if (this.getDeviceToken().asString() != null){
            params.put("deviceToken", this.getDeviceToken().asString());
        }

        if (this.getTokenErr() != null){
            params.put("tokenErr", this.getTokenErr());
        }

        if (this.getTokenStatus() != null){
            params.put("tokenStatus", this.getTokenStatus());
        }

        if (this.getDeviceType() != null){
            params.put("deviceType", this.getDeviceType());
        }

        if (this.getUserId().asString() != null){
            params.put("userId", this.getUserId().asString());
        }

        if (this.getSubscriptions() != null){
            params.put("subscriptions", this.getSubscriptions());
        }

        if (this.getTimeZone() != null){
            params.put("timeZone", this.getTimeZone());
        }

        if (this.getStatus() != null){
            params.put("status", this.getStatus());
        }

        params.put("adIdEnabled", adIdEnabled);

        return params;
    }

    private void fillDefaults() {
        try {
            setAppVersion(getPackageInfo().versionName);
        } catch (final PackageManager.NameNotFoundException e) {
            // This should never happen, as it would mean the currently
            // running application is not installed on the device
        }

        setStatus(InstallationModel.STATUS_ACTIVE);
        setTimeZone(TimeZone.getDefault().getID());
    }

    private void loadSharedPreferences() {
        final SharedPreferences prefs = getSharedPreferences();
//        AdpPushClient client = AdpPushClient.get();
        // Get the stored Installation ID. We are storing the value
        // in a JSON array to preserve it's type (NoSQL databases usually
        // use String, SQL databases use Number).

        final String idJson = ChabokCrypto.decrypt(applicationContext, prefs.getString(PROPERTY_INSTALLATION_ID, null));
        if (idJson != null) {
            try {
                id = new JSONArray(idJson).get(0);
            } catch (JSONException e) {
                String msg = "Cannot parse installation id '" + idJson + "'";
                Logger.e("LoopBack", msg, e);
            }
        }

        // Check if app was updated; if so, it must clear the device token
        // (the GCM registration ID) since the existing registration ID
        // is not guaranteed to work with the new app version.
        int savedVersionCode = prefs.getInt(PROPERTY_APP_VERSION, Integer.MIN_VALUE);
        int currentVersionCode = fetchAppVersionCode();
        if (savedVersionCode == currentVersionCode)
            deviceToken = SecureString.instance(prefs.getString(PROPERTY_DEVICE_TOKEN, null));
    }

    void saveInstallationId() {
        final SharedPreferences.Editor editor = getSharedPreferences().edit();
        final String json = new JSONArray().put(id).toString();
        editor.putString(PROPERTY_INSTALLATION_ID, ChabokCrypto.encrypt(applicationContext, json));
        editor.apply();
    }

    private void saveDeviceToken() {
        final SharedPreferences.Editor editor = getSharedPreferences().edit();
        editor.putString(PROPERTY_DEVICE_TOKEN, getDeviceToken().asString());
        editor.putInt(PROPERTY_APP_VERSION, fetchAppVersionCode());
        editor.apply();
    }

    private int fetchAppVersionCode() {
        try {
            return getPackageInfo().versionCode;
        } catch (final PackageManager.NameNotFoundException e) {
            // This should never happen, as it would mean the currently
            // running application is not installed on the device
            return Integer.MIN_VALUE;
        }
    }

    private PackageInfo getPackageInfo()
            throws PackageManager.NameNotFoundException {
        return applicationContext.getPackageManager()
                .getPackageInfo(applicationContext.getPackageName(), 0);
    }

    private SharedPreferences getSharedPreferences() {
        return ChabokLocalStorage.getSharedPreferences(applicationContext);
    }

}
