package com.adpdigital.push;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class JsonUtils {

    static List<Object> jsonArrayToList(JSONArray array) throws JSONException {
        List<Object> list = new ArrayList<>();
        for (int i = 0; i < array.length(); i++) {
            Object value = array.get(i);
            if (value instanceof JSONArray) {
                value = jsonArrayToList((JSONArray) value);
            } else if (value instanceof JSONObject) {
                value = MapUtils.jsonObjectToMap((JSONObject) value);
            }
            list.add(value);
        }
        return list;
    }

    /**
     * Converts a Java object to a JSON object. A {@link java.util.Map} is
     * converted to a {@link org.json.JSONObject}, a {@link java.util.List} or
     * array is converted to a {@link org.json.JSONArray},
     * and <code>null</code> is converted to  {@link org.json.JSONObject#NULL}.
     * Other objects, like {@link java.lang.Number}, {@link java.lang.String},
     * and {@link java.lang.Boolean} are returned without conversion.
     * @param object The object to convert.
     * @return a JSON object.
     * @throws JSONException If the object cannot be converted.
     */
    public static Object mapToJson(Object object) throws JSONException {
        if (object == null || object == JSONObject.NULL) {
            return JSONObject.NULL;
        }
        else if (object instanceof Map) {
            Map<?,?> map = ((Map<?,?>)object);
            JSONObject json = new JSONObject();
            for (Map.Entry<?,?> entry : map.entrySet()) {
                json.put(String.valueOf(entry.getKey()),
                        mapToJson(entry.getValue()));
            }
            return json;
        }
        else if (object instanceof Iterable) {
            JSONArray json = new JSONArray();
            for (Object value : (Iterable<?>)object) {
                json.put(mapToJson(value));
            }
            return json;
        }
        else if (object.getClass().isArray()) {
            JSONArray json = new JSONArray();
            int length = Array.getLength(object);
            for (int i = 0; i < length; i++) {
                json.put(mapToJson(Array.get(object, i)));
            }
            return json;
        }
        else if (object instanceof Number) {
            double d = ((Number)object).doubleValue();
            if (Double.isInfinite(d) || Double.isNaN(d)) {
                throw new JSONException("Numbers cannot be infinite or NaN.");
            }
            return object;
        }
        else if (object instanceof JSONObject ||
                object instanceof JSONArray ||
                object instanceof Boolean ||
                object instanceof String) {
            return object;
        }
        else {
            return object.toString();
        }
    }
}