package com.adpdigital.push;

import android.content.Context;
import android.content.SharedPreferences;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Set;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Comparator;

import static com.adpdigital.push.Logger.TAG;

public class ChabokLocalStorage {

    private static SharedPreferences sharedPreferencesInstance;

    private static final String NO_GCM_HEX = "6e6f47434d";
    private static final String USER_ID_HEX = "757365724964";
    private static final String PASSWORD_HEX = "70617373776f7264";
    private static final String USERNAME_HEX = "757365726e616d65";
    private static final String APPLICATION_ID_HEX = "6170706c69636174696f6e4964";
    private static final String RE_REGISTERED_FROM_HEX = "72657265676973746572656446726f6d";

    private static final String SHARED_PREF_NAME = "com.MRR.NZV.NZV.LMH";

    /*
    ⚠️ Note: We use the Loopback class name for creating SharedPreferences storage.
     When user use the dictionary in their ProGuard rules class name will be changed
      and for each version of applications storage changed and user data will be lost.
     */
    private static SharedPreferences getSharedPref(Context context) {
        if (sharedPreferencesInstance != null) {
            return sharedPreferencesInstance;
        }

        SharedPreferences sharedPreferences = getSharedPreferencesFromName(context, SHARED_PREF_NAME);
        if (isChabokPrefFile(sharedPreferences)) {
            Logger.d(TAG, "=======> Open last pref file.");
            sharedPreferencesInstance = sharedPreferences;
            return sharedPreferencesInstance;
        }

        File[] files;
        File prefsDir = new File(context.getApplicationInfo().dataDir, "shared_prefs");

        files = sortToLastModifiedFileChanges(prefsDir);

        sharedPreferencesInstance = getLastModifiedSharedPref(files, prefsDir, context);

        if (sharedPreferencesInstance == null) {
            Logger.d(TAG, "sharedPreferencesInstance is null");
            sharedPreferencesInstance = context.getSharedPreferences(
                    SHARED_PREF_NAME,
                    Context.MODE_PRIVATE
            );
        }

        Logger.d(TAG, "sharedPreferencesInstance keys = " + sharedPreferencesInstance.getAll().size());
        return sharedPreferencesInstance;
    }

    private static String readFile(File file) {
        //Read text from file
        StringBuilder text = new StringBuilder();

        try {
            BufferedReader br = new BufferedReader(new FileReader(file));
            String line;

            while ((line = br.readLine()) != null) {
                text.append(line);
                text.append('\n');
            }
            br.close();
        }
        catch (IOException e) {
            e.printStackTrace();
            Logger.e(TAG, "readFile: " + e.getMessage(), e);
        }

        return text.toString();
    }

    @Nullable
    private static File[] sortToLastModifiedFileChanges(@NonNull File prefsDir) {
        File[] files = null;

        if(prefsDir.exists() && prefsDir.isDirectory()) {
            files = prefsDir.listFiles();

            Arrays.sort(files,  new Comparator<File>() {
                public int compare(File f1, File f2) {
                    if (f1.lastModified() > f2.lastModified()) {
                        return -1;
                    }

                    if (f1.lastModified() < f2.lastModified()) {
                        return +1;
                    }

                    return 0;
                }
            });
        }
        return files;
    }

    @Nullable
    private static SharedPreferences getLastModifiedSharedPref(File[] files, File prefsDir, Context context){
        if (files == null) {
            return null;
        }

        for (File file : files) {
            String filename = file.getName();
            int pos = filename.lastIndexOf(".");
            String justName = pos > 0 ? filename.substring(0, pos) : filename;

            Logger.d(TAG, "=======> getSharedPref: " + justName + ". filename = " + filename);
            SharedPreferences sharedPreferences = getSharedPreferencesFromName(context, justName);
            if (isChabokPrefFile(sharedPreferences)) {
                Logger.d(TAG, "=======> isChabokPref");
                File changeFile = new File(prefsDir.getPath() + "/" + SHARED_PREF_NAME + ".xml");
                if (file.renameTo(changeFile)) {
                    Logger.d(TAG, "=======> Renamed to " + changeFile.getName());
                }
                return sharedPreferences;
            }
        }

        return null;
    }

    private static boolean isChabokPrefFile(SharedPreferences sharedPreferences){
        if (sharedPreferences != null &&
                sharedPreferences.contains("host") &&
                sharedPreferences.contains("applicationId") &&
                sharedPreferences.contains("AppLaunched") &&
                sharedPreferences.contains("userId") &&
                sharedPreferences.contains("port") &&
                sharedPreferences.contains("chabokInstallation")) {
            return true;
        }

        return false;
    }

    @Nullable
    private static SharedPreferences getSharedPreferencesFromName(Context context, String prefName) {
        return context.getSharedPreferences(
                prefName,
                Context.MODE_PRIVATE
        );
    }

    public static SharedPreferences getSharedPreferences(Context context) {
        return getSharedPref(context);
    }

    public static void setValueToStorage(Context context, SecureString value, String hexKey){
        String key = ChabokCrypto.convertHexToString(hexKey);
        String enc = ChabokCrypto.encrypt(context, value.asString());

        getSharedPreferences(context).edit().putString(key, enc).apply();
    }
    private static void removeKeyFromStorage(Context context, String hexKey) {
        String key = ChabokCrypto.convertHexToString(hexKey);
        getSharedPreferences(context).edit().remove(key).apply();
    }


    @Nullable
    public static SecureString getEncryptedValueFromStorage(@NonNull Context context, @NonNull String hexKey){
        String key = ChabokCrypto.convertHexToString(hexKey);
        String value = getSharedPreferences(context).getString(key, null);

        return SecureString.instance(ChabokCrypto.decrypt(context, value));
    }

    public static void setIsSupportPlayServices(Context context, boolean supportPlayServices){
        String key = ChabokCrypto.convertHexToString(NO_GCM_HEX);
        getSharedPreferences(context)
                .edit()
                .putBoolean(key, supportPlayServices)
                .apply();
    }
    public static boolean isSupportPlayServices(Context context){
        String key = ChabokCrypto.convertHexToString(NO_GCM_HEX);
        return getSharedPreferences(context).getBoolean(key, false);
    }

    public static void setUsername(Context context, String username){
        setValueToStorage(context, SecureString.instance(username), USERNAME_HEX);
    }
    public static String getUsername(Context context){
        return getEncryptedValueFromStorage(context, USERNAME_HEX).asString();
    }

    public static void setPassword(Context context, String password){
        setValueToStorage(context, SecureString.instance(password), PASSWORD_HEX);
    }
    public static String getPassword(Context context){
        return getEncryptedValueFromStorage(context, PASSWORD_HEX).asString();
    }

    public static void setUserId(Context context, SecureString userId){
        setValueToStorage(context, userId, USER_ID_HEX);
    }
    public static SecureString getUserId(Context context){
        return getEncryptedValueFromStorage(context, USER_ID_HEX);
    }
    public static void removeUserId(Context context){
        removeKeyFromStorage(context, USER_ID_HEX);
    }

    public static void setReregisteredFrom(Context context, SecureString userId){
        setValueToStorage(context, userId, RE_REGISTERED_FROM_HEX);
    }
    public static SecureString getReregisteredFrom(Context context){
        return getEncryptedValueFromStorage(context, RE_REGISTERED_FROM_HEX);
    }
    public static void removeReregisteredFrom(Context context){
        removeKeyFromStorage(context, RE_REGISTERED_FROM_HEX);
    }

    public static void setAppId(Context context, String appId){
        setValueToStorage(context, SecureString.instance(appId), APPLICATION_ID_HEX);
    }
    public static String getAppId(Context context){
        return getEncryptedValueFromStorage(context, APPLICATION_ID_HEX).asString();
    }
    public static void removeAppId(Context context){
        removeKeyFromStorage(context, APPLICATION_ID_HEX);
    }

    public static void setSubscriptionDirty(Context context, boolean isDirty){
        getSharedPreferences(context)
                .edit()
                .putBoolean("subscriptionDirty", isDirty)
                .apply();
    }
    public static boolean isSubscriptionDirty(Context context){
        return getSharedPreferences(context).getBoolean("subscriptionDirty", true);
    }

    public static void setTopics(Context context, Set<String> topics){
        getSharedPreferences(context).edit().putStringSet("topics", topics).apply();
    }
    public static Set<String> getTopics(Context context){
        return getSharedPreferences(context).getStringSet("topics", new HashSet<String>());
    }
    public static Set<String> getTopics(Context context, HashSet<String> defaultValue){
        return getSharedPreferences(context).getStringSet("topics", defaultValue);
    }
    public static void removeTopicsFromStorage(Context context){
        getSharedPreferences(context).edit().remove("topics").apply();
    }
}
