package com.adpdigital.push;

/**
 * Created by behrad on 5/17/15.
 * based on https://gist.github.com/steveliles/11116937
 */

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;


/**
 * Usage:
 *
 * 1. Get the ForegroundManager Singleton, passing a Context or Application object unless you
 * are sure that the Singleton has definitely already been initialised elsewhere.
 *
 * 2.a) Perform a direct, synchronous check: ForegroundManager.isForeground() / .isBackground()
 *
 * or
 *
 * 2.b) Register to be notified (useful in Service or other non-UI components):
 *
 *   ForegroundManager.AppListener myListener = new ForegroundManager.AppListener(){
 *       public void onBecameForeground(){
 *           // ... whatever you want to do
 *       }
 *       public void onBecameBackground(){
 *           // ... whatever you want to do
 *       }
 *   }
 *
 *   public void onCreate(){
 *      super.onCreate();
 *      ForegroundManager.get(this).addAppListener(listener);
 *   }
 *
 *   public void onDestroy(){
 *      super.onCreate();
 *      ForegroundManager.get(this).removeListener(listener);
 *   }
 */
public class ForegroundManager implements Application.ActivityLifecycleCallbacks {

    private static final String TAG = ForegroundManager.class.getName();
    private static String currentActivity = null;
    private static final long CHECK_DELAY = 500;

    private static ForegroundManager instance;

    private boolean foreground = false, paused = true;
    private Handler handler = new Handler();
    private List<AppListener> listeners = new CopyOnWriteArrayList<AppListener>();
    private Runnable check;

    /**
     * Its not strictly necessary to use this method - _usually_ invoking
     * get with a Context gives us a path to retrieve the Application and
     * initialise, but sometimes (e.g. in test harness) the ApplicationContext
     * is != the Application, and the docs make no guarantees.
     *
     * @param application
     * @return an initialised ForegroundManager instance
     */
    public static ForegroundManager init(Application application){
        if (instance == null) {
            instance = new ForegroundManager();
            application.registerActivityLifecycleCallbacks(instance);
        }
        return instance;
    }

    public static ForegroundManager get(Application application){
        if (instance == null) {
            init(application);
        }
        return instance;
    }

    public static ForegroundManager get(Context ctx){
        if (instance == null) {
            Context appCtx = ctx.getApplicationContext();
            if (appCtx instanceof Application) {
                return init((Application)appCtx);
            }
            throw new IllegalStateException(
                    "ForegroundManager is not initialised and " +
                            "cannot obtain the Application object");
        }
        return instance;
    }

    public static ForegroundManager get(){
        if (instance == null) {
            throw new IllegalStateException(
                    "ForegroundManager is not initialised - invoke " +
                            "at least once with parameterised init/get");
        }
        return instance;
    }

    public boolean isForeground(){
        return foreground;
    }

    public boolean isBackground(){
        return !foreground;
    }

    public void addListener(AppListener listener){
        listeners.add(listener);
    }

    public void removeListener(AppListener listener){
        listeners.remove(listener);
    }

    public void removeListeners() {
        listeners.clear();
    }

    public String getActiveActivityClassName() {
        return currentActivity;
    }

    @Override
    public void onActivityResumed(Activity activity) {
        paused = false;
        boolean wasBackground = !foreground;
        foreground = true;

        if (check != null)
            handler.removeCallbacks(check);

        if (wasBackground){
            Log.d(TAG, "app foreground");
            for (AppListener l : listeners) {
                try {
                    l.onBecameForeground( activity.getClass() );
                } catch (Exception exc) {
                    Log.e(TAG, "AppListener threw exception!", exc);
                }
            }
        } else {
            Log.d(TAG, "still foreground");
        }
        currentActivity = activity.getClass().getName();
    }

    @Override
    public void onActivityPaused(Activity activity) {
        paused = true;

        if (check != null)
            handler.removeCallbacks(check);

        handler.postDelayed(check = new Runnable(){
            @Override
            public void run() {
                if (foreground && paused) {
                    foreground = false;
                    Log.d(TAG, "app background");
                    for (AppListener l : listeners) {
                        try {
                            l.onBecameBackground();
                        } catch (Exception exc) {
                            Log.e(TAG, "AppListener threw exception!", exc);
                        }
                    }
                } else {
                    Log.d(TAG, "still foreground");
                }
            }
        }, CHECK_DELAY);
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) {}

    @Override
    public void onActivityStarted(Activity activity) {}

    @Override
    public void onActivityStopped(Activity activity) {}

    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) {}

    @Override
    public void onActivityDestroyed(Activity activity) {}
}