package com.adobe.pdfservices.operation.internal.service.ExtractPDFService;

import com.adobe.pdfservices.operation.exception.SdkException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.api.CPFApi;
import com.adobe.pdfservices.operation.internal.cpf.constants.CPFConstants;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.*;
import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.CPFContentAnalyzerResponse;
import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.ExtractPDF.ExtractPDFOutput;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.internal.http.MultiPartHttpResponse;
import com.adobe.pdfservices.operation.pdfops.options.extractpdf.ExtractPDFOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.mail.internet.MimeBodyPart;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class ExtractPDFService {

    private static final String INPUT_FORMAT_SPECIFIER = "application/pdf";
    private static final String OUTPUT_EXTRACT_INFO_FORMAT_SPECIFIER = "application/json";
    private static final String OUTPUT_EXTRACT_RENDITION_FORMAT_SPECIFIER = "text/directory";
    private static final Logger LOGGER = LoggerFactory.getLogger(ExtractPDFService.class);

    public static String extractPdf(InternalExecutionContext context,
                                    FileRefImpl sourceFileRef, ExtractPDFOptions extractPdfOptions, String operation)
            throws FileNotFoundException {

        try{
            long startTimeMs = System.currentTimeMillis();
            ExtractPDFParams extractPDFParams = ExtractPDFParams.createFrom(extractPdfOptions);
            Inputs inputs = Inputs.builder(INPUT_FORMAT_SPECIFIER)
                    .setParams(extractPDFParams)
                    .build();
            Outputs outputs = new ExtractPDFOutputFormat(OUTPUT_EXTRACT_INFO_FORMAT_SPECIFIER,
                    OUTPUT_EXTRACT_RENDITION_FORMAT_SPECIFIER);

            String extractAnalyzerID = CPFConstants.ExtractPDF.PAPI_CHAIN_ASSET_ID;
            CPFContentAnalyzerRequests CPFContentAnalyzerRequests
                    = new CPFContentAnalyzerRequests(extractAnalyzerID, inputs, outputs);

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = new ArrayList<>();
            sourceFileRefList.add(sourceFileRef);

            HttpResponse<String> response = CPFApi.cpfCreateOpsApi(context, CPFContentAnalyzerRequests,
                    sourceFileRefList, String.class, operation);
            LOGGER.debug("Upload Operation Success Info - Latency(ms): {}", System.currentTimeMillis() - startTimeMs);
            return response.getHeaders().get(DefaultRequestHeaders.LOCATION_HEADER_NAME);
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    public static void downloadAndSave(InternalExecutionContext context, String location, String destinationPath,
                                       ExtensionMediaTypeMapping outputFormat) {

        long startTimeMs = System.currentTimeMillis();
        HttpResponse response = CPFApi.cpfStatusApi(context, location, CPFContentAnalyzerResponse.class);
        LOGGER.debug("Download Operation Success Info - Latency(ms): {}", System.currentTimeMillis() - startTimeMs);
        MultiPartHttpResponse multiPartData = (MultiPartHttpResponse) response;
        List<MimeBodyPart> responseData = multiPartData.getResponseBodyParts();
        try {
            startTimeMs = System.currentTimeMillis();
            ExtractPDFOutput extractPDFOutput = ExtractDataParser.frameExtractSpecificOutput(responseData);
            ExtractDataZipper.zipExtractOutput(extractPDFOutput, destinationPath);
            LOGGER.debug("Response Manipulation Success Info - Latency(ms): {}", System.currentTimeMillis() - startTimeMs);
        } catch (IOException e) {
            LOGGER.error("Error {} while writing downloaded file to location {} ", e, destinationPath);
            throw new SdkException("Exception encountered while downloading file", e);
        } catch (Exception e) {
            LOGGER.debug(e.getMessage());
        } finally {
            try {
                response.consume();
            } catch (IOException e) {
                LOGGER.error("Error while consuming file download response ", e);
            }
        }
    }

}
