/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.service;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import com.adobe.pdfservices.operation.internal.cpf.constants.CPFConstants;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.CPFContentAnalyzerRequests;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.Inputs;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.createpdf.CreatePDFParams;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.htmltopdf.HTMLToPDFParams;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.api.CPFApi;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.pdfops.options.createpdf.CreatePDFOptions;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;

public class CreatePDFService {

    private static final String OUTPUT_FORMAT_SPECIFIER = "application/pdf";

    public static String convertToPdf(InternalExecutionContext context,
                                      FileRefImpl sourceFileRef,
                                      CreatePDFOptions createPDFOptions,
                                      String operation) throws FileNotFoundException {
        try {
            Inputs inputs = null;
            CPFContentAnalyzerRequests cpfContentAnalyzerRequests;
            HTMLToPDFParams htmlToPDFParams;

            ExtensionMediaTypeMapping extensionMediaTypeMapping
                    = ExtensionMediaTypeMapping.getFromMimeType(sourceFileRef.getMediaType());

            if(sourceFileRef.getSourceURL() != null){
                // Create HTMLToPDFParams
                htmlToPDFParams = HTMLToPDFParams.createFrom(createPDFOptions);
                // Create cpf inputs from HTMLToPDFParams
                inputs = Inputs.builder(sourceFileRef.getMediaType())
                        .setParams(htmlToPDFParams)
                        .setSourceUrl(sourceFileRef.getSourceURL())
                        .build();
                // Create CPFContentAnalyzerRequests from cpf inputs and cpf outputs
                cpfContentAnalyzerRequests = new CPFContentAnalyzerRequests(CPFConstants.HTMLToPDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, OUTPUT_FORMAT_SPECIFIER);
            } else {
                assert extensionMediaTypeMapping != null;
                // Keeping switch case because more format specific options will come in the future
                // This switch handles format specific APIs. Currently, there are 2 APIs, one for HTML and other for non-HTML formats
                switch (extensionMediaTypeMapping) {
                    //This case handles both HTML2PDF conversions via HTML with inline CSS and via URL
                    case HTML:
                    case ZIP:
                        // Create HTMLToPDFParams
                        htmlToPDFParams = HTMLToPDFParams.createFrom(createPDFOptions);
                        // Create cpf inputs from HTMLToPDFParams
                        inputs = Inputs.builder(sourceFileRef.getMediaType())
                                .setParams(htmlToPDFParams)
                                .build();

                        // Create CPFContentAnalyzerRequests from cpf inputs and cpf outputs
                        cpfContentAnalyzerRequests = new CPFContentAnalyzerRequests(CPFConstants.HTMLToPDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, OUTPUT_FORMAT_SPECIFIER);

                        break;

                    default:
                        // Create CreatePDFParams
                        CreatePDFParams createPDFParams = CreatePDFParams.createFrom(createPDFOptions);

                        // Create cpf inputs
                        inputs = Inputs.builder(sourceFileRef.getMediaType())
                                .setParams(createPDFParams)
                                .build();

                        // Create CPFContentAnalyzerRequests from cpf inputs and cpf outputs
                        cpfContentAnalyzerRequests = new CPFContentAnalyzerRequests(CPFConstants.CreatePDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, OUTPUT_FORMAT_SPECIFIER);
                }
            }

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = new ArrayList<>();
            sourceFileRefList.add(sourceFileRef);

            HttpResponse<String> response = CPFApi.cpfCreateOpsApi(context, cpfContentAnalyzerRequests,
                    sourceFileRefList, String.class, operation);

            return response.getHeaders().get(DefaultRequestHeaders.LOCATION_HEADER_NAME);
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }
}
