/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob;

import com.google.common.base.Predicate;
import com.google.common.base.Stopwatch;
import com.google.common.cache.AbstractCache;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.Weigher;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.jackrabbit.oak.cache.CacheLIRS;
import org.apache.jackrabbit.oak.commons.FileIOUtils;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.commons.concurrent.ExecutorCloser;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheStatsMBean;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheUpgradeUtils;
import org.apache.jackrabbit.oak.plugins.blob.DataStoreCacheUtils;
import org.apache.jackrabbit.oak.plugins.blob.FileCacheStats;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCache
extends AbstractCache<String, File>
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(FileCache.class);
    protected static final String DOWNLOAD_DIR = "download";
    private File parent;
    private File cacheRoot;
    private CacheLIRS<String, File> cache;
    private FileCacheStats cacheStats;
    private ExecutorService executor;
    private CacheLoader<String, File> cacheLoader;
    private static final Weigher<String, File> weigher = new Weigher<String, File>(){

        @Override
        public int weigh(String key, File value) {
            return Math.round(value.length() / 4096L);
        }
    };
    private static final Weigher<String, File> memWeigher = new Weigher<String, File>(){

        @Override
        public int weigh(String key, File value) {
            return StringUtils.estimateMemoryUsage(key) + StringUtils.estimateMemoryUsage(value.getAbsolutePath()) + 48;
        }
    };

    private FileCache(long maxSize, File root, final CacheLoader<String, InputStream> loader, @Nullable ExecutorService executor) {
        this.parent = root;
        this.cacheRoot = new File(root, DOWNLOAD_DIR);
        long size = Math.round(maxSize / 4096L);
        this.cacheLoader = new CacheLoader<String, File>(){

            @Override
            public File load(String key) throws Exception {
                File cachedFile = DataStoreCacheUtils.getFile(key, FileCache.this.cacheRoot);
                if (cachedFile.exists()) {
                    return cachedFile;
                }
                InputStream is = null;
                boolean threw = true;
                try {
                    is = (InputStream)loader.load(key);
                    FileIOUtils.copyInputStreamToFile(is, cachedFile);
                    threw = false;
                }
                catch (Exception e) {
                    try {
                        LOG.warn("Error reading object for id [{}] from backend", (Object)key, (Object)e);
                        throw e;
                    }
                    catch (Throwable throwable) {
                        Closeables.close(is, threw);
                        throw throwable;
                    }
                }
                Closeables.close(is, threw);
                return cachedFile;
            }
        };
        this.cache = new CacheLIRS.Builder().maximumWeight(size).recordStats().weigher(weigher).evictionCallback(new CacheLIRS.EvictionCallback<String, File>(){

            @Override
            public void evicted(@NotNull String key, @Nullable File cachedFile, @NotNull RemovalCause cause) {
                try {
                    if (cachedFile != null && cachedFile.exists() && cause != RemovalCause.REPLACED) {
                        DataStoreCacheUtils.recursiveDelete(cachedFile, FileCache.this.cacheRoot);
                        LOG.info("File [{}] evicted with reason [{}]", (Object)cachedFile, (Object)cause.toString());
                    }
                }
                catch (IOException e) {
                    LOG.info("Cached file deletion failed after eviction", e);
                }
            }
        }).build();
        this.cacheStats = new FileCacheStats(this.cache, weigher, memWeigher, maxSize);
        this.executor = executor == null ? Executors.newSingleThreadExecutor() : executor;
        this.executor.submit(new CacheBuildJob());
    }

    private FileCache() {
    }

    public static FileCache build(long maxSize, File root, CacheLoader<String, InputStream> loader, @Nullable ExecutorService executor) {
        if (maxSize > 0L) {
            return new FileCache(maxSize, root, loader, executor);
        }
        return new FileCache(){

            @Override
            public void put(String key, File file) {
            }

            @Override
            public boolean containsKey(String key) {
                return false;
            }

            @Override
            @Nullable
            public File getIfPresent(String key) {
                return null;
            }

            @Override
            public File get(String key) throws IOException {
                return null;
            }

            @Override
            public void invalidate(Object key) {
            }

            @Override
            public DataStoreCacheStatsMBean getStats() {
                return new FileCacheStats(this, weigher, memWeigher, 0L);
            }

            @Override
            public void close() {
            }
        };
    }

    @Override
    public void put(String key, File file) {
        this.put(key, file, true);
    }

    private void put(String key, File file, boolean copy) {
        try {
            File cached = DataStoreCacheUtils.getFile(key, this.cacheRoot);
            if (!cached.exists()) {
                if (copy) {
                    FileUtils.copyFile(file, cached);
                } else {
                    FileUtils.moveFile(file, cached);
                }
            }
            this.cache.put(key, cached);
        }
        catch (IOException e) {
            LOG.error("Exception adding id [{}] with file [{}] to cache, root cause: {}", key, file, e.getMessage());
            LOG.debug("Root cause", e);
        }
    }

    public boolean containsKey(String key) {
        return this.cache.containsKey(key);
    }

    @Nullable
    public File getIfPresent(String key) {
        try {
            return this.cache.getIfPresent(key);
        }
        catch (Exception e) {
            LOG.error("Error in retrieving [{}] from cache", (Object)key, (Object)e);
            return null;
        }
    }

    @Override
    @Nullable
    public File getIfPresent(Object key) {
        return this.getIfPresent((String)key);
    }

    public File get(String key) throws IOException {
        try {
            return this.cache.get(key, () -> this.cacheLoader.load(key));
        }
        catch (ExecutionException e) {
            LOG.error("Error loading [{}] from cache", (Object)key);
            throw new IOException(e);
        }
    }

    @Override
    public void invalidate(Object key) {
        this.cache.invalidate(key);
    }

    public DataStoreCacheStatsMBean getStats() {
        return this.cacheStats;
    }

    @Override
    public void close() {
        LOG.info("Cache stats on close [{}]", (Object)this.cacheStats.cacheInfoAsString());
        new ExecutorCloser(this.executor).close();
    }

    private int build() {
        int count = 0;
        DataStoreCacheUpgradeUtils.moveDownloadCache(this.parent);
        for (File toBeSyncedFile : Files.fileTreeTraverser().postOrderTraversal(this.cacheRoot).filter(new Predicate<File>(){

            @Override
            public boolean apply(File input) {
                return input.isFile() && !FilenameUtils.normalizeNoEndSeparator(input.getParent()).equals(FileCache.this.cacheRoot.getAbsolutePath());
            }
        })) {
            try {
                this.put(toBeSyncedFile.getName(), toBeSyncedFile, false);
                ++count;
                LOG.trace("Added file [{}} to in-memory cache", (Object)toBeSyncedFile);
            }
            catch (Exception e) {
                LOG.error("Error in putting cached file in map[{}]", (Object)toBeSyncedFile);
            }
        }
        LOG.trace("[{}] files put in im-memory cache", (Object)count);
        return count;
    }

    private class CacheBuildJob
    implements Callable {
        private CacheBuildJob() {
        }

        public Integer call() {
            Stopwatch watch = Stopwatch.createStarted();
            int count = FileCache.this.build();
            LOG.info("Cache built with [{}] files from file system in [{}] seconds", (Object)count, (Object)watch.elapsed(TimeUnit.SECONDS));
            return count;
        }
    }
}

