/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.acs.commons.packagegarbagecollector;

import java.io.IOException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import org.apache.jackrabbit.vault.packaging.JcrPackage;
import org.apache.jackrabbit.vault.packaging.JcrPackageDefinition;
import org.apache.jackrabbit.vault.packaging.JcrPackageManager;
import org.apache.jackrabbit.vault.packaging.PackageId;
import org.apache.jackrabbit.vault.packaging.Packaging;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.event.jobs.Job;
import org.apache.sling.event.jobs.consumer.JobConsumer;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(service={JobConsumer.class}, immediate=true, property={"job.topics=com/adobe/acs/commons/PackageGarbageCollectionJob"})
public class PackageGarbageCollectionJob
implements JobConsumer {
    public static final DateTimeFormatter LOCALIZED_DATE_FORMATTER = DateTimeFormatter.ofLocalizedDate(FormatStyle.MEDIUM);
    private static final Logger LOG = LoggerFactory.getLogger(PackageGarbageCollectionJob.class);
    private static final String SERVICE_USER = "package-garbage-collection";
    @Reference
    Packaging packaging;
    @Reference
    ResourceResolverFactory resourceResolverFactory;

    public JobConsumer.JobResult process(Job job) {
        String groupName = (String)job.getProperty("groupName", String.class);
        Integer maxAgeInDays = (Integer)job.getProperty("maxAgeInDays", Integer.class);
        boolean removeNotInstalledPackages = (Boolean)job.getProperty("removeNotInstalledPackages", (Object)false);
        int packagesRemoved = 0;
        LOG.debug("Job Configuration: [Group Name: {}, Service User: {}, Age of Package {} days,Remove not installed packages: {}]", new Object[]{groupName, SERVICE_USER, maxAgeInDays, removeNotInstalledPackages});
        try (ResourceResolver resourceResolver = this.resourceResolverFactory.getServiceResourceResolver(Collections.singletonMap("sling.service.subservice", SERVICE_USER));){
            Session session = (Session)resourceResolver.adaptTo(Session.class);
            JcrPackageManager packageManager = this.packaging.getPackageManager(session);
            List packages = packageManager.listPackages(groupName, false);
            Iterator iterator = packages.iterator();
            while (iterator.hasNext()) {
                JcrPackage tmpPackage;
                JcrPackage jcrPackage = tmpPackage = (JcrPackage)iterator.next();
                try {
                    JcrPackageDefinition definition = jcrPackage.getDefinition();
                    if (definition == null) {
                        LOG.warn("Skipping package without definition: {}", (Object)jcrPackage.getNode().getPath());
                    }
                    String packageDescription = this.getPackageDescription(definition);
                    LOG.info("Processing package {}", (Object)packageDescription);
                    if (this.isPackageOldEnough(definition, maxAgeInDays)) {
                        if (removeNotInstalledPackages && !this.isInstalled(definition)) {
                            packageManager.remove(jcrPackage);
                            ++packagesRemoved;
                            LOG.info("Deleted not-installed package {}", (Object)packageDescription);
                            continue;
                        }
                        if (this.isInstalled(definition) && !this.isLatestInstalled(definition, packageManager.listPackages(groupName, false).stream())) {
                            packageManager.remove(jcrPackage);
                            ++packagesRemoved;
                            LOG.info("Deleted installed package {} since it is not the latest installed version.", (Object)packageDescription);
                            continue;
                        }
                        LOG.info("Not removing package because it's the current installed one {}", (Object)packageDescription);
                        continue;
                    }
                    LOG.debug("Not removing package because it's not old enough {}", (Object)packageDescription);
                }
                finally {
                    if (jcrPackage == null) continue;
                    jcrPackage.close();
                }
            }
        }
        catch (IOException | RepositoryException | LoginException e) {
            if (packagesRemoved > 0) {
                LOG.error("Package Garbage Collector job partially failed - Removed {} packages", (Object)packagesRemoved);
            }
            LOG.error("Unable to finish clearing packages", e);
            return JobConsumer.JobResult.FAILED;
        }
        LOG.info("Package Garbage Collector job finished - Removed {} packages", (Object)packagesRemoved);
        return JobConsumer.JobResult.OK;
    }

    private boolean isInstalled(JcrPackageDefinition pkgDefinition) {
        return pkgDefinition.getLastUnpacked() != null;
    }

    private boolean isLatestInstalled(JcrPackageDefinition referencePkgDefinition, Stream<JcrPackage> installedPackages) throws RepositoryException {
        try {
            Optional<JcrPackageDefinition> lastInstalledPckDefinitionOptional = installedPackages.map(p -> {
                try {
                    return p.getDefinition();
                }
                catch (RepositoryException e) {
                    String pckPath;
                    try {
                        pckPath = p.getNode().getPath();
                    }
                    catch (RepositoryException nestedException) {
                        pckPath = "Unknown";
                    }
                    throw new UncheckedRepositoryException(new RepositoryException("Cannot read package definition of package " + pckPath, (Throwable)e));
                }
            }).filter(def -> PackageGarbageCollectionJob.isSameNameAndGroup(referencePkgDefinition.getId(), def.getId())).filter(def -> def.getLastUnpacked() != null).max(Comparator.comparing(def -> def.getLastUnpacked()));
            if (lastInstalledPckDefinitionOptional.isPresent()) {
                return lastInstalledPckDefinitionOptional.get().getId().equals((Object)referencePkgDefinition.getId());
            }
            return false;
        }
        catch (UncheckedRepositoryException e) {
            throw e.getCause();
        }
    }

    public static boolean isSameNameAndGroup(PackageId thisPackageId, PackageId otherPackageId) {
        return otherPackageId.getGroup().equals(thisPackageId.getGroup()) && otherPackageId.getName().equals(thisPackageId.getName());
    }

    private boolean isPackageOldEnough(JcrPackageDefinition pkgDefinition, Integer maxAgeInDays) throws RepositoryException, IOException {
        Calendar packageUploadedDate;
        Period maxAge = Period.ofDays(maxAgeInDays);
        LocalDate oldestAge = LocalDate.now().minus(maxAge);
        try {
            packageUploadedDate = pkgDefinition.getLastUnwrapped();
            if (packageUploadedDate == null) {
                LOG.warn("Package [ {} ] has no lastUnwrapped (uploaded) date, assuming it's NOT old enough", (Object)pkgDefinition.getNode().getPath());
                return false;
            }
        }
        catch (RepositoryException e) {
            LOG.error("Unable to get lastUnwrapped (uploaded) date for package [ {} ]", (Object)pkgDefinition.getNode().getPath(), (Object)e);
            return false;
        }
        LocalDate packageUploadedAt = LocalDateTime.ofInstant(packageUploadedDate.toInstant(), packageUploadedDate.getTimeZone().toZoneId()).toLocalDate();
        String packageDescription = this.getPackageDescription(pkgDefinition);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Checking if package is old enough: Name: {}, Uploaded at: {}, Oldest age: {}", new Object[]{packageDescription, packageUploadedAt.format(LOCALIZED_DATE_FORMATTER), oldestAge.format(LOCALIZED_DATE_FORMATTER)});
        }
        return !packageUploadedAt.isAfter(oldestAge);
    }

    private String getPackageDescription(JcrPackageDefinition definition) throws RepositoryException {
        return String.format("%s:%s:v%s [%s]", definition.getId().getName(), definition.getId().getGroup(), definition.getId().getVersionString(), definition.getNode().getPath());
    }

    private static final class UncheckedRepositoryException
    extends RuntimeException {
        private static final long serialVersionUID = 8851421623772855854L;

        protected UncheckedRepositoryException(RepositoryException e) {
            super(e);
        }

        @Override
        public RepositoryException getCause() {
            return (RepositoryException)super.getCause();
        }
    }
}

