package com.adi.lib.chart;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.RectF;
import android.os.Handler;
import android.os.Looper;
import android.support.v4.content.ContextCompat;
import android.util.AttributeSet;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.View;

import com.adi.lib.R;
import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.Scene;

import java.util.List;

import static com.adi.lib.chart.ChartView.THEME_DARK;
import static com.adi.lib.chart.ChartView.THEME_LIGHT;

public class SliderView extends View {

    List<ChartLine> chartLines;

    float verticalBorder, horizontalBorder;

    long minX, maxX, sliderMinX, sliderMaxX, sliderWidth, minSliderWith;

    Scene scene;
    private ChangeListener changeListener;
    GestureDetector gestureDetector;

    short scrollMode; // 0 - left side size change, 1 - right side size change, 2 - move
    Paint paint = new Paint();
    Paint clear = new Paint(Paint.ANTI_ALIAS_FLAG);
    RectF rectf = new RectF();
    int sliderDarkColor;
    int sliderLightColor;
    float screenSlideMinX, screenSlideMaxX, padding;
    float cornerRadius;
    float screenSliderMinX, screenSliderMaxX;
    float whiteDashWidth, whiteDashHeight, whiteDashCornerRadius;
    Runnable stopScrollDetector;
    Handler handler = new Handler(Looper.getMainLooper());

    public SliderView(Context context) {
        super(context);
        init();
    }

    public SliderView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public SliderView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        stopScrollDetector = () -> {
            changeListener.onStopScroll();
        };
        this.setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        paint.setAntiAlias(true);
        clear.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
        verticalBorder = Math.round(AndroidUtils.dpToPx(this.getContext(), 1));
        whiteDashWidth = Math.round(AndroidUtils.dpToPx(this.getContext(), 2));
        whiteDashHeight = Math.round(AndroidUtils.dpToPx(this.getContext(), 11));
        whiteDashCornerRadius = Math.round(AndroidUtils.dpToPx(this.getContext(), 2));
        horizontalBorder = Math.round(AndroidUtils.dpToPx(this.getContext(), 11));
        cornerRadius = Math.round(AndroidUtils.dpToPx(this.getContext(), 6));
        gestureDetector = new GestureDetector(new GestureDetector.SimpleOnGestureListener() {
            @Override
            public boolean onDown(MotionEvent e) {
                screenSlideMinX = getScreenX(sliderMinX);
                screenSlideMaxX = getScreenX(sliderMaxX);
                padding = (screenSlideMaxX - screenSlideMinX) / 5;
                if (e.getX() < screenSlideMinX - padding || e.getX() > screenSlideMaxX + padding) {
                    return false;
                } else {
                    if (e.getX() > screenSlideMinX + padding && e.getX() < screenSlideMaxX - padding) {
                        scrollMode = 2;
                    } else {
                        if (e.getX() <= screenSlideMinX + padding && e.getX() >= screenSlideMinX - padding) {
                            scrollMode = 0;
                        } else {
                            scrollMode = 1;
                        }
                    }
                    return true;
                }
            }

            @Override
            public boolean onScroll(MotionEvent e1, MotionEvent e2, float distanceX, float distanceY) {
                if (e1.getX() > e2.getX()) {
                    scene.setMoveDirection(0); //left
                } else {
                    scene.setMoveDirection(1); //right
                }
                float scrollInPercent = distanceX / (float) getWidth();
                if (scrollMode == 2) {
                    moveSlider(scrollInPercent);
                    changeListener.onSliderChanged(sliderMinX, sliderMaxX, false);
                } else {
                    changeSliderWidth(scrollInPercent);
                    changeListener.onSliderChanged(sliderMinX, sliderMaxX, true);
                }
                handler.removeCallbacks(stopScrollDetector);
                handler.postDelayed(stopScrollDetector, 100);
                invalidate();
                return true;
            }
        });

        applyTheme();
    }

    public void applyTheme(){
        if (ChartView.THEME == THEME_LIGHT) {
            sliderDarkColor = ContextCompat.getColor(getContext(), R.color.sliderDark);
            sliderLightColor = ContextCompat.getColor(getContext(), R.color.sliderLight);
        } else {
            sliderDarkColor = ContextCompat.getColor(getContext(), R.color.darkSliderDark);
            sliderLightColor = ContextCompat.getColor(getContext(), R.color.darkSliderLight);
        }
    }


    private void changeSliderWidth(float delta) {
        if (scrollMode == 0) {
            if (delta > 0) {
                sliderMinX = sliderMinX - (long) Math.abs((maxX - minX) * delta);
                if (sliderMinX < minX) {
                    sliderMinX = minX;
                }
            } else {
                sliderMinX = sliderMinX + (long) Math.abs((maxX - minX) * delta);
                if (sliderMaxX - sliderMinX < minSliderWith) {
                    sliderMinX = sliderMaxX - minSliderWith;
                }
            }
        } else {
            if (delta > 0) {
                sliderMaxX = sliderMaxX - (long) Math.abs((maxX - minX) * delta);
                if (sliderMaxX - sliderMinX < minSliderWith) {
                    sliderMaxX = sliderMinX + minSliderWith;
                }
            } else {
                sliderMaxX = sliderMaxX + (long) Math.abs((maxX - minX) * delta);
                if (sliderMaxX > maxX) {
                    sliderMaxX = maxX;
                }
            }
        }
        sliderWidth = sliderMaxX - sliderMinX;
    }


    private void moveSlider(float delta) {
        if (delta > 0) {
            sliderMinX = sliderMinX - (long) Math.abs((maxX - minX) * delta);
            if (sliderMinX < minX) {
                sliderMinX = minX;
            }
            sliderMaxX = sliderMinX + sliderWidth;
        } else {
            sliderMaxX = sliderMaxX + (long) Math.abs((maxX - minX) * delta);
            if (sliderMaxX > maxX) {
                sliderMaxX = maxX;
            }
            sliderMinX = sliderMaxX - sliderWidth;
        }
    }


    public void updateChart(List<ChartLine> chartLines, Scene scene) {
        this.chartLines = chartLines;
        if (this.scene == null) {
            this.scene = scene;
            minX = chartLines.get(0).getX()[0];
            maxX = chartLines.get(0).getX()[chartLines.get(0).getX().length - 1];
            sliderMinX = maxX - scene.getMinSliderWidth();
            sliderMaxX = maxX;
            sliderWidth = sliderMaxX - sliderMinX;
            minSliderWith = scene.getMinSliderWidth();
        }
        changeListener.onSliderChanged(sliderMinX, sliderMaxX, true);
        invalidate();
    }


    @Override
    protected void onDraw(Canvas canvas) {
        screenSliderMinX = getScreenX(sliderMinX);
        screenSliderMaxX = getScreenX(sliderMaxX);
        if (ChartView.THEME == THEME_DARK) {
            paint.setColor(ContextCompat.getColor(getContext(), R.color.darkChartViewBackgroundColor));
        } else {
            paint.setColor(ContextCompat.getColor(getContext(), R.color.chartViewBackgroundColor));
        }
        paint.setStyle(Paint.Style.FILL);

        canvas.drawRect(0, 0, getWidth(), getHeight(), paint);

        paint.setColor(sliderLightColor);
        rectf.set(0, verticalBorder, getWidth(), getHeight() - verticalBorder);
        canvas.drawRoundRect(rectf, cornerRadius, cornerRadius, clear);
        canvas.drawRoundRect(rectf, cornerRadius, cornerRadius, paint);


        paint.setColor(sliderDarkColor);
        rectf.set(screenSliderMinX, 0F, screenSliderMaxX, getHeight());
        canvas.drawRoundRect(rectf, cornerRadius, cornerRadius, paint);
        canvas.drawRect(screenSliderMinX + horizontalBorder, 0F + verticalBorder, screenSliderMaxX - horizontalBorder, getHeight() - verticalBorder, clear);

        paint.setColor(Color.WHITE);
        rectf.set(screenSliderMinX + (horizontalBorder - whiteDashWidth) / 2f, 0F + ((float) getHeight() - whiteDashHeight) / 2f, screenSliderMinX + (horizontalBorder - whiteDashWidth) / 2f + whiteDashWidth, getHeight() - ((float) getHeight() - whiteDashHeight) / 2f);
        canvas.drawRoundRect(rectf, whiteDashCornerRadius, whiteDashCornerRadius, paint);
        rectf.set(screenSliderMaxX - (horizontalBorder - whiteDashWidth) / 2f - whiteDashWidth, 0F + ((float) getHeight() - whiteDashHeight) / 2f, screenSliderMaxX - (horizontalBorder - whiteDashWidth) / 2f, getHeight() - ((float) getHeight() - whiteDashHeight) / 2f);
        canvas.drawRoundRect(rectf, whiteDashCornerRadius, whiteDashCornerRadius, paint);

    }

    public interface ChangeListener {
        void onSliderChanged(long minX, long maxX, boolean widthChanged);

        void onStopScroll();
    }

    public void setChangeListener(ChangeListener changeListener) {
        this.changeListener = changeListener;
    }

    private float getScreenX(long x) {
        return (((float) (x - minX) / (maxX - minX)) * (float) getWidth());
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        getParent().getParent().getParent().requestDisallowInterceptTouchEvent(true);
        if (this.gestureDetector.onTouchEvent(event)) {
            return true;
        }

        return super.onTouchEvent(event);
    }

    @Override
    public boolean performClick() {
        return super.performClick();
    }
}
