package com.adi.lib.chart;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.animation.ValueAnimator;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.support.v4.content.ContextCompat;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.View;

import com.adi.lib.R;
import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.Chart;
import com.adi.lib.chart.entity.PickerDetails;
import com.adi.lib.chart.entity.Scene;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import static com.adi.lib.chart.ChartView.ANIMATOR_DURATION;
import static com.adi.lib.chart.ChartView.THEME_DARK;


public class PickerView extends View {

    private float cardVerticalMarginTop;
    private float cardVerticalMarginLeft;
    private float cardHorizontalMargin;
    private float cardVerticalPadding;
    private float cardHorizontalPadding;
    private float cardBottomPadding;
    private float rawHeight;
    private float dateRawHeight;
    private float arrowSize;
    private float arrowPaddingRight;
    private float arrowPaddingTop;
    private float paddingBetweenDates;
    private long[] xArray;
    protected long minX, maxX;
    protected long horizontalPaddingInPercent;
    protected int horizontalPadding;
    protected float topPadding = 0;
    protected float dateTopPadding;
    float dateFontSize;
    float valFontSize;
    float date1Width;
    float date2Width;
    float valueWidth;
    DateFormat df1 = new SimpleDateFormat("EEE, dd", Locale.US);
    DateFormat df2 = new SimpleDateFormat("MMM yyyy", Locale.US);

    RectF cardRect = new RectF();
    Paint dateTextPaint = new Paint();
    Paint showDateTextPaint = new Paint();
    Paint hideDateTextPaint = new Paint();
    Paint valTextPaint = new Paint();
    Paint showValTextPaint = new Paint();
    Paint hideValTextPaint = new Paint();
    Paint nameTextPaint = new Paint();

    float cornerRoundRadius;

    float circleBorderWidth, circleRadius;

    Paint paint = new Paint();
    Paint circlePaint = new Paint();
    Paint platePaint = new Paint();
    Scene scene;
    private float height, width;
    GestureDetector gestureDetector;
    private PickListener pickListener;
    Chart chart;
    float minCardWidth;
    float left, top, x;
    float cardWidth, cardHeight;
    private boolean picked;
    private PickerDetails pickerDetails = new PickerDetails();
    private PickerDetails oldPickerDetails;
    Date date = new Date();
    Drawable arrow;
    int idx;
    private float screenX;
    private boolean animateValues;
    private boolean animateDate1;
    private boolean animateDate2;

    private ValueAnimator hideSizeAnimator = new ValueAnimator();
    private ValueAnimator showSizeAnimator = new ValueAnimator();
    private ValueAnimator hideAlphaAnimator = new ValueAnimator();
    private ValueAnimator showAlphaAnimator = new ValueAnimator();
    private AnimatorSet animatorSet = new AnimatorSet();
    int showValueAlpha, hideValueAlpha;
    private String fullDate1;
    private String fullDate2;


    public PickerView(Context context, Chart chart) {
        super(context);
        this.chart = chart;
        xArray = chart.getChartLines().get(0).getX();
        init();
    }

    private void init() {
        this.setLayerType(View.LAYER_TYPE_SOFTWARE, null);

        arrow = ContextCompat.getDrawable(getContext(), R.drawable.arrow);
        arrowSize = AndroidUtils.dpToPx(this.getContext(), 24);
        arrowPaddingRight = AndroidUtils.dpToPx(this.getContext(), 6);
        arrowPaddingTop = AndroidUtils.dpToPx(this.getContext(), 8);
        minCardWidth = AndroidUtils.dpToPx(this.getContext(), 156);
        cardVerticalMarginTop = AndroidUtils.dpToPx(this.getContext(), 0);
        cardVerticalMarginLeft = AndroidUtils.dpToPx(this.getContext(), 14);
        cardHorizontalMargin = AndroidUtils.dpToPx(this.getContext(), 14);
        cardHorizontalPadding = AndroidUtils.dpToPx(this.getContext(), 15);
        cardVerticalPadding = AndroidUtils.dpToPx(this.getContext(), 14);
        cardHorizontalPadding = AndroidUtils.dpToPx(this.getContext(), 13);
        cardBottomPadding = AndroidUtils.dpToPx(this.getContext(), 6);
        paddingBetweenDates = AndroidUtils.dpToPx(this.getContext(), 6);
        dateRawHeight = AndroidUtils.dpToPx(this.getContext(), 34);
        rawHeight = AndroidUtils.dpToPx(this.getContext(), 24);
        circleBorderWidth = AndroidUtils.dpToPx(getContext(), 2);
        circleRadius = AndroidUtils.dpToPx(getContext(), 5);
        cornerRoundRadius = AndroidUtils.dpToPx(getContext(), 4);
        dateTopPadding = AndroidUtils.dpToPx(getContext(), 10);
        paint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        paint.setAntiAlias(true);
        paint.setStrokeCap(Paint.Cap.ROUND);
        paint.setStyle(Paint.Style.STROKE);


        dateFontSize = AndroidUtils.dpToPx(this.getContext(), 14);
        dateTextPaint.setAntiAlias(true);
        dateTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));
        dateTextPaint.setTextSize(dateFontSize);
        dateTextPaint.setTextAlign(Paint.Align.RIGHT);

        showDateTextPaint.setAntiAlias(true);
        showDateTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));
        showDateTextPaint.setTextSize(dateFontSize);
        showDateTextPaint.setTextAlign(Paint.Align.LEFT);

        hideDateTextPaint.setAntiAlias(true);
        hideDateTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));
        hideDateTextPaint.setTextSize(dateFontSize);
        hideDateTextPaint.setTextAlign(Paint.Align.RIGHT);

        valFontSize = AndroidUtils.dpToPx(this.getContext(), 14);
        valTextPaint.setAntiAlias(true);
        valTextPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 14));
        valTextPaint.setStyle(Paint.Style.FILL);
        valTextPaint.setTextAlign(Paint.Align.LEFT);
        valTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));

        showValTextPaint.setAntiAlias(true);
        showValTextPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 14));
        showValTextPaint.setStyle(Paint.Style.FILL);
        showValTextPaint.setTextAlign(Paint.Align.LEFT);
        showValTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));

        hideValTextPaint.setAntiAlias(true);
        hideValTextPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 14));
        hideValTextPaint.setStyle(Paint.Style.FILL);
        hideValTextPaint.setTextAlign(Paint.Align.RIGHT);
        hideValTextPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));

        nameTextPaint.setAntiAlias(true);
        nameTextPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 14));
        nameTextPaint.setStyle(Paint.Style.FILL);


        circlePaint = new Paint();
        circlePaint.setColor(Color.WHITE);
        circlePaint.setStrokeWidth(2);
        circlePaint.setStyle(Paint.Style.STROKE);
        platePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        platePaint.setColor(Color.WHITE);

        gestureDetector = new GestureDetector(new GestureDetector.SimpleOnGestureListener() {
            @Override
            public boolean onDown(MotionEvent e) {
                if (picked) {
                    if (e.getX() > left && e.getX() < left + cardWidth && e.getY() > top && e.getY() < top + cardHeight) {
                        //TODO action
                    } else {
                        pick(e.getX());
                    }
                } else {
                    pick(e.getX());
                }
                return true;
            }
        });
        applyTheme();

        hideSizeAnimator.addUpdateListener(a -> {
            hideDateTextPaint.setTextSize((Float) a.getAnimatedValue());
            hideValTextPaint.setTextSize((Float) a.getAnimatedValue());
        });
        showSizeAnimator.addUpdateListener(a -> {
            showDateTextPaint.setTextSize((Float) a.getAnimatedValue());
            showValTextPaint.setTextSize((Float) a.getAnimatedValue());
        });

        hideAlphaAnimator.addUpdateListener(a -> {
            hideDateTextPaint.setAlpha((int) a.getAnimatedValue());
            hideValueAlpha = (int) a.getAnimatedValue();
        });
        showAlphaAnimator.addUpdateListener(a -> {
            showDateTextPaint.setAlpha((int) a.getAnimatedValue());
            showValueAlpha = (int) a.getAnimatedValue();
            invalidate();
        });
        animatorSet.playTogether(hideSizeAnimator, showSizeAnimator, hideAlphaAnimator, showAlphaAnimator);
        animatorSet.setDuration(ANIMATOR_DURATION);
        animatorSet.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationEnd(Animator animation) {
                super.onAnimationEnd(animation);
                oldPickerDetails = pickerDetails.copy();
                animateDate1 = false;
                animateDate2 = false;
                animateValues = false;
            }
        });
    }


    public void applyTheme() {
        if (ChartView.THEME == THEME_DARK) {
            paint.setColor(getResources().getColor(R.color.darkPickerLineColor));
            dateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.darkPlaneTextColor));
            showDateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.darkPlaneTextColor));
            hideDateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.darkPlaneTextColor));
            platePaint.setColor(ContextCompat.getColor(getContext(), R.color.darkPlaneColor));
            nameTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.darkPlaneTextColor));
            platePaint.setShadowLayer(AndroidUtils.dpToPx(getContext(), 3), 0, 0, ContextCompat.getColor(getContext(), R.color.darkPlateShadowColor));

        } else {
            dateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.planeTextColor));
            showDateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.planeTextColor));
            hideDateTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.planeTextColor));
            paint.setColor(getResources().getColor(R.color.pickerLineColor));
            platePaint.setColor(ContextCompat.getColor(getContext(), R.color.planeColor));
            nameTextPaint.setColor(ContextCompat.getColor(getContext(), R.color.planeTextColor));
            platePaint.setShadowLayer(AndroidUtils.dpToPx(getContext(), 3), 0, 0, ContextCompat.getColor(getContext(), R.color.plateShadowColor));
        }

    }


    public void refresh() {
        if (picked) {
            drawPick(screenX, true);
        }
    }

    public void pick(float screenX) {
        drawPick(screenX, false);
    }

    private void drawPick(float screenX, boolean exec) {
        this.screenX = screenX;
        int visibleCount = ChartCore.getVisibleChartLinesCount(chart.getChartLines());
        if (visibleCount == 0) {
            picked = false;
            pickListener.clearPick();
            invalidate();
            return;
        }
        PickerDetails pickerDetails = getPickDetails(screenX, exec);
        pickListener.pick(pickerDetails);


        cardWidth = minCardWidth;
        cardHeight = dateRawHeight + visibleCount * rawHeight + cardBottomPadding;

        //определяем положение карточки
        float topHeight = getScreenY(pickerDetails.maxValue);
        if (exec) {
            x = getScreenX(pickerDetails.x);
        } else {
            x = getScreenX(xArray[pickerDetails.closestXPosition]);
        }

        date.setTime(pickerDetails.x);
        pickerDetails.date1 = df1.format(date);
        pickerDetails.date2 = df2.format(date);
        if (topHeight >= cardHeight + cardVerticalMarginTop * 2 + topPadding) { // если есть место сверху
            left = x - cardWidth / 2f;
            top = cardVerticalMarginTop + topPadding;
            if (left + cardWidth > width) { //не помещается справа
                //рисуем слева
                top = cardVerticalMarginLeft + topPadding;
                left = x - cardWidth - cardHorizontalMargin;
            }
            if (left < 0) { //не помещается cлева
                //рисуем справа
                top = cardVerticalMarginLeft + topPadding;
                left = x + cardHorizontalMargin;
            }
        } else {
            top = cardVerticalMarginLeft + topPadding;
            float leftWidth = getScreenX(pickerDetails.x);
            if (leftWidth > cardWidth) { // если есть место слева
                left = x - cardWidth - cardHorizontalMargin;
            } else {
                left = x + cardHorizontalMargin;
            }
        }
        if (left + cardWidth > width) {
            left = width - cardWidth;
        }
        if (left < 0) {
            left = 0;
        }
        cardRect.set(left, top, left + cardWidth, top + cardHeight);

        if (!picked || oldPickerDetails == null) {
            oldPickerDetails = pickerDetails.copy();
            invalidate();
        } else {
            animateValues = true;
            animateDate1 = !pickerDetails.date1.equals(oldPickerDetails.date1);
            animateDate2 = !pickerDetails.date2.equals(oldPickerDetails.date2);
            hideSizeAnimator.setFloatValues(dateFontSize, 0f);
            showSizeAnimator.setFloatValues(0f, dateFontSize);
            hideAlphaAnimator.setIntValues(255, 0);
            showAlphaAnimator.setIntValues(0, 255);
            animatorSet.start();
        }
        picked = true;
    }


    public void updateScene(Scene scene) {
        this.scene = scene;
        horizontalPaddingInPercent = (long) ((double) (scene.getMaxX() - scene.getMinX()) * (double) horizontalPadding / (double) width);
        minX = scene.getMinX() - horizontalPaddingInPercent;
        maxX = scene.getMaxX() + horizontalPaddingInPercent;
    }

    public void clear() {
        picked = false;
        pickListener.clearPick();
        invalidate();
    }


    @Override
    protected void onDraw(Canvas canvas) {
        if (picked) {
            canvas.drawRoundRect(cardRect, cornerRoundRadius, cornerRoundRadius, platePaint);
            drawDates(canvas, pickerDetails);
            idx = 0;
            for (int i = 0; i < chart.getChartLines().size(); i++) {
                if (chart.getChartLines().get(i).isVisible()) {
                    canvas.drawText(chart.getChartLines().get(i).getName(), 0, chart.getChartLines().get(i).getName().length, left + cardHorizontalPadding, top + cardVerticalPadding + dateRawHeight + rawHeight * idx, nameTextPaint);
                    drawValue(canvas, i);
                    idx++;
                }
            }
            arrow.setBounds((int) ((left + cardWidth) - arrowPaddingRight - arrowSize), (int) (top + arrowPaddingTop), (int) (left + cardWidth - arrowPaddingRight), (int) (top + arrowPaddingTop + arrowSize));
            arrow.draw(canvas);
        }
    }

    private void drawValue(Canvas canvas, int index) {
        valueWidth = valTextPaint.measureText((int) pickerDetails.values[index] + "");
        if (animateValues && pickerDetails.values[index] != oldPickerDetails.values[index]) {
            showValTextPaint.setColor(chart.getChartLines().get(index).getColor());
            hideValTextPaint.setColor(chart.getChartLines().get(index).getColor());
            showValTextPaint.setAlpha(showValueAlpha);
            hideValTextPaint.setAlpha(hideValueAlpha);
            canvas.drawText(((int) pickerDetails.values[index]) + "", left + cardWidth - valueWidth - cardHorizontalPadding, top + cardVerticalPadding + dateRawHeight + rawHeight * idx, showValTextPaint);
            canvas.drawText(((int) oldPickerDetails.values[index]) + "", left + cardWidth - cardHorizontalPadding, top + cardVerticalPadding + dateRawHeight + rawHeight * idx, hideValTextPaint);
        } else {
            valTextPaint.setColor(chart.getChartLines().get(index).getColor());
            canvas.drawText(((int) pickerDetails.values[index]) + "", left + cardWidth - valueWidth - cardHorizontalPadding, top + cardVerticalPadding + dateRawHeight + rawHeight * idx, valTextPaint);
        }
    }

    private void drawDates(Canvas canvas, PickerDetails pickerDetails) {
        date1Width = dateTextPaint.measureText(pickerDetails.date1);
        date2Width = dateTextPaint.measureText(pickerDetails.date2);
        if (animateDate1 && animateDate2) {
            fullDate1 = pickerDetails.date1 + " " + pickerDetails.date2;
            fullDate2 = oldPickerDetails.date1 + " " + oldPickerDetails.date2;
            date1Width = dateTextPaint.measureText(fullDate1);
            canvas.drawText(fullDate1, left + cardHorizontalPadding, top + cardVerticalPadding + dateTopPadding, showDateTextPaint);
            canvas.drawText(fullDate2, left + cardHorizontalPadding + date1Width, top + cardVerticalPadding + dateTopPadding, hideDateTextPaint);
        } else {
            if (animateDate1) {
                canvas.drawText(pickerDetails.date1, left + cardHorizontalPadding, top + cardVerticalPadding + dateTopPadding, showDateTextPaint);
                canvas.drawText(oldPickerDetails.date1, left + cardHorizontalPadding + date1Width, top + cardVerticalPadding + dateTopPadding, hideDateTextPaint);
            } else {
                canvas.drawText(pickerDetails.date1, left + cardHorizontalPadding + date1Width, top + cardVerticalPadding + dateTopPadding, dateTextPaint);
            }
            if (animateDate2) {
                canvas.drawText(pickerDetails.date2, left + cardHorizontalPadding + date1Width + paddingBetweenDates, top + cardVerticalPadding + dateTopPadding, showDateTextPaint);
                canvas.drawText(oldPickerDetails.date2, left + cardHorizontalPadding + date1Width + paddingBetweenDates + date2Width, top + cardVerticalPadding + dateTopPadding, hideDateTextPaint);
            } else {
                canvas.drawText(df2.format(date), left + cardHorizontalPadding + date1Width + paddingBetweenDates + date2Width, top + cardVerticalPadding + dateTopPadding, dateTextPaint);
            }
        }
    }


    float getScreenX(long x) {
        return (((float) (x - minX) / (maxX - minX)) * width);
    }

    float getScreenY(double y) {
        return (float) ((height) - (y - scene.getMinY()) / (scene.getMaxY() - scene.getMinY()) * height) + topPadding;
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        width = right - left;
        height = bottom - top - topPadding;
    }

    private PickerDetails getPickDetails(float screenX, boolean exect) {
        double x = (double) scene.getMinX() + (((double) screenX / (double) width)) * ((double) scene.getMaxX() - (double) scene.getMinX());

        long x1 = 0, x2 = 0;
        int position1 = 0, position2 = 0;
        for (int i = 0; i < xArray.length - 1; i++) {
            x1 = xArray[i];
            x2 = xArray[i + 1];
            if (x >= x1 && x <= x2) {
                position1 = i;
                position2 = i + 1;
                break;
            }
        }

        double prcnt = (x - x1) / (double) (x2 - x1);
        if (prcnt > 0.5f) {
            pickerDetails.closestXPosition = position2;
        } else {
            pickerDetails.closestXPosition = position1;
        }
        if (chart.getChartLines().get(0).getType() == 1) {
            pickerDetails.maxValue = Double.MIN_VALUE;
        } else {
            pickerDetails.maxValue = 0;
        }
        if (exect) {
            pickerDetails.x = (long) (x1 + prcnt * ((double) x2 - (double) x1));
            pickerDetails.values = new double[chart.getChartLines().size()];
            for (int i = 0; i < chart.getChartLines().size(); i++) {
                pickerDetails.values[i] = chart.getChartLines().get(i).getY()[position1] + (chart.getChartLines().get(i).getY()[position2] - chart.getChartLines().get(i).getY()[position1]) * prcnt;
                if (chart.getChartLines().get(0).getType() == 1) {
                    if (pickerDetails.maxValue < pickerDetails.values[i]) {
                        pickerDetails.maxValue = pickerDetails.values[i];
                    }
                } else {
                    pickerDetails.maxValue += pickerDetails.values[i];
                }
            }
        } else {
            pickerDetails.x = xArray[pickerDetails.closestXPosition];
            pickerDetails.values = new double[chart.getChartLines().size()];
            for (int i = 0; i < chart.getChartLines().size(); i++) {
                pickerDetails.values[i] = chart.getChartLines().get(i).getY()[pickerDetails.closestXPosition];
                if (chart.getChartLines().get(0).getType() == 1) {
                    if (pickerDetails.maxValue < pickerDetails.values[i]) {
                        pickerDetails.maxValue = pickerDetails.values[i];
                    }
                } else {
                    pickerDetails.maxValue += pickerDetails.values[i];
                }
            }
        }
        return pickerDetails;
    }

    public void setTopPadding(float topPadding) {
        this.topPadding = topPadding;
    }

    public void setHorizontalPadding(int horizontalPadding) {
        this.horizontalPadding = horizontalPadding;
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (this.gestureDetector.onTouchEvent(event)) {
            return true;
        }
        return super.onTouchEvent(event);
    }

    public interface PickListener {
        void pick(PickerDetails pickerDetails);

        void clearPick();
    }

    public void setPickListener(PickListener pickListener) {
        this.pickListener = pickListener;
    }
}
