package com.adi.lib.chart;

import android.content.Context;
import android.support.v4.content.ContextCompat;
import android.util.AttributeSet;
import android.view.ViewGroup;

import com.adi.lib.R;
import com.adi.lib.chart.entity.Chart;
import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.PickerDetails;
import com.adi.lib.chart.entity.Scene;
import com.adi.lib.utils.AndroidUtils;


public class ChartView extends ViewGroup {

    public static final int THEME_LIGHT = 0;
    public static final int THEME_DARK = 1;
    public static int THEME = THEME_LIGHT;

    public static int ANIMATOR_DURATION = 300;

    private int topPanelHeight;
    private int navigationPanelHeight;
    private int xAxisHeight;
    private int horizontalPadding;
    private int chartHeight;
    private int filterHeight;
    private ChartTopPanel chartTopPanel;
    private BaseChartView chartLinesView;
    private PickerView pickerView;
    private ChartGridView chartGridView;
    private XAxisView xAxisView;
    private BaseChartView navigationChartLineView;
    private SliderView sliderView;
    private ChartFilter chartFilter;
    private ChartLineAnimator chartLineAnimator = new ChartLineAnimator();
    private ChartPercentageAnimator chartPercentageAnimator = new ChartPercentageAnimator();
    private ChartLineAnimator navigationChartLineAnimator = new ChartLineAnimator();
    private ChartGridAnimator greedAnimator = new ChartGridAnimator();
    private Chart chart;
    private String title;
    Scene scene;
    Scene oldScene;
    int width, height;
    boolean viewInitialized;

    ChartCore chartCore = new ChartCore();

    public ChartView(Context context) {
        super(context);
        init();
    }

    public ChartView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public ChartView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        chartHeight = (int) AndroidUtils.dpToPx(getContext(), 296);
        topPanelHeight = (int) AndroidUtils.dpToPx(getContext(), 70);
        navigationPanelHeight = (int) AndroidUtils.dpToPx(getContext(), 48);
        xAxisHeight = (int) AndroidUtils.dpToPx(getContext(), 40);
        horizontalPadding = (int) AndroidUtils.dpToPx(getContext(), 18);
    }

    private void createViews() {
        this.removeAllViews();
        if (chart.getChartLines().get(0).getType() == 1) { //line
            chartLinesView = new ChartLinesView(getContext(), 2, true);
            navigationChartLineView = new ChartLinesView(getContext(), 1, false);
        } else {
            if (chart.getChartLines().get(0).getType() == 3) { //area
                chartLinesView = new ChartPercentageStackedView(getContext(), true);
                navigationChartLineView = new ChartPercentageStackedView(getContext(), false);
            } else { //bar
                chartLinesView = new ChartStackedView(getContext(), true);
                navigationChartLineView = new ChartStackedView(getContext(), false);
            }
        }
        pickerView = new PickerView(getContext(), chart);
        sliderView = new SliderView(getContext());
        chartGridView = new ChartGridView(getContext(), chart);
        xAxisView = new XAxisView(getContext(), chart);
        chartFilter = new ChartFilter(getContext(), chart.getChartLines());
        chartTopPanel = new ChartTopPanel(getContext(), chart);
        addView(xAxisView);
        addView(chartLinesView);
        addView(chartGridView);
        addView(navigationChartLineView);
        addView(sliderView);
        addView(pickerView);
        addView(chartFilter);
        addView(chartTopPanel);
        pickerView.setPickListener(new PickerView.PickListener() {
            @Override
            public void pick(PickerDetails pickerDetails) {
                chartLinesView.pick(pickerDetails);
            }

            @Override
            public void clearPick() {
                chartLinesView.clearPick();
            }
        });
        sliderView.setChangeListener(new SliderView.ChangeListener() {
            @Override
            public void onSliderChanged(long minX, long maxX, boolean widthChanged) {
                if (oldScene == null) {
                    oldScene = scene.copy();
                }
                if (widthChanged) {
                    if (ChartCore.getVisibleChartLinesCount(chart.getChartLines()) != 0) {
                        chartCore.updateSceneY(chart, ChartView.this.scene);
                    }
                }
                chartCore.updateSceneX(chart, ChartView.this.scene, minX, maxX, true);
                chartLinesView.updateScene(ChartView.this.scene);
                xAxisView.updateScene(ChartView.this.scene);
                pickerView.updateScene(ChartView.this.scene);
                pickerView.clear();
            }

            @Override
            public void onStopScroll() {
                if (ChartCore.roundDouble(oldScene.getMinY()) != ChartCore.roundDouble(scene.getMinY())
                        || ChartCore.roundDouble(oldScene.getMaxY()) != ChartCore.roundDouble(scene.getMaxY())) {
                    greedAnimator.animate(chartGridView, oldScene, scene, chart);
                    oldScene = scene.copy();
                }
            }
        });

        chartLineAnimator.setAnimationListener(scene -> {
            if (ChartCore.getVisibleChartLinesCount(chart.getChartLines()) != 0) {
                chartCore.updateSceneY(chart, this.scene);
                chartCore.updateSceneX(chart, this.scene, this.scene.getMinX(), this.scene.getMaxX(), false);
            }
        });

        chartPercentageAnimator.setAnimationListener(scene -> {
            chartCore.updateSceneY(chart, scene);
            chartCore.updateSceneX(chart, scene, scene.getMinX(), scene.getMaxX(), false);
            this.scene = scene;
        });
        chartFilter.setFilterListener(new ChartFilter.FilterListener() {
            @Override
            public void hide(int position) {
                ChartView.this.hide(position);
            }

            @Override
            public void show(int position) {
                ChartView.this.show(position);
            }
        });
        applyTheme();
    }

    private void applyTheme() {
        if (ChartView.THEME == THEME_DARK) {
            this.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.darkChartViewBackgroundColor));
        } else {
            this.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.chartViewBackgroundColor));
        }
    }

    private void initializeViews() {
        this.scene = chartCore.initializeScene(chart);
        chartLinesView.updateChartLines(chart.getChartLines());
        navigationChartLineView.updateChartLines(chart.getChartLines());
        navigationChartLineView.updateScene(scene.copy());
        sliderView.updateChart(chart.getChartLines(), scene.copy());
        chartGridView.updateChartLines(chart);
        chartGridView.updateScene(scene, chart);
        pickerView.updateScene(this.scene);
        chartTopPanel.drawTitle(title);
        chartTopPanel.drawShadow();

    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        if (chart.getChartLines() != null) {
            chartTopPanel.layout(0, 0, width, topPanelHeight);
            chartLinesView.setHorizontalPadding(horizontalPadding);
            chartLinesView.setTopPadding(topPanelHeight);
            chartLinesView.layout(0, 0, width, chartHeight + topPanelHeight);
            chartGridView.setTopPadding(topPanelHeight);
            chartGridView.layout(horizontalPadding, 0, width - horizontalPadding, chartHeight + topPanelHeight);
            pickerView.setTopPadding(topPanelHeight);
            pickerView.setHorizontalPadding(horizontalPadding);
            pickerView.layout(0, 0, width, chartHeight + topPanelHeight);
            xAxisView.layout(0, topPanelHeight + chartHeight, width, topPanelHeight + chartHeight + xAxisHeight);

            navigationChartLineView.layout(horizontalPadding,
                    chartHeight + topPanelHeight + xAxisHeight, width - horizontalPadding, chartHeight + topPanelHeight + xAxisHeight + navigationPanelHeight);

            sliderView.layout(horizontalPadding,
                    chartHeight + topPanelHeight + xAxisHeight, width - horizontalPadding, chartHeight + topPanelHeight + xAxisHeight + navigationPanelHeight);
            chartFilter.layout(0, topPanelHeight + chartHeight + xAxisHeight + navigationPanelHeight, width, topPanelHeight + chartHeight + xAxisHeight + navigationPanelHeight + filterHeight);
            if (!viewInitialized) {
                initializeViews();
                viewInitialized = true;
            }
        }

    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        filterHeight = (int) chartFilter.calculate(width);
        width = MeasureSpec.getSize(widthMeasureSpec);
        height = topPanelHeight + chartHeight + xAxisHeight + navigationPanelHeight + filterHeight;

        int tpWidth = MeasureSpec.makeMeasureSpec(width, MeasureSpec.EXACTLY);
        int tpHeight = MeasureSpec.makeMeasureSpec(topPanelHeight, MeasureSpec.EXACTLY);
        chartTopPanel.measure(tpWidth, tpHeight);
        setMeasuredDimension(width, height);
    }


    public void updateChart(String title, Chart chart) {
        this.chart = chart;
        this.title = title;
        createViews();
    }

    public void hide(int index) {
        pickerView.refresh();
        for (ChartLine chartLine : chart.getChartLines()) {
            chartLine.setAnimationInProcess(false);
        }
        if (chart.isPercentage()) {
            chartPercentageAnimator.hide(chartLinesView, navigationChartLineView, chart.getChartLines().get(index), true, true);
        } else {
            Scene newScene = chartCore.initializeScene(chart);
            boolean animateScale = ChartCore.getVisibleChartLinesCount(chart.getChartLines()) != 0;
            navigationChartLineAnimator.hide(navigationChartLineView, newScene, chart.getChartLines().get(index), animateScale, true);
            Scene newScene2 = scene.copy();
            chartCore.updateSceneY(chart, newScene2);
            chartCore.updateSceneX(chart, newScene2, scene.getMinX(), scene.getMaxX(), true);
            chartLineAnimator.hide(chartLinesView, newScene2, null, animateScale, false);
            if (ChartCore.getVisibleChartLinesCount(chart.getChartLines()) == 0) {
                greedAnimator.disappear(chartGridView, scene, chart);
            } else {
                greedAnimator.animate(chartGridView, scene, newScene2, chart);
            }
        }
    }

    public void show(int index) {
        pickerView.refresh();
        for (ChartLine chartLine : chart.getChartLines()) {
            chartLine.setAnimationInProcess(false);
        }
        if (chart.isPercentage()) {
            chartPercentageAnimator.show(chartLinesView, navigationChartLineView, chart.getChartLines().get(index), true, true);
        } else {
            Scene newScene = chartCore.initializeScene(chart);
            boolean animateScale = ChartCore.getVisibleChartLinesCount(chart.getChartLines()) != 1;
            navigationChartLineAnimator.show(navigationChartLineView, newScene, chart.getChartLines().get(index), animateScale, true);
            Scene newScene2 = scene.copy();
            chartCore.updateSceneY(chart, newScene2);
            chartCore.updateSceneX(chart, newScene2, scene.getMinX(), scene.getMaxX(), true);
            chartLineAnimator.show(chartLinesView, newScene2, null, animateScale, false);
            if (ChartCore.getVisibleChartLinesCount(chart.getChartLines()) == 1) {
                greedAnimator.appear(chartGridView, newScene2, chart);
            } else {
                greedAnimator.animate(chartGridView, scene, newScene2, chart);
            }
        }
    }


    public void changeTheme() {
        if (this.chart.getChartLines() != null && title != null) {
            sliderView.applyTheme();
            sliderView.invalidate();
            chartLinesView.applyTheme();
            chartLinesView.invalidate();
            chartGridView.applyTheme();
            chartGridView.invalidate();
            chartTopPanel.applyTheme();
            chartTopPanel.drawShadow();
            chartTopPanel.invalidate();
            xAxisView.applyTheme();
            xAxisView.invalidate();
            pickerView.applyTheme();
            pickerView.invalidate();
            this.applyTheme();
            this.invalidate();
        }
    }

}
