package com.adi.lib.chart;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;

import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.PickerDetails;
import com.adi.lib.chart.entity.Scene;

import java.util.List;


@SuppressLint("ViewConstructor")
public class ChartStackedView extends BaseChartView {
    private Paint paint = new Paint();
    private float[][] points;
    float minY;
    double lastPosition;


    public ChartStackedView(Context context, boolean chartMode) {
        super(context, chartMode);
        init();
    }

    private void init() {
        paint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        paint.setStyle(Paint.Style.FILL);
    }

    @Override
    void updateChartLines(List<ChartLine> chartLines) {
        super.updateChartLines(chartLines);
    }

    @Override
    void updateScene(Scene newScene) {
        super.updateScene(newScene);
        minY = height + topPadding;
        points = new float[chartLines.size()][(xArray.length) * 4];

        calculatePoints();
        postInvalidate(0, 0, (int) width, (int) height);
    }

    @Override
    void animateChart(float maxY, float minY) {
        this.scene.setMaxY(maxY);
    }

    @Override
    public void animateChartPercent(float percent) {
        super.animateChartPercent(percent);
        calculatePointsForAnimation(percent);
        postInvalidate(0, 0, (int) width, (int) height);
    }

    private void calculatePoints() {
        for (i = start; i <= stop; i++) {
            lastPosition = 0;
            for (int k = 0; k < chartLines.size(); k++) {
                if (chartLines.get(k).isVisible()) {
                    points[k][(i - start) * 4 + 3] = getScreenY(lastPosition); //top
                    lastPosition = lastPosition + chartLines.get(k).getY()[i];
                    points[k][(i - start) * 4 + 1] = getScreenY(lastPosition); //move
                    points[k][(i - start) * 4] = getScreenX(xArray[i]); //left
                    points[k][(i - start) * 4 + 2] = getScreenX(xArray[i]); //right
                }
            }
        }
    }

    private void calculatePointsForAnimation(float percent) {
        for (i = start; i <= stop; i++) {
            lastPosition = 0;
            for (int k = 0; k < chartLines.size(); k++) {
                if (chartLines.get(k).isAnimationInProcess()) {
                    points[k][(i - start) * 4 + 3] = getScreenY(lastPosition); //top
                    lastPosition = lastPosition + chartLines.get(k).getY()[i] * percent;
                    points[k][(i - start) * 4 + 1] = getScreenY(lastPosition); //move
                    points[k][(i - start) * 4] = getScreenX(xArray[i]); //left
                    points[k][(i - start) * 4 + 2] = getScreenX(xArray[i]); //right
                } else {
                    if (chartLines.get(k).isVisible()) {
                        points[k][(i - start) * 4 + 3] = getScreenY(lastPosition); //top
                        lastPosition = lastPosition + chartLines.get(k).getY()[i];
                        points[k][(i - start) * 4 + 1] = getScreenY(lastPosition); //move
                    }
                }
            }
        }
    }


    //отодвигаем старт влево для того чтобы не было видно левого отступа
    @Override
    protected void calculateStartAndStopX() {
        super.calculateStartAndStopX();
        int count = (int) (horizontalPadding / gapInPixel);
        start = start - count;
        if (start < 0) start = 0;
        if (stop != xArray.length - 1) {
            stop = stop + 1;
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (chartLines != null && scene != null && scene.getMaximums() != null) {
            paint.setStrokeWidth(gapInPixel);
            for (int i = 0; i < chartLines.size(); i++) {
                paint.setColor(chartLines.get(i).getColor());
                if (animationInProcess) {
                    paint.setAlpha(chartLines.get(i).getAlpha());
                    drawChart(canvas, i);
                } else {
                    paint.setAlpha(255);
                    if (chartLines.get(i).isVisible()) {
                        drawChart(canvas, i);
                    }
                }
            }
        }
    }

    @Override
    public void pick(PickerDetails pickerDetails) {
        super.pick(pickerDetails);
        invalidate();
    }

    private void drawChart(Canvas canvas, int index) {
        if (!picked) {
            canvas.drawLines(points[index], paint);
        } else {
            paint.setAlpha(180);
            canvas.drawLines(points[index], paint);
            paint.setAlpha(255);
            canvas.drawLine(points[index][(pickerDetails.closestXPosition-start) * 4], points[index][(pickerDetails.closestXPosition-start) * 4 + 1], points[index][(pickerDetails.closestXPosition-start) * 4 + 2], points[index][(pickerDetails.closestXPosition-start) * 4 + 3], paint);
        }
    }
}
