package com.adi.lib.chart;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.util.Log;

import com.adi.lib.R;
import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.PickerDetails;
import com.adi.lib.chart.entity.Scene;

import java.util.List;

import static com.adi.lib.chart.ChartView.THEME_DARK;


@SuppressLint("ViewConstructor")
public class ChartLinesView extends BaseChartView {
    private Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG);
    private Paint pickerLinePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
    private Paint circlePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
    private float[][] points;
    private boolean initialized;
    float circleBorderWidth, circleRadius;
    float lineAlpha;

    public ChartLinesView(Context context, int lineWidth, boolean chartMode) {
        super(context, chartMode);
        init(lineWidth);
    }

    private void init(int lineWidth) {
        paint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), lineWidth));
        paint.setStyle(Paint.Style.STROKE);
        paint.setStyle(Paint.Style.FILL);
        circleBorderWidth = AndroidUtils.dpToPx(getContext(), 2);
        circleRadius = AndroidUtils.dpToPx(getContext(), 5);
        circlePaint.setColor(Color.WHITE);
        circlePaint.setStrokeWidth(2);
        circlePaint.setStyle(Paint.Style.STROKE);
        pickerLinePaint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        pickerLinePaint.setStrokeCap(Paint.Cap.ROUND);
        pickerLinePaint.setStyle(Paint.Style.STROKE);
        pickerLinePaint.setAlpha(255 / 10);
        applyTheme();
    }


    @Override
    public void applyTheme() {
        super.applyTheme();
        if (ChartView.THEME == THEME_DARK) {
            pickerLinePaint.setColor(getResources().getColor(R.color.darkGridHorizontalLine));
            lineAlpha = 0.1f;
        } else {
            pickerLinePaint.setColor(getResources().getColor(R.color.gridHorizontalLine));
            lineAlpha = 0.1f;
        }
    }

    @Override
    void updateScene(Scene newScene) {
        super.updateScene(newScene);
        if (chartMode) {
            points = new float[chartLines.size()][(stop - start + 1) * 4];
            calculatePoints();
            postInvalidate(0, 0, (int) width, (int) height);
        } else {//для слайдра не нужно пересчитывать точки при движении
            if (!initialized) {
                points = new float[chartLines.size()][(stop - start + 1) * 4];
                calculatePoints();
                postInvalidate(0, 0, (int) width, (int) height);
                initialized = true;
            }
        }
    }

    @Override
    void animateChart(float maxY, float minY) {
        if (chartMode) {
            Log.d("ADIC", "maxY: " + maxY + " minY: " + minY);
        }
        initialized = false;
        this.scene.setMaxY(maxY);
        this.scene.setMinY(minY);
        calculatePointsForAnimation();
        postInvalidate(0, 0, (int) width, (int) height);
    }

    @Override
    void updateChartLines(List<ChartLine> chartLines) {
        super.updateChartLines(chartLines);
        initialized = false;
    }

    private void calculatePoints() {
        for (j = 0; j < chartLines.size(); j++) {
            for (i = start; i < stop; i++) {
                points[j][(i - start) * 4] = getScreenX(xArray[i]);
                points[j][(i - start) * 4 + 1] = getScreenY(chartLines.get(j).getY()[i]);
                points[j][(i - start) * 4 + 2] = getScreenX(xArray[i + 1]);
                points[j][(i - start) * 4 + 3] = getScreenY(chartLines.get(j).getY()[i + 1]);
            }
        }
    }

    private void calculatePointsForAnimation() {
        for (j = 0; j < chartLines.size(); j++) {
            if (!chartMode && chartLines.get(j).isAnimationInProcess()) {
                //для слайдера не анимируем изменяющуюся линию
            } else {
                for (i = start; i < stop; i++) {
                    points[j][(i - start) * 4 + 1] = getScreenY(chartLines.get(j).getY()[i]);
                    points[j][(i - start) * 4 + 3] = getScreenY(chartLines.get(j).getY()[i + 1]);
                }
            }
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (chartLines != null && scene != null && scene.getMaximums() != null) {
            for (int i = 0; i < chartLines.size(); i++) {
                paint.setColor(chartLines.get(i).getColor());
                if (animationInProcess) {
                    paint.setAlpha(chartLines.get(i).getAlpha());
                    drawChartLine(canvas, i);
                } else {
                    paint.setAlpha(255);
                    if (chartLines.get(i).isVisible()) {
                        drawChartLine(canvas, i);
                    }
                }
            }

            if (picked) {
                pickerLinePaint.setAlpha((int) (255 * lineAlpha));
                canvas.drawLine(getScreenX(pickerDetails.x), topPadding, getScreenX(pickerDetails.x), height + topPadding, pickerLinePaint);
                for (int i = 0; i < chartLines.size(); i++) {
                    if (chartLines.get(i).isVisible()) {
                        fillCircleStrokeBorder(canvas, getScreenX(pickerDetails.x), getScreenY(pickerDetails.values[i]), circleRadius, Color.WHITE, circleBorderWidth, chartLines.get(i).getColor(), circlePaint);
                    }
                }
            }
           /* paint.setAlpha(100);
            float max = (float) ArrayUtils.max(scene.getMaximums());
            float min = (float) ArrayUtils.min(scene.getMinimums());
            for (int i = 0; i < scene.getMaximums().length; i++) {
                float yMax = (float) (height - (scene.getMaximums()[i]-min) / (max - min) * height) + topPadding;
                float yMin = (float) (height - (scene.getMinimums()[i]-min) / (max - min) * height) + topPadding;
                canvas.drawLine(getScreenX(chartLines.get(0).getX()[i]), yMin, getScreenX(chartLines.get(0).getX()[i]), yMax, paint);
            }
            canvas.drawLine(0,getScreenY(scene.getMinY()),width,getScreenY(scene.getMinY()),paint);
            canvas.drawLine(0,getScreenY(scene.getMaxY()),width,getScreenY(scene.getMaxY()),paint);*/
        }
    }

    //отодвигаем старт влево для того чтобы не было видно левого отступа
    @Override
    protected void calculateStartAndStopX() {
        super.calculateStartAndStopX();
        int count = (int) (horizontalPadding / gapInPixel);
        start = start - count - 1;
        if (start < 0) start = 0;
        if (stop != xArray.length - 1) {
            stop = stop + 1;
        }
    }

    @Override
    public void pick(PickerDetails pickerDetails) {
        super.pick(pickerDetails);
        invalidate();
    }

    public static void fillCircleStrokeBorder(
            Canvas c, float cx, float cy, float radius,
            int circleColor, float borderWidth, int borderColor, Paint p) {

        int saveColor = p.getColor();
        p.setColor(circleColor);
        Paint.Style saveStyle = p.getStyle();
        p.setStyle(Paint.Style.FILL);
        c.drawCircle(cx, cy, radius, p);
        if (borderWidth > 0) {
            p.setColor(borderColor);
            p.setStyle(Paint.Style.STROKE);
            float saveStrokeWidth = p.getStrokeWidth();
            p.setStrokeWidth(borderWidth);
            c.drawCircle(cx, cy, radius - (borderWidth / 2), p);
            p.setStrokeWidth(saveStrokeWidth);
        }
        p.setColor(saveColor);
        p.setStyle(saveStyle);
    }

    private void drawChartLine(Canvas canvas, int index) {
        canvas.drawLines(points[index], paint);
    }

}
