package com.adi.lib.chart;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.support.v4.content.ContextCompat;
import android.view.View;

import com.adi.lib.R;
import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.Chart;
import com.adi.lib.chart.entity.GridState;
import com.adi.lib.chart.entity.Scene;

import static com.adi.lib.chart.ChartView.THEME_LIGHT;


@SuppressLint("ViewConstructor")
public class ChartGridView extends View {
    Paint paint = new Paint();
    Paint fontPaint = new Paint();
    Paint scaledFontPaint = new Paint();
    Paint bottomLinePaint = new Paint();
    Chart chart;
    Scene scene;
    int numberBottomPadding;
    float scaledTextRightPadding;
    private int height;
    Rect textBounds = new Rect();
    float topPadding = 0;
    float linePadding;
    float lineStep;
    float gridScaleCoef;
    float fontAlpha;
    int[] gridNumbers;
    GridState grid1;
    GridState grid2;
    GridState scaledGrid1;
    GridState scaledGrid2;

    public ChartGridView(Context context, Chart chart) {
        super(context);
        this.chart = chart;
        init();
    }

    private void init() {
        paint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        paint.setAntiAlias(true);
        paint.setStyle(Paint.Style.STROKE);


        bottomLinePaint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        bottomLinePaint.setAntiAlias(true);
        bottomLinePaint.setStyle(Paint.Style.STROKE);

        fontPaint.setAntiAlias(true);
        fontPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 12));
        fontPaint.setStyle(Paint.Style.FILL);

        scaledFontPaint.setAntiAlias(true);
        scaledFontPaint.setTextSize(AndroidUtils.dpToPx(this.getContext(), 12));
        scaledFontPaint.setStyle(Paint.Style.FILL);
        scaledFontPaint.setTextAlign(Paint.Align.RIGHT);
        numberBottomPadding = (int) AndroidUtils.dpToPx(this.getContext(), 7);
        scaledTextRightPadding = (int) AndroidUtils.dpToPx(this.getContext(), 0);

        applyTheme();
        fontPaint.getTextBounds("1", 0, 1, textBounds);
        linePadding = AndroidUtils.dpToPx(getContext(), 28);
    }

    public void applyTheme() {
        if (ChartView.THEME == THEME_LIGHT) {
            paint.setColor(ContextCompat.getColor(getContext(), R.color.gridHorizontalLine));
            bottomLinePaint.setColor(ContextCompat.getColor(getContext(), R.color.gridHorizontalLineDark));
            if (chart.getChartLines().get(0).getType() == 1) {
                fontPaint.setColor(getResources().getColor(R.color.lineYText));
                fontAlpha = 1.0f;
            } else {
                fontPaint.setColor(getResources().getColor(R.color.barYText));
                fontAlpha = 0.5f;
            }
        } else {
            paint.setColor(ContextCompat.getColor(getContext(), R.color.darkGridHorizontalLine));
            bottomLinePaint.setColor(ContextCompat.getColor(getContext(), R.color.darkGridHorizontalLineDark));
            if (chart.getChartLines().get(0).getType() == 1) {
                fontPaint.setColor(getResources().getColor(R.color.darkLineYText));
                fontAlpha = 0.6f;
            } else {
                fontPaint.setColor(getResources().getColor(R.color.darkBarYText));
                fontAlpha = 0.5f;
            }
        }
    }

    public void updateChartLines(Chart chart) {
        this.chart = chart;
    }

    public void updateScene(Scene newScene, Chart chart) {
        this.scene = newScene;
        if (chart.isyScaled()) {
            if (chart.getChartLines().get(scene.getNotScaledYIndex()).isVisible()) {
                if (grid1 == null) {
                    grid1 = new GridState();
                }
                grid1.set(scene.getMinY(), scene.getMaxY(), 255, 1.0f, false, chart.getChartLines().get(scene.getNotScaledYIndex()).getColor());
            } else {
                grid1 = null;
            }
            if (chart.getChartLines().get(scene.getScaledYIndex()).isVisible()) {
                if (scaledGrid1 == null) {
                    scaledGrid1 = new GridState();
                }
                scaledGrid1.set(scene.getMinY(), scene.getMaxY(), 255, 1.0f, true, chart.getChartLines().get(scene.getScaledYIndex()).getColor());
            } else {
                scaledGrid1 = null;
            }
        } else {
            if (grid1 == null) {
                grid1 = new GridState();
            }
            grid1.set(scene.getMinY(), scene.getMaxY(), 255, 1.f, false);
            grid1.greedCenter = 1;
        }
        invalidate();
    }

    public void update(GridState grid1, GridState grid2, GridState scaledGrid1, GridState scaledGrid2) {
        this.grid1 = grid1;
        this.grid2 = grid2;
        this.scaledGrid1 = scaledGrid1;
        this.scaledGrid2 = scaledGrid2;
        invalidate();
    }

    public void update(GridState sourceGrid, GridState targetGrid) {
        this.grid1 = sourceGrid;
        this.grid2 = targetGrid;
        invalidate();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (chart.isPercentage()) {
            gridScaleCoef = 4.f;
        } else {
            gridScaleCoef = 5.6f;
        }
        if (grid1 != null) {
            drawGrid(canvas, grid1);
        }
        if (grid2 != null) {
            drawGrid(canvas, grid2);
        }
        if (scaledGrid1 != null) {
            drawGrid(canvas, scaledGrid1);
        }
        if (scaledGrid2 != null) {
            drawGrid(canvas, scaledGrid2);
        }
    }

    private void drawGrid(Canvas canvas, GridState grid) {
        if (grid.color != -1) {
            fontPaint.setColor(grid.color);
            fontPaint.setAlpha(grid.alpha);
            if (grid.scaled) {
                scaledFontPaint.setColor(grid.color);
                scaledFontPaint.setAlpha(grid.alpha);
            }
        } else {
            fontPaint.setAlpha((int) (grid.alpha * fontAlpha));
        }
        if (grid.scaled) {
            gridNumbers = ChartCore.getGridNumbers(grid.minY / scene.getyScaleFactor(), grid.maxY / scene.getyScaleFactor(), gridScaleCoef);
        } else {
            gridNumbers = ChartCore.getGridNumbers(grid.minY, grid.maxY, gridScaleCoef);
        }
        paint.setAlpha((int) (grid.alpha * 0.1f));
        lineStep = (float) ((grid.maxY - grid.minY) / gridScaleCoef);
        double topScalePadding = 0;
        if (grid.greedCenter == 0) { //относительно центра
            topScalePadding = (height * grid.scaleFactor - height) / 2;
        } else if (grid.greedCenter == -1) { //относительно вершины
            topScalePadding = height * grid.scaleFactor - height;
        } else {
            topScalePadding = 0;
        }
        if (grid.scaled && chart.getChartLines().get(scene.getNotScaledYIndex()).isVisible()) {
            //если осовной график виден не нужно рисовать линии для scaled
        } else {
            for (int i = 0; i < (int) gridScaleCoef + 1; i++) {
                canvas.drawLine(0, (float) (getScreenY(lineStep * i, grid) + topScalePadding), getWidth(), (float) (getScreenY(lineStep * i, grid) + topScalePadding), paint);
            }
        }

        if (!grid.scaled) {
            for (int i = 0; i < (int) gridScaleCoef + 1; i++) {
                canvas.drawText(ChartCore.numberToString(gridNumbers[i]), 0, (float) (getScreenY(lineStep * i, grid) + topScalePadding) - numberBottomPadding, fontPaint);
            }
        } else {
            for (int i = 0; i < (int) gridScaleCoef + 1; i++) {
                canvas.drawText(ChartCore.numberToString(gridNumbers[i]) + "", getWidth() - scaledTextRightPadding, (float) (getScreenY(lineStep * i, grid) + topScalePadding) - numberBottomPadding, scaledFontPaint);
            }
        }
    }

    private float getScreenY(double y, GridState grid) {
        return (float) ((double) (height) - (y / (grid.maxY - grid.minY) * grid.scaleFactor * (double) (height))) + topPadding;
    }


    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        height = (int) (bottom - top - topPadding);
    }

    public Scene getScene() {
        return scene;
    }

    public void setTopPadding(float topPadding) {
        this.topPadding = topPadding;
    }
}
