package com.adi.lib.chart;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.support.v4.content.ContextCompat;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.View;

import com.adi.lib.R;
import com.adi.lib.utils.AndroidUtils;
import com.adi.lib.chart.entity.ChartLine;

import java.util.List;

@SuppressLint("ViewConstructor")
public class ChartFilter extends View {
    float iconSize;
    float textPaddingTop;
    float checkHorizontalPadding;
    float textPaddingLeft;
    float textPaddingRight;
    float cornerRadius;
    float fontHeight;
    float paddingBetweenChecks;
    float rawSpaceWidth;
    float viewPadding;
    float clearBorder;
    float checkHeight;
    List<ChartLine> chartLines;
    RectF rectF = new RectF();
    Drawable icon;
    Rect textBounds = new Rect();

    RectF[] checkRects;

    float[] checkTextWidths;
    float[] checkTextHeights;

    float xPosition;
    int currentRawNumber;
    float lastXPosition;

    Paint paint = new Paint(Paint.ANTI_ALIAS_FLAG);
    Paint clear = new Paint(Paint.ANTI_ALIAS_FLAG);
    Paint fontPaint = new Paint(Paint.ANTI_ALIAS_FLAG);

    private FilterListener filterListener;

    GestureDetector gestureDetector;

    public ChartFilter(Context context, List<ChartLine> chartLines) {
        super(context);
        this.setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        icon = ContextCompat.getDrawable(getContext(), R.drawable.ic_check_white_24dp);
        clear.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
        paint.setStrokeWidth(AndroidUtils.dpToPx(getContext(), 1));
        paint.setStyle(Paint.Style.STROKE);
        paint.setStyle(Paint.Style.FILL);
        cornerRadius = AndroidUtils.dpToPx(getContext(), 20);
        fontHeight = AndroidUtils.dpToPx(this.getContext(), 16);
        fontPaint.setTextSize(fontHeight);
        fontPaint.setStyle(Paint.Style.FILL);
        fontPaint.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.NORMAL));
        textPaddingTop = (int) AndroidUtils.dpToPx(getContext(), 1);
        iconSize = (int) AndroidUtils.dpToPx(getContext(), 20);
        checkHeight = (int) AndroidUtils.dpToPx(getContext(), 41);
        checkHorizontalPadding = (int) AndroidUtils.dpToPx(getContext(), 11);
        textPaddingLeft = (int) AndroidUtils.dpToPx(getContext(), 4);
        textPaddingRight = (int) AndroidUtils.dpToPx(getContext(), 8);
        paddingBetweenChecks = (int) AndroidUtils.dpToPx(getContext(), 6);
        viewPadding = AndroidUtils.dpToPx(getContext(), 18);
        clearBorder = AndroidUtils.dpToPx(getContext(), 2);
        this.chartLines = chartLines;
        gestureDetector = new GestureDetector(new GestureDetector.SimpleOnGestureListener() {
            @Override
            public boolean onDown(MotionEvent e) {
                click(e.getX(), e.getY());
                return true;
            }

        });
    }

    private void click(float x, float y) {
        for (int i = 0; i < checkRects.length; i++) {
            if (checkRects[i].contains(x, y)) {
                if (chartLines.get(i).isVisible()) {
                    chartLines.get(i).setVisible(false);
                    filterListener.hide(i);
                } else {
                    chartLines.get(i).setVisible(true);
                    filterListener.show(i);
                }
                break;
            }
        }
        invalidate();
    }

    /**
     * calculate all view parameters and return view height
     *
     * @param width of filter view
     */
    public float calculate(float width) {
        rawSpaceWidth = width - viewPadding * 2;
        checkTextWidths = new float[chartLines.size()];
        checkTextHeights = new float[chartLines.size()];
        checkRects = new RectF[chartLines.size()];
        currentRawNumber = 0;
        lastXPosition = viewPadding;
        for (int i = 0; i < chartLines.size(); i++) {
            checkRects[i] = new RectF();
            fontPaint.getTextBounds(chartLines.get(i).getName(), 0, chartLines.get(i).getName().length, textBounds);
            checkTextWidths[i] = textBounds.width();
            checkTextHeights[i] = textBounds.height();
            if(i==0){
                xPosition = lastXPosition;
            }else {
                xPosition = lastXPosition + paddingBetweenChecks;
            }
            if (xPosition + getFullCheckWidth(textBounds.width()) > viewPadding + rawSpaceWidth) {
                currentRawNumber++;
                xPosition = viewPadding;
            }
            lastXPosition = xPosition + getFullCheckWidth(textBounds.width());

            checkRects[i].left = xPosition;
            checkRects[i].right = checkRects[i].left + getFullCheckWidth(textBounds.width());
            checkRects[i].top = currentRawNumber * checkHeight + currentRawNumber * paddingBetweenChecks + viewPadding;
            checkRects[i].bottom = checkRects[i].top + checkHeight;
        }
        return viewPadding * 2f + currentRawNumber * checkHeight + checkHeight + currentRawNumber * paddingBetweenChecks;
    }

    @Override
    protected void onDraw(Canvas canvas) {
        for (int i = 0; i < chartLines.size(); i++) {
            drawCheck(canvas, chartLines.get(i), checkRects[i], i, chartLines.get(i).isVisible());
        }
    }

    private void drawCheck(Canvas canvas, ChartLine chartLine, RectF checkRect, int index, boolean checked) {
        paint.setColor(chartLine.getColor());
        canvas.drawRoundRect(checkRect, cornerRadius, cornerRadius, paint);

        if (!checked) {
            fontPaint.setColor(chartLine.getColor());
            rectF.set(checkRect.left + clearBorder, checkRect.top + clearBorder, checkRect.right - clearBorder, checkRect.bottom - clearBorder);
            canvas.drawRoundRect(rectF, cornerRadius, cornerRadius, clear);
            canvas.drawText(chartLine.getName(), 0, chartLine.getName().length, checkRect.left + ((checkRect.right - checkRect.left) - checkTextWidths[index]) / 2f, checkRect.top + (checkHeight - checkTextHeights[index]) / 2f + checkTextHeights[index] - textPaddingTop, fontPaint);
        } else {
            fontPaint.setColor(Color.WHITE);
            icon.setBounds((int) checkRect.left + (int) checkHorizontalPadding, (int) checkRect.top + (int) ((checkHeight - iconSize) / 2f), (int) checkRect.left + (int) iconSize + (int) checkHorizontalPadding, (int) checkRect.top + (int) iconSize + (int) ((checkHeight - iconSize) / 2f));
            icon.draw(canvas);
            canvas.drawText(chartLine.getName(), 0, chartLine.getName().length, checkRect.left + (checkHorizontalPadding + iconSize + textPaddingLeft), checkRect.top + (checkHeight - checkTextHeights[index]) / 2f + checkTextHeights[index] - textPaddingTop, fontPaint);
        }
    }

    private float getFullCheckWidth(float textWidth) {
        return checkHorizontalPadding * 2f + iconSize + textPaddingLeft + textPaddingRight + textWidth;
    }

    public void setFilterListener(FilterListener filterListener) {
        this.filterListener = filterListener;
    }

    public interface FilterListener {
        void hide(int position);

        void show(int position);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        getParent().getParent().getParent().getParent().requestDisallowInterceptTouchEvent(true);
        if (this.gestureDetector.onTouchEvent(event)) {
            return true;
        }
        return super.onTouchEvent(event);
    }
}
