package com.adi.lib.chart;

import com.adi.lib.utils.ArrayUtils;
import com.adi.lib.chart.entity.Chart;
import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.Extremum;
import com.adi.lib.chart.entity.Interval;
import com.adi.lib.chart.entity.Scene;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static com.adi.lib.utils.ArrayUtils.getDelta;
import static com.adi.lib.utils.ArrayUtils.makeInterpolation;
import static com.adi.lib.utils.ArrayUtils.max;
import static com.adi.lib.utils.ArrayUtils.min;
import static com.adi.lib.utils.ArrayUtils.round;

class ChartCore {

    private static int MIN_DISTANCE_BETWEEN_PEAKS = 20;
    public static int CHART_PARTS_COUNT = 6;

    Scene initializeScene(Chart chart) {
        Scene scene = new Scene();
        long minX = chart.getChartLines().get(0).getX()[0];
        long maxX = chart.getChartLines().get(0).getX()[chart.getChartLines().get(0).getX().length - 1];
        if (chart.isyScaled()) {
            scale(scene, chart.getChartLines());
        }
        scene.setMinSliderWidth((maxX - minX) / (long) CHART_PARTS_COUNT);
        updateSceneY(chart, scene);
        updateSceneX(chart, scene, chart.getChartLines().get(0).getX()[0], chart.getChartLines().get(0).getX()[chart.getChartLines().get(0).getX().length - 1], false);
        return scene;
    }

    private void scale(Scene scene, List<ChartLine> chartLines) {
        double[] chartMaxYs = new double[chartLines.size()];
        for (int i = 0; i < chartLines.size(); i++) {
            chartMaxYs[i] = ArrayUtils.max(chartLines.get(i).getY());
        }
        int maxPosition = 0;
        double max = chartMaxYs[0];
        for (int i = 1; i < chartMaxYs.length; i++) {
            if (chartMaxYs[i] > max) {
                maxPosition = i;
                max = chartMaxYs[i];
            }
        }
        for (int i = 0; i < chartLines.size(); i++) {
            if (i != maxPosition) {
                float scaleFactor = (float) (max / chartMaxYs[i]);
                scene.setyScaleFactor(scaleFactor);
                scaleChartLine(chartLines.get(i), scaleFactor);
                scene.setScaledYIndex(i);
            } else {
                scene.setNotScaledYIndex(i);
            }
        }
    }

    private void scaleChartLine(ChartLine chartLine, float scaleFactor) {
        for (int i = 0; i < chartLine.getY().length; i++) {
            chartLine.getY()[i] = chartLine.getY()[i] * scaleFactor;
        }
    }

    /**
     * расчет максимальных и минимальных значений Y на всем промежутке
     * нужно вызывать когда меняется кол-во графиков
     *
     * @param chart
     * @param scene
     */
    void updateSceneY(Chart chart, Scene scene) {
        if(ChartCore.getVisibleChartLinesCount(chart.getChartLines()) != 0) {
            calculateYMaxMin(chart, scene, 0); //maximum
            calculateYMaxMin(chart, scene, 1); //minimum
        }
    }

    /**
     * расчет mixY и maxY, а также значений для анимации greed
     * нужно вызывать когда меняется положение слайдера
     *
     * @param chart
     * @param scene
     */
    void updateSceneX(Chart chart, Scene scene, long minX, long maxX, boolean calculateXAxis) {
        scene.setMinX(minX);
        scene.setMaxX(maxX);
        if (scene.getMaximums() != null) {
            long[] xArray = chart.getChartLines().get(0).getX();
            double[] yArray = scene.getMaximums();
            if (!chart.isPercentage()) {
                calculateYAxis(scene, xArray, yArray, minX, maxX);
            } else {
                scene.setMaxY(100);
            }
            if (calculateXAxis)
                calculateXAxis(scene, xArray, minX, maxX);
        }
    }

    /**
     * трансформируем массив x в массив с равными промежутками для анимации оси x
     *
     * @param xArray
     * @param minX
     * @param maxX
     * @return
     */
    private long[] getIntervals(long[] xArray, long minX, long maxX) {
        double interval = ((double) (maxX - minX) / 6d);
        int count = (int) Math.ceil((double) (xArray[xArray.length - 1] - xArray[0]) / interval) + 1;
        long[] intervals = new long[count];
        long nextVal;
        for (int i = 0; i < count; i++) {
            nextVal = (long) (xArray[xArray.length - 1] - i * interval);
            intervals[count - i - 1] = nextVal;
        }
        return intervals;
    }

    private long[] getXChangePoints(long[] xArray, long minSliderWidth) {
        int count = (int) Math.ceil((double) (xArray[xArray.length - 1] - xArray[0]) / (double) minSliderWidth);
        long[] points = new long[count + 1];
        long nextVal;
        for (int i = 0; i <= count; i++) {
            nextVal = (i * minSliderWidth);
            points[i] = nextVal;
        }
        return points;
    }


    /**
     * расчет значений для оси x (даты)
     *
     * @param scene
     * @param xArray
     * @param minX
     * @param maxX
     */
    private void calculateXAxis(Scene scene, long[] xArray, long minX, long maxX) {
        if (scene.getXChangePoints() == null) {
            scene.setXChangePoints(getXChangePoints(xArray, scene.getMinSliderWidth()));
        }
        if (scene.getXArray() == null) {
            scene.setXArray(getIntervals(xArray, xArray[xArray.length - 1] - scene.getMinSliderWidth(), xArray[xArray.length - 1]));
        }
        long[] intervals = scene.getXArray();
        long width = maxX - minX;

        long[] changePoints = scene.getXChangePoints();
        long p1 = 0, p2 = 0;
        for (int i = 0; i < changePoints.length - 1; i++) {
            p1 = changePoints[i];
            p2 = changePoints[i + 1];
            if (width > p1 && width <= p2) {
                break;
            }
        }
        float persent;
        if (p1 == 0) {
            persent = 1.0f;
        } else {
            persent = 1 - (float) (((double) width - (double) p1) / ((double) p2 - (double) p1));
        }

        int count, idx;
        long[] newIntervals;
        for (int i = 0; i < changePoints.length; i++) {
            if (changePoints[i] > scene.getMinSliderWidth() && changePoints[i] <= width) {
                count = 0;
                for (int j = 0; j < intervals.length; j++) {
                    if (j % 2 == 0) {
                        count++;
                    }
                }
                newIntervals = new long[count];
                idx = 0;
                for (int j = 0; j < intervals.length; j++) {
                    if (j % 2 == 0) {
                        newIntervals[count - idx - 1] = intervals[intervals.length - 1 - j];
                        idx++;
                    }
                }

                if (i >= (float) changePoints.length / 2f) {
                    persent = 1f;
                }
                if (i < (float) changePoints.length / 2f + 1f) {
                    intervals = newIntervals;
                }
            }

        }


        scene.setDatePercent(persent);

        float length = (float) intervals.length / 2;
        long[] scene1 = new long[(int) Math.floor(length)];
        long[] scene2 = new long[(int) Math.ceil(length)];
        int idx1 = 0, idx2 = 0;
        for (int i = 0; i < intervals.length; i++) {
            if (i % 2 == 0) {
                scene2[idx2] = intervals[i];
                idx2++;
            } else {
                scene1[idx1] = intervals[i];
                idx1++;
            }
        }

        if (scene1[scene1.length - 1] == xArray[xArray.length - 1]) {
            scene.setDates1(scene2);
            scene.setDates2(scene1);
        } else {
            scene.setDates1(scene1);
            scene.setDates2(scene2);
        }

    }

    /**
     * расчет значений для оси Y
     */
    private void calculateYAxis(Scene scene, long[] xArray, double[] yArray, long minX, long maxX) {
        float maxY = (float) ArrayUtils.getMaxYFromXRange(xArray, scene.getMaximums(), minX, maxX);
        float minY = (float) ArrayUtils.getMinYFromXRange(xArray, scene.getMinimums(), minX, maxX);
        scene.setMinY(minY);
        scene.setMaxY(maxY);
    }


    private void setMinY(Scene scene, double[] yArray, int start, int stop) {
        scene.setMinY(min(Arrays.copyOfRange(yArray, start, stop)));
    }


    /**
     * расчет максимальных или минимальных значений y на всем протяжении графика
     *
     * @param chart
     * @param scene
     * @param type  0 - maximums, 1 - minimums
     */
    private void calculateYMaxMin(Chart chart, Scene scene, int type) {
        double[] yArray;
        if (type == 0) {
            yArray = getGlobalYMaximums(chart);
        } else {
            yArray = getGlobalYMinimums(chart);
        }
        if (!chart.isPercentage()) {
            extremumFilter(chart, yArray, type);
            List<Interval> intervals = getIntervals(yArray);
            if (type == 0) {
                scene.setMaxIntervals(intervals);
            } else {
                scene.setMinIntervals(intervals);
            }
            noiseFilter(yArray, intervals, type);
        } else {
            scene.setMaxIntervals(new ArrayList<>());
            scene.setMinIntervals(new ArrayList<>());
        }
        if (type == 0) {
            scene.setMaximums(yArray);
        } else {
            scene.setMinimums(yArray);
        }
    }


    private List<Interval> getIntervals(double[] array) {
        double minDeltaThreshold = ArrayUtils.max(array) / 10f;
        double[] delta = getDelta(array);
        round(delta);
        double current = delta[0];
        List<Interval> intervals = new ArrayList<>();
        Interval interval = new Interval();
        interval.start = 0;
        interval.delta = 0;
        for (int i = 1; i < delta.length; i++) {
            if (delta[i] != current) { //position changed
                interval.stop = i - 1;
                intervals.add(interval);
                interval = new Interval();
                interval.start = i;
                interval.delta += delta[i];
            } else {
                interval.delta += delta[i];
            }
            current = delta[i];
            if (i == delta.length - 1) {
                interval.stop = i;
                intervals.add(interval);
            }
        }

        List<Interval> cleanIntervals = new ArrayList<>();
        for (Interval nextInterval : intervals) {
            if (Math.abs(array[nextInterval.stop] - array[nextInterval.start]) > minDeltaThreshold
                    && nextInterval.stop - nextInterval.start > 5
                    && roundDouble(array[nextInterval.start]) != roundDouble(array[nextInterval.stop])) {
                cleanIntervals.add(nextInterval);
            }
        }
        return cleanIntervals;
    }

    /**
     * убирает шум, неровности графика
     *
     * @param array
     * @param intervals
     * @param type
     */
    private void noiseFilter(double[] array, List<Interval> intervals, int type) {
        Interval interval;
        double targetVal;
        int lastPosition = 0;
        for (int i = 0; i < intervals.size(); i++) {
            interval = intervals.get(i);
            if (type == 0) { //maximums
                targetVal = max(array, lastPosition, interval.start);
            } else { //minimums
                targetVal = min(array, lastPosition, interval.start);
            }
            for (int j = lastPosition; j <= interval.start; j++) {
                array[j] = targetVal;
            }
            lastPosition = interval.stop;
            if (i == intervals.size() - 1) {
                if (type == 0) { //maximums
                    targetVal = max(array, interval.stop, array.length - 1);
                } else { //minimums
                    targetVal = min(array, interval.stop, array.length - 1);
                }
                for (int j = interval.stop; j <= array.length - 1; j++) {
                    array[j] = targetVal;
                }
            }
        }

        for (Interval nextInterval : intervals) {
            makeInterpolation(array, nextInterval.start, nextInterval.stop);
        }

        round(array);
    }


    /**
     * проходит окном по массиву и убирает не нужные экстремумы
     *
     * @param windowSize размер слайдера
     * @param yArray
     * @param type       0 - for maximums 1 - for minimums
     */
    private void windowFilter(int windowSize, double[] yArray, int type) {
        double[] result = new double[yArray.length];
        int startPosition = 0;
        int endPosition = windowSize;
        double y; //для расчета максимальных значений ищем max y, для
        while (endPosition <= yArray.length - 1) {
            if (type == 0) { //for maximums
                y = ArrayUtils.max(yArray, startPosition, endPosition);
            } else {
                y = min(yArray, startPosition, endPosition);
            }

            if (startPosition == 0) {
                for (int i = startPosition; i <= endPosition; i++) {
                    result[i] = y;
                }
            } else {
                result[endPosition] = y;
            }
            endPosition++;
            startPosition++;
        }
        System.arraycopy(result, 0, yArray, 0, yArray.length);
    }


    /**
     * фильтрует значения пиков, убирая соседние пики по возрастанию или убыванию
     *
     * @param yArray
     * @param type   0 - for maximums 1 - for minimums
     */
    private void extremumFilter(Chart chart, double[] yArray, int type) {
        List<Extremum> extremums = getExtremums(yArray);
        List<Extremum> sortedExtremums = new ArrayList<>(extremums);
        if (type == 0) {
            Collections.sort(sortedExtremums, (p1, p2) -> Double.compare(p2.value, p1.value));
        } else {
            Collections.sort(sortedExtremums, (p1, p2) -> Double.compare(p1.value, p2.value));
        }

        List<Extremum> cleanExtremums = getCleanExtremums(sortedExtremums, extremums);


        boolean collisionExist = true;
        while (collisionExist) {
            Collections.sort(cleanExtremums, (p1, p2) -> Integer.compare(p1.index, p2.index));
            cleanExtremums.get(0).startPosition = 0;
            cleanExtremums.get(cleanExtremums.size() - 1).endPosition = extremums.size() - 1;

            Extremum p1, p2;
            for (int i = 0; i < cleanExtremums.size() - 1; i++) {
                p1 = cleanExtremums.get(i);
                p2 = cleanExtremums.get(i + 1);
                yArray[p1.index] = p1.value;
                yArray[p2.index] = p2.value;
                makeInterpolation(yArray, p1.index, p2.index);
                if (i == 0) { // first peak
                    for (int j = p1.startPosition; j <= p1.index; j++) {
                        yArray[j] = p1.value;
                    }
                }
                if (i == cleanExtremums.size() - 2) { // last peak
                    for (int j = p2.index; j < p2.endPosition; j++) {
                        yArray[j] = p2.value;
                    }
                }
            }

            collisionExist = false;
            //может быть ситуация, когда пик вылезает из рамок
            //в этом случае добавляем еще экстремум
            if (type == 0) {
                for (int i = 0; i < yArray.length; i++) {
                    for (ChartLine chartLine : chart.getChartLines()) {
                        if (chartLine.isVisible()) {
                            if (!collisionExist) {
                                if (chartLine.getY()[i] > yArray[i]) {
                                    collisionExist = true;
                                    cleanExtremums.add(new Extremum(i, i, i, chartLine.getY()[i]));
                                    break;
                                }
                            } else {
                                break;
                            }
                        }
                    }
                }
            } else {
                for (int i = 0; i < yArray.length; i++) {
                    for (ChartLine chartLine : chart.getChartLines()) {
                        if (chartLine.isVisible()) {
                            if (!collisionExist) {
                                if (chartLine.getY()[i] < yArray[i]) {
                                    collisionExist = true;
                                    cleanExtremums.add(new Extremum(i, i, i, chartLine.getY()[i]));
                                    break;
                                }
                            } else {
                                break;
                            }
                        }
                    }
                }
            }
        }
    }


    /**
     * Возвращает массив с максимальными значениями графиков
     *
     * @param chart
     * @return
     */
    private double[] getGlobalYMaximums(Chart chart) {
        double[] yMaximums = new double[chart.getChartLines().get(0).getX().length];
        if (chart.isStacked() || chart.isPercentage()) {
            double maxY;
            for (int i = 0; i < chart.getChartLines().get(0).getX().length; i++) {
                maxY = 0;
                for (ChartLine chartLine : chart.getChartLines()) {
                    if (chartLine.isVisible()) {
                        maxY += chartLine.getY()[i];
                    }
                }
                yMaximums[i] = maxY;
            }
        } else {
            double maxY;
            for (int i = 0; i < chart.getChartLines().get(0).getX().length; i++) {
                maxY = Double.MIN_VALUE;
                for (ChartLine chartLine : chart.getChartLines()) {
                    if (chartLine.isVisible()) {
                        if (chartLine.getY()[i] > maxY) {
                            maxY = chartLine.getY()[i];
                        }
                    }
                }
                yMaximums[i] = maxY;
            }
        }
        return yMaximums;
    }

    /**
     * Возвращает массив с минимальными значениями графиков
     *
     * @param chart
     * @return
     */
    private double[] getGlobalYMinimums(Chart chart) {
        double[] minimums = new double[chart.getChartLines().get(0).getX().length];
        if (chart.isStacked() || chart.getChartLines().get(0).getType() == 2) {
            for (int i = 0; i < chart.getChartLines().get(0).getX().length; i++) {
                minimums[i] = 0;
            }
        } else {
            double minY;
            for (int i = 0; i < chart.getChartLines().get(0).getX().length; i++) {
                minY = Double.MAX_VALUE;
                for (ChartLine chartLine : chart.getChartLines()) {
                    if (chartLine.isVisible()) {
                        if (chartLine.getY()[i] < minY) {
                            minY = chartLine.getY()[i];
                        }
                    }
                }
                minimums[i] = minY;
            }
        }
        return minimums;
    }


    /**
     * удаляет конфликтующие экстремумы
     *
     * @param sortedPeaks
     * @param peaks
     * @return
     */
    private List<Extremum> getCleanExtremums(List<Extremum> sortedPeaks, List<Extremum> peaks) {
        List<Extremum> cleanExtremums = new ArrayList<>();
        while (sortedPeaks.size() != 0) {
            Extremum peak = sortedPeaks.get(0);
            cleanExtremums.add(peak);
            sortedPeaks.remove(peak);
            for (Extremum nextPeak : peaks) {
                if ((nextPeak.startPosition >= peak.startPosition && nextPeak.startPosition <= peak.endPosition) || (nextPeak.endPosition >= peak.startPosition && nextPeak.endPosition <= peak.endPosition)) {
                    sortedPeaks.remove(nextPeak);
                }
            }
        }
        return cleanExtremums;
    }

    private List<Extremum> getExtremums(double[] y) {
        List<Extremum> peaks = new ArrayList<>();
        for (int i = 0; i < y.length; i++) {
            int startPosition = i - MIN_DISTANCE_BETWEEN_PEAKS;
            if (startPosition < 0) startPosition = 0;
            int endPosition = i + MIN_DISTANCE_BETWEEN_PEAKS;
            if (endPosition > y.length + 1) endPosition = y.length + 1;
            peaks.add(new Extremum(i, startPosition, endPosition, y[i]));
        }
        return peaks;
    }

    static int[] getGridNumbers(double minY, double maxY, float count) {
        minY = roundDouble(minY);
        int[] result = new int[(int) (count + 1)];
        double step = Math.floor((maxY - minY) / (double) count);
        int roundStep = roundDouble(step);
        for (int i = 0; i <= count; i++) {

            result[i] = (int) (minY + (roundStep * i));
        }
        return result;
    }

    static String numberToString(int number){
        if(number > 1000000){
            return number/1000000 + "M";
        }
        return number + "";
    }

    static int[] maxYToGreedNumbers(double maxY, float count) {
        int[] result = new int[(int) (count + 1)];
        double step = Math.floor(maxY / (double) count);
        int roundStep = roundDouble(step);
        for (int i = 0; i <= count; i++) {
            result[i] = (roundStep * i);
        }
        return result;
    }

    static public int roundDouble(double val) {
        int round = 1;
        if (val >= 600000) {
            round = 100000;
        } else if (val >= 60000 && val < 600000) {
            round = 10000;
        } else if (val >= 6000 && val <= 60000) {
            round = 1000;
        } else if (val >= 600 && val <= 6000) {
            round = 100;
        } else if (val >= 100 && val <= 600) {
            round = 10;
        }
        return (((int) val + round / 2) / round * round);
    }

    public static int getVisibleChartLinesCount(List<ChartLine> chartLines) {
        int i = 0;
        for (ChartLine chartLine : chartLines) {
            if (chartLine.isVisible()) {
                i++;
            }
        }
        return i;
    }


}
