package com.adi.lib.chart;

import android.content.Context;
import android.view.View;

import com.adi.lib.chart.entity.ChartLine;
import com.adi.lib.chart.entity.PickerDetails;
import com.adi.lib.chart.entity.Scene;

import java.util.List;

abstract class BaseChartView extends View {
    protected boolean animationInProcess;
    protected Scene scene;
    protected List<ChartLine> chartLines;
    protected long[] xArray;
    protected long gap;
    protected float gapInPixel;
    protected int start, stop;
    protected long minX, maxX;
    protected int j, i;
    protected float height, width;
    protected long horizontalPaddingInPercent;
    protected int horizontalPadding;
    protected float topPadding = 0;
    protected boolean chartMode;
    protected boolean picked;
    protected PickerDetails pickerDetails;

    public void pick(PickerDetails pickerDetails) {
        this.pickerDetails = pickerDetails;
        picked = true;
    }

    public void clearPick() {
        picked = false;
    }

    public void applyTheme() {
    }

    public BaseChartView(Context context, boolean chartMode) {
        super(context);
        this.chartMode = chartMode;
    }

    void updateChartLines(List<ChartLine> chartLines) {
        this.chartLines = chartLines;
        xArray = chartLines.get(0).getX();
        gap = xArray[1] - xArray[0];
    }

    void updateScene(Scene newScene) {
        this.scene = newScene;
        horizontalPaddingInPercent = (long) ((double) (scene.getMaxX() - scene.getMinX()) * (double) horizontalPadding / (double) width);
        minX = scene.getMinX() - horizontalPaddingInPercent;
        maxX = scene.getMaxX() + horizontalPaddingInPercent;
        gapInPixel = getScreenX(xArray[1]) - getScreenX(xArray[0]);
        calculateStartAndStopX();
    }

    abstract void animateChart(float maxY, float minY);

    public void animateChartPercent(float percent) {
    }

    public Scene getScene() {
        return scene;
    }

    public void setAnimationInProcess(boolean animationInProcess) {
        this.animationInProcess = animationInProcess;
    }

    protected void calculateStartAndStopX() {
        start = -1;
        stop = -1;
        if (maxX >= xArray[xArray.length - 1]) {
            stop = xArray.length - 1;
        }
        if (minX <= xArray[0]) {
            start = 0;
        }
        if (start != -1 && stop != -1) {
            return;
        }
        for (i = 0; i < xArray.length; i++) {
            if (xArray[i] + gap >= scene.getMinX()) {
                if (start == -1) {
                    start = i;
                }
            }
            if (xArray[i] >= maxX) {
                if (xArray[i] == maxX && i != xArray.length - 1) {
                    if (stop == -1) {
                        stop = i + 1;
                        break;
                    }
                } else {
                    if (stop == -1) {
                        stop = i;
                        break;
                    }
                }
            }
        }
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        width = right - left;
        height = bottom - top - topPadding;
    }

    public void setTopPadding(float topPadding) {
        this.topPadding = topPadding;
    }

    public void setHorizontalPadding(int horizontalPadding) {
        this.horizontalPadding = horizontalPadding;
    }

    float getScreenX(long x) {
        return (((float) (x - minX) / (maxX - minX)) * width);
    }

    float getScreenX(long x, long minX, long maxX) {
        return (float) (((double) (x - minX) / (double) (maxX - minX)) * (double) width);
    }

    float getScreenY(double y) {
        return (float) ((height) - (y - scene.getMinY()) / (scene.getMaxY() - scene.getMinY()) * height) + topPadding;
    }

    float getScreenY(double y, double minY, double maxY) {
        return (float) ((height) - (y - minY) / (maxY - minY) * height) + topPadding;
    }

}
