package com.acecounter.android.acetm.common.policy;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.controltower.ACEControlTower;
import com.acecounter.android.acetm.common.file.ACEFileUtil;
import com.acecounter.android.acetm.common.http.ACENetworkResult;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.parameter.IACEParameterUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.HttpURLConnection;
import java.util.Map;

public final class ACEPolicyParameterUtil {
    private static final String TAG = ACEPolicyParameterUtil.class.getSimpleName();

    private ACEPolicyParameterUtil() { }

    //region 싱글턴
    private static class Singleton {
        private static final ACEPolicyParameterUtil INSTANCE = new ACEPolicyParameterUtil();
    }

    @NonNull
    public static ACEPolicyParameterUtil getInstance() {
        return ACEPolicyParameterUtil.Singleton.INSTANCE;
    }
    //endregion 싱글턴

    //region Parameter Helpers
    //region Policy
    public synchronized Boolean isEnableByPolicy() {
        String result = ACEPolicyParameters.getInstance().getCpAllow();
        if (TextUtils.isEmpty(result)) {
            return false;
        }
        else {
            return result.equals(ACEStaticConfig.POLICY.FLAG_SDK_ENABLE);
        }
    }

    public synchronized void savePolicy(@NonNull final ACENetworkResult response) {
        ACELog.v(TAG, "policy response: " + response.toString());
        if (response.getResponseCode() == HttpURLConnection.HTTP_OK) {
            ACELog.i(TAG, "Received policy succeed.");
            Map<String, String> _map = response.getResponseHeaders();
            for (String key : _map.keySet()) {
                if (key == null) {
                    continue;
                }

                Context _context = ACECommonStaticConfig.getContext();
                switch (key) {
                    case ACEStaticConfig.POLICY.RESPONSE_SDK_ENABLE:
                        ACEPolicyParameters.getInstance().setCpAllow(_map.get(key));
                        if (!isEnableByPolicy()) {
                            ACEPolicy.shutdownPolicyScheduledExecutor();
                        }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_APPLIST_ENABLE:
                        String _RESPONSE_APPLIST_ENABLE = _map.get(key);
                        if (!TextUtils.isEmpty(_RESPONSE_APPLIST_ENABLE)) {
                            try {
                                ACEPolicyParameters.getInstance().setCpApp(Integer.parseInt(_RESPONSE_APPLIST_ENABLE));
                            }
                            catch (NumberFormatException e) {
                                e.printStackTrace();
                            }
                        }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_CID: {
                        String value = _map.get(key);
                        ACEPolicyParameters.getInstance().setCpCid(value);
                        IACEParameterUtil _parameterUtil = ACECommonStaticConfig.getParameterUtil();
                        if (_parameterUtil != null) {
                            _parameterUtil.loadUniqueKeyForSDK();
                        }

                        // timeInterval [S]
                        if (!TextUtils.isEmpty(value) && value.length() >= 13) {
                            long serverTime = 0L;
                            try {
                                serverTime = Long.parseLong(value.substring(0, 13));
                            } catch (NumberFormatException e) {
                                e.printStackTrace();
                            }
                            long timeInterval = Math.abs(serverTime - System.currentTimeMillis());
                            ACEPolicyParameters.getInstance().setTimeInterval(timeInterval);
                        }
                        // timeInterval [E]
                    }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_DEBUG:
                        ACEPolicyParameters.getInstance().setCpDebug(_map.get(key));
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_DOMAIN:
                        ACEPolicyParameters.getInstance().setCpDomain(_map.get(key));
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_PRIVATE:
                        ACEPolicyParameters.getInstance().setCpPrivate(_map.get(key));

                        IACEParameterUtil _parameterUtil = ACECommonStaticConfig.getParameterUtil();
                        if (_parameterUtil != null) {
                            _parameterUtil.setterForString("cpy", ACEPolicyParameters.getInstance().getCpPrivate());
                        }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_SOURCE_IP:
                        ACEPolicyParameters.getInstance().setCpSourceIP(_map.get(key));
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_FORCE_STOP:
                        String _RESPONSE_FORCE_STOP = _map.get(key);
                        if (!TextUtils.isEmpty(_RESPONSE_FORCE_STOP) &&
                                _RESPONSE_FORCE_STOP.equals(ACEStaticConfig.POLICY.FLAG_SDK_FORCE_STOP)) {
                            ACEControlTower.enableForceStop();
                            ACEPolicy.shutdownPolicyScheduledExecutor();
                        }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_FORCE_DELETE_FAILEDFILE:
                        String _RESPONSE_FORCE_DELETE_FAILEDFILE = _map.get(key);
                        if (!TextUtils.isEmpty(_RESPONSE_FORCE_DELETE_FAILEDFILE) &&
                                _RESPONSE_FORCE_DELETE_FAILEDFILE.equals(ACEStaticConfig.POLICY.FLAG_FORCE_DELETE_FAILEDFILE)) {
                            if (_context != null) {
                                ACEFileUtil.overwriteFailedLogFile(_context, new JSONArray());
                            }
                        }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_DEBUG_LOG_URL:
                        ACEPolicyParameters.getInstance().setCpCrashDomain(_map.get(key));
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_POLICY_INTERVAL: {
                        long interval = ACEPolicy.REPEAT_PULLING_INTERVAL_SECOND;
                        String value = _map.get(key);
                        if (!TextUtils.isEmpty(value)) {
                            try {
                                interval = Long.parseLong(value);
                                if (interval < ACEStaticConfig.ACECONSTANT_INTEGER.TWO_MINUTES) {
                                    interval = ACEStaticConfig.ACECONSTANT_INTEGER.TWO_MINUTES;
                                }
                            }
                            catch (NumberFormatException e) {
                                e.printStackTrace();
                            }
                        }
                        ACEPolicy.REPEAT_PULLING_INTERVAL_SECOND = interval;
                        if (_context != null) {
                            ACEPolicy.reRequestPolicy(_context);
                        }
                    }
                        break;
                    case ACEStaticConfig.POLICY.RESPONSE_TOAST_APPKEY: {
                        String value = _map.get(key);
                        ACEPolicyParameters.getInstance().setToastAppKey(value);
                    }
                        break;
                }
            }
        }
        else {
            ACELog.d(TAG, "http response code not ok: " + response.toString());
        }
        ACELog.d(TAG, "policy result: " + ACEPolicyParameters.getInstance().toString());
    }

    public synchronized void resetPolicyParameters() {
        ACEPolicyParameters.getInstance().setCpAllow(null);
        ACEPolicyParameters.getInstance().setCpApp(0);
        ACEPolicyParameters.getInstance().setCpCid(null);
        ACEPolicyParameters.getInstance().setCpDebug(null);
        ACEPolicyParameters.getInstance().setCpDomain(null);
        ACEPolicyParameters.getInstance().setCpPrivate(null);
        ACEPolicyParameters.getInstance().setCpSourceIP(null);
        ACEPolicyParameters.getInstance().setTimeInterval(0L);
    }

    @NonNull
    public synchronized Boolean isGatherByPolicy() {
        if (TextUtils.isEmpty(ACEPolicyParameters.getInstance().getCpDebug())) {
            return false;
        }

        return ACEPolicyParameters.getInstance().getCpDebug().equals(ACEStaticConfig.POLICY.FLAG_ACELOG_GATHERING);
    }
    //endregion Policy

    //region LOG
    public String getURL() {
        return ACEStaticConfig.POLICY.REQUEST_URL;
    }
    //endregion LOG
    //endregion Parameter Helpers

    //region JSON, toString
    public JSONObject getParamsToJSONobject() throws JSONException {
        return ACEPolicyParameters.getInstance().getParamsToJSONobject();
    }

    @NonNull
    @Override
    public String toString() {
        try {
            return getParamsToJSONobject().toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return super.toString();
    }
    //endregion JSON, toString
}
