package com.acecounter.android.acetm.common.util;

import android.Manifest;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresPermission;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;
import java.util.Locale;

public final class NetworkUtils {
    private NetworkUtils() { }

    private static String simCountryIso;

    public static String getLocale(Context context) {
        String simCountry = getSimCountry(context);

        if (TextUtils.isEmpty(simCountry)) {
            String deviceCountry = Locale.getDefault().getCountry();

            if (TextUtils.isEmpty(deviceCountry)) {
                return "ZZ";
            } else {
                if (deviceCountry.length() > 2) {
                    deviceCountry = deviceCountry.substring(0, 2);
                }
                return deviceCountry;
            }

        } else {
            if (simCountry.length() > 2) {
                simCountry = simCountry.substring(0, 2);
            }
            return simCountry;
        }
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isNetworkConnected(Context context) {
        ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (manager == null) {
            return false;
        }

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
            Network network = manager.getActiveNetwork();
            NetworkCapabilities capabilities = manager.getNetworkCapabilities(network);
            return capabilities != null && (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) || capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR));
        }
        else {
            NetworkInfo networkInfo = manager.getActiveNetworkInfo();
            if (networkInfo != null) {
                // connected to the internet
                return networkInfo.getType() == ConnectivityManager.TYPE_WIFI || networkInfo.getType() == ConnectivityManager.TYPE_MOBILE;
            }
            else {
                // not connected to the internet
                return false;
            }
        }
    }

    @Nullable
    public static String getLocalIpAddress() throws SocketException {
        String localIp = null;
        for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
            NetworkInterface intf = en.nextElement();
            for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                InetAddress inetAddress = enumIpAddr.nextElement();
                if (!inetAddress.isLoopbackAddress()) {
                    localIp = inetAddress.getHostAddress();
                }
            }
        }

        return localIp;
    }

    @Nullable
    public static String getLocalIpv4Address() throws SocketException {
        String localIp = null;
        for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
            NetworkInterface intf = en.nextElement();
            for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                InetAddress inetAddress = enumIpAddr.nextElement();
                if (!inetAddress.isLoopbackAddress() && inetAddress instanceof Inet4Address) {
                    localIp = inetAddress.getHostAddress();
                }
            }
        }

        return localIp;
    }

    @Nullable
    public static String getLocalIpv6Address() throws SocketException {
        String localIp = null;
        for (Enumeration<NetworkInterface> en = NetworkInterface.getNetworkInterfaces(); en.hasMoreElements(); ) {
            NetworkInterface intf = en.nextElement();
            for (Enumeration<InetAddress> enumIpAddr = intf.getInetAddresses(); enumIpAddr.hasMoreElements(); ) {
                InetAddress inetAddress = enumIpAddr.nextElement();
                if (!inetAddress.isLoopbackAddress() && inetAddress instanceof Inet6Address) {
                    localIp = inetAddress.getHostAddress();
                }
            }
        }

        return localIp;
    }

    @Nullable
    private static String getSimCountry(Context context) throws NullPointerException {
        if (simCountryIso == null) {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager != null) {
                simCountryIso = telephonyManager.getSimCountryIso();
            }
        }

        return simCountryIso;
    }
}