package com.acecounter.android.acetm.common.util;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import android.text.TextUtils;

import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;

import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

public final class EncryptSharedPreferences implements SharedPreferences {
    static final private String TAG = EncryptSharedPreferences.class.getSimpleName();
    static final private String TAG_SHARED_PREFERENCES = "AceTM";

    private boolean isEncrypt;
    private SharedPreferences sharedPreferences;
    private EncryptUtil encryptUtil;

    private EncryptSharedPreferences() {
        ACELog.v(TAG, "EncryptSharedPreferences 초기화 합니다.");
    }

    //region 싱글턴
    private static class Singleton {
        private static final EncryptSharedPreferences INSTANCE = new EncryptSharedPreferences();
    }

    @NonNull
    public static EncryptSharedPreferences getInstance(@Nullable final Context context) {
        return EncryptSharedPreferences.Singleton.INSTANCE.init(context);
    }
    //endregion 싱글턴

    private EncryptSharedPreferences init(@Nullable final Context context) {
        if (context != null) {
            sharedPreferences = context.getSharedPreferences(TAG_SHARED_PREFERENCES, Context.MODE_PRIVATE);
        }

        return this;
    }

    public void setEncryptKey(String key) {
        try {
            if (TextUtils.isEmpty(key)) {
                this.isEncrypt = false;
            } else {
                this.isEncrypt = true;
                this.encryptUtil = new EncryptUtil(key);
                ACELog.i(TAG, "암호화 기능 사용");
            }
        } catch (UnsupportedEncodingException e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "UTF-8 인코딩을 지원하지 않아 암호화 하지 않습니다.").toString());
            this.isEncrypt = false;
        }
        catch (EncryptKeyException e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "key 길이로 인해 암호화 하지 않습니다.").toString());
            this.isEncrypt = false;
        }
    }

    @Deprecated
    @Override
    public Map<String, ?> getAll() {
        return sharedPreferences.getAll();
    }

    @Nullable
    @Override
    public String getString(String key, String defValue) throws ClassCastException {
        // ClassCastException
        String encryptValue = sharedPreferences.getString(key, defValue);
        try {
            // 저장된 값이 없음
            if (encryptValue == null) {
                return null;
            }
            // 저장된 것이 있지만 암호화 키를 제공하지 않음
            else if (!isEncrypt) {
                // '=' 으로 끝나면 지우고 defValue
                if (encryptValue.trim().endsWith("=")) {
                    edit().remove(key).apply();
                    return defValue;
                }
                // 그 외 저장된 값
                else {
                    return encryptValue;
                }
            } else if (encryptValue.equals(defValue)) {
                return defValue;
            } else {
                return encryptUtil.aesDecode(encryptValue);
            }
        } catch (UnsupportedEncodingException | BadPaddingException | IllegalBlockSizeException |
                NoSuchPaddingException | NoSuchAlgorithmException | InvalidAlgorithmParameterException |
                InvalidKeyException e) {
            ACEDebugLog.wtf(TAG,
                    new ACEException(
                            e,
                            String.format(Locale.getDefault(), "getString 과정에 예외 발생 key: %s 삭제", key)).toString());
            edit().remove(key).apply();
            return defValue;
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.HONEYCOMB)
    @Nullable
    @Deprecated
    @Override
    public Set<String> getStringSet(String key, Set<String> defValues) {
        return sharedPreferences.getStringSet(key, defValues);
    }

    @Override
    public int getInt(String key, int defValue) {
        try {
            if (!isEncrypt) {
                try {
                    return sharedPreferences.getInt(key, defValue);
                } catch (ClassCastException e) {
                    edit().remove(key).apply();
                    return defValue;
                }
            } else {
                return Integer.parseInt(getString(key, Integer.toString(defValue)));
            }
        } catch (NumberFormatException e) {
            return defValue;
        } catch (ClassCastException e) {
            int data = sharedPreferences.getInt(key, defValue);
            edit().putInt(key, data).apply();
            return data;
        }
    }

    @Override
    public long getLong(String key, long defValue) {
        try {
            if (!isEncrypt) {
                try {
                    return sharedPreferences.getLong(key, defValue);
                } catch (ClassCastException e) {
                    edit().remove(key).apply();
                    return defValue;
                }
            } else {
                return Long.parseLong(getString(key, Long.toString(defValue)));
            }
        } catch (NumberFormatException e) {
            return defValue;
        } catch (ClassCastException e) {
            long data = sharedPreferences.getLong(key, defValue);
            edit().putLong(key, data).apply();
            return data;
        }
    }

    @Override
    public float getFloat(String key, float defValue) {
        try {
            if (!isEncrypt) {
                try {
                    return sharedPreferences.getFloat(key, defValue);
                } catch (ClassCastException e) {
                    edit().remove(key).apply();
                    return defValue;
                }
            } else {
                return Float.parseFloat(getString(key, Float.toString(defValue)));
            }
        } catch (NumberFormatException e) {
            return defValue;
        } catch (ClassCastException e) {
            float data = sharedPreferences.getFloat(key, defValue);
            edit().putFloat(key, data).apply();
            return data;
        }
    }

    @Override
    public boolean getBoolean(String key, boolean defValue) {
        try {
            if (!isEncrypt) {
                try {
                    return sharedPreferences.getBoolean(key, defValue);
                } catch (ClassCastException e) {
                    edit().remove(key).apply();
                    return defValue;
                }
            } else {
                return Boolean.parseBoolean(getString(key, Boolean.toString(defValue)));
            }
        } catch (NumberFormatException e) {
            return defValue;
        } catch (ClassCastException e) {
            boolean data = sharedPreferences.getBoolean(key, defValue);
            edit().putBoolean(key, data).apply();
            return data;
        }
    }

    @Override
    public boolean contains(String key) {
        return sharedPreferences.contains(key);
    }

    @Override
    public Editor edit() {
        return new EncryptSharedPreferences.Editor(sharedPreferences.edit());
    }

    @Override
    public void registerOnSharedPreferenceChangeListener(OnSharedPreferenceChangeListener listener) {
        sharedPreferences.registerOnSharedPreferenceChangeListener(listener);
    }

    @Override
    public void unregisterOnSharedPreferenceChangeListener(OnSharedPreferenceChangeListener listener) {
        sharedPreferences.unregisterOnSharedPreferenceChangeListener(listener);
    }

    public class Editor implements SharedPreferences.Editor {
        private SharedPreferences.Editor editor;

        private Editor(SharedPreferences.Editor editor) {
            this.editor = editor;
        }


        @Override
        public EncryptSharedPreferences.Editor putString(String key, String value) {
            try {
                if (!isEncrypt || value == null) {
                    editor.putString(key, value);
                } else {
                    String encryptValue = encryptUtil.aesEncode(value);
                    editor.putString(key, encryptValue);
                }
            } catch (UnsupportedEncodingException | IllegalBlockSizeException | BadPaddingException | NoSuchPaddingException | InvalidKeyException | InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
                e.printStackTrace();
                editor.remove(key);
            }
            return this;
        }

        @RequiresApi(api = Build.VERSION_CODES.HONEYCOMB)
        @Deprecated
        @Override
        public EncryptSharedPreferences.Editor putStringSet(String key, Set<String> values) {
            editor.putStringSet(key, values);
            return this;
        }

        @Override
        public EncryptSharedPreferences.Editor putInt(String key, int value) {
            if (!isEncrypt) {
                editor.putInt(key, value);
                return this;
            }
            return putString(key, Integer.toString(value));
        }

        @Override
        public EncryptSharedPreferences.Editor putLong(String key, long value) {
            if (!isEncrypt) {
                editor.putLong(key, value);
                return this;
            }
            return putString(key, Long.toString(value));
        }

        @Override
        public EncryptSharedPreferences.Editor putFloat(String key, float value) {
            if (!isEncrypt) {
                editor.putFloat(key, value);
                return this;
            }
            return putString(key, Float.toString(value));
        }

        @Override
        public EncryptSharedPreferences.Editor putBoolean(String key, boolean value) {
            if (!isEncrypt) {
                editor.putBoolean(key, value);
                return this;
            }
            return putString(key, Boolean.toString(value));
        }

        @Override
        public EncryptSharedPreferences.Editor remove(String key) {
            editor.remove(key);
            return this;
        }

        @Override
        public EncryptSharedPreferences.Editor clear() {
            editor.clear();
            return this;
        }

        @Override
        public boolean commit() {
            return editor.commit();
        }

        @Override
        public void apply() {
            editor.apply();
        }
    }
}
